﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_AccessFailureDetection.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/result/result_HandlingUtility.h>

#include "fs_EventNotifierObjectAdapter.h"
#include "fs_FileSystemProxyServiceObject.h"

namespace nn { namespace fs {

    Result GetAccessFailureDetectionEvent(os::SystemEvent* outValue) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        sf::NativeHandle sfHandle;
        NN_FS_RESULT_DO(fileSystemProxy->GetAccessFailureDetectionEvent(sf::Out<sf::NativeHandle>(&sfHandle)));
        outValue->AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), os::EventClearMode_ManualClear);
        sfHandle.Detach();
        NN_RESULT_SUCCESS;
    }

    bool IsAccessFailureDetected(nn::Bit64 processId) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        bool isAccessFailureDetected = false;
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->IsAccessFailureDetected(&isAccessFailureDetected, processId));
        return isAccessFailureDetected;
    }

    Result OpenAccessFailureDetectionEventNotifier(std::unique_ptr<IEventNotifier>* outValue, nn::Bit64 processId) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier> eventNotifier;
        NN_FS_RESULT_DO(fileSystemProxy->OpenAccessFailureDetectionEventNotifier(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier>>(&eventNotifier), processId));
        outValue->reset(new detail::EventNotifierObjectAdapter(std::move(eventNotifier)));
        NN_FS_RESULT_THROW_UNLESS(*outValue, ResultAllocationMemoryFailedInSdCardB());
        NN_RESULT_SUCCESS;
    }

    void ResolveAccessFailure(nn::Bit64 processId) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto result = fileSystemProxy->ResolveAccessFailure(processId);
        if (ResultDeferredProcessRetryFailure::Includes(result))
        {
            return;
        }
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(result);
        return;
    }

    void AbandonAccessFailure(nn::Bit64 processId) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->AbandonAccessFailure(processId));
    }

    Result OpenAccessFailureResolver(std::unique_ptr<AccessFailureResolver>* outValue, nn::Bit64 processId) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier> eventNotifier;
        NN_FS_RESULT_DO(fileSystemProxy->OpenAccessFailureDetectionEventNotifier(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IEventNotifier>>(&eventNotifier), processId));
        outValue->reset(new AccessFailureResolver(std::move(eventNotifier), processId));
        NN_FS_RESULT_THROW_UNLESS(*outValue, ResultAllocationMemoryFailedNew());
        NN_RESULT_SUCCESS;
    }

}}
