﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/sf/sf_Result.h>
#include <nn/svc/svc_Result.h>

#define NN_DETAIL_FS_RESULT_ERROR(resultClass, result, message) \
    do \
    { \
        if( resultClass::Includes(result) ) \
        { \
            NN_DETAIL_FS_ERROR(NN_MACRO_STRINGIZE(resultClass) ": " message); \
            return; \
        } \
    } while(NN_STATIC_CONDITION(0))


namespace nn { namespace fs {

namespace {
    bool g_IsResultHandledByApplication = false;
}

void SetResultHandledByApplication(bool isApplication) NN_NOEXCEPT
{
    g_IsResultHandledByApplication = isApplication;
}

namespace detail {
    bool IsAbortNeeded(Result result) NN_NOEXCEPT
    {
        if (result.IsSuccess())
        {
            return false;
        }

        auto abortSpecifier = GetCurrentThreadFsContext()->HandleResult(result);

        switch (abortSpecifier)
        {
            case AbortSpecifier::Default:
            {
                if (g_IsResultHandledByApplication)
                {
                    return !ResultHandledByAllProcess::Includes(result);
                }
                else
                {
                    return !(ResultHandledByAllProcess::Includes(result) || ResultHandledBySystemProcess::Includes(result));
                }
            }
            case AbortSpecifier::Abort:
            {
                return true;
            }
            case AbortSpecifier::ReturnResult:
            {
                return false;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    void LogResultErrorMessage(Result result) NN_NOEXCEPT
    {
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultPartitionNotFound           , result, "Error: Specified partition is not found.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultTargetNotFound              , result, "Error: Specified target is not found.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultMmcAccessFailed             , result, "Error: Failed to access MMC.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultSdCardAccessFailed          , result, "Error: Failed to access SD card.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultGameCardAccessFailed        , result, "Error: Failed to access game card.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultRomCorrupted                , result, "Error: ROM is corrupted.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultSaveDataCorrupted           , result, "Error: Save data is corrupted.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultNcaCorrupted                , result, "Error: NCA is corrupted.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultPartitionFileSystemCorrupted, result, "Error: Partition FS is corrupted.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultBuiltInStorageCorrupted     , result, "Error: Built-in-storage is corrupted.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultFatFileSystemCorrupted      , result, "Error: FAT FS is corrupted.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultHostFileSystemCorrupted     , result, "Error: HOST FS is corrupted.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultIntegrityVerificationStorageCorrupted, result, "Error: Integrity verification failed.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultDataCorrupted               , result, "Error: Data is corrupted.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultAllocationMemoryFailed      , result, "Error: Failed to allocate memory.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultMountNameAlreadyExists      , result, "Error: Specified mount name already exists.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultOutOfRange                  , result, "Error: Specified value is out of range.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultUnexpected                  , result, "Error: Unexpected failure occurred.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultNotImplemented              , result, "Error: Specified operation is not implemented.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultPermissionDenied            , result, "Error: Permission denied.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultUnsupportedOperation        , result, "Error: Unsupported operation.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend, result, "Error: OpenMode_AllowAppend is required for implicit extension of file size by WriteFile().\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultInvalidOperationForOpenMode , result, "Error: Invalid operation for the open mode.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultNullptrArgument             , result, "Error: Null pointer argument was specified.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultInvalidSize                 , result, "Error: Invalid size was specified.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultInvalidOffset               , result, "Error: Invalid offset was specified.\n");
        // ResultTooLongPath, ResultInvalidCharacter
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultInvalidPath                 , result, "Error: Invalid path was specified.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultInvalidMountName            , result, "Error: Invalid mount name was specified.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultExtensionSizeTooLarge       , result, "Error: Extension size exceeds max value set in nmeta file.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultExtensionSizeInvalid        , result, "Error: Extension size is not a multiple of nn::fs::SaveDataExtensionUnitSize.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultInvalidArgument             , result, "Error: Invalid argument was specified.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultMappingTableFull            , result, "Error: Enough journal space is not left.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultPreconditionViolation       , result, "Error: Precondition violation.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultMapFull                     , result, "Error: Save data extension count reached the limitation.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultOpenCountLimit              , result, "Error: The open count of files and directories reached the limitation.\n");
        // (SIGLO-15521, etc.)
        NN_DETAIL_FS_RESULT_ERROR(nn::svc::ResultInvalidCurrentMemory       , result, "Error: Passed buffer is not usable for fs library.\nSee a manual of fs library for more information.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::sf::ResultMemoryAllocationFailed      , result, "Error: Failed to allocate memory.\n");
        NN_DETAIL_FS_RESULT_ERROR(nn::fs::ResultNotMounted                  , result, "Error: Specified mount name is not found.\n");

        // （ここまで来たら、対応するログがまだ登録されていない）
    }

    // リザルトに沿ったログを表示する
    void LogErrorMessage(Result result, const char* functionName) NN_NOEXCEPT
    {
        NN_UNUSED(functionName);

        if( result.IsSuccess() )
        {
            return;
        }

        NN_SDK_LOG("****** FS ERROR INFORMATION ******\n");
        NN_DETAIL_FS_ERROR("Error: Error occurred at %s().\n", functionName);

        LogResultErrorMessage(result);

        if( !nn::fs::detail::IsEnabledAccessLog() )
        {
            NN_DETAIL_FS_ERROR("More infomation may be provided by enabling 'FS Access Log Mode' at 'DevMenu: Debug'.\n\n");
            if( !g_IsResultHandledByApplication && !nn::fs::detail::IsEnabledAccessLog(nn::fs::detail::AccessLogTarget_System) )
            {
                NN_DETAIL_FS_ERROR("Please call 'nn::fs::SetLocalSystemAccessLogForDebug(true);' to enable access log.\n\n");
            }
        }
    } // NOLINT(impl/function_size)
}}}
