﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_PathTool.h>

namespace nn { namespace fs {

/**
* @brief パスの正規化を行います。
*
*        正規化を行うとパス文字列は以下のように変化します。
*
*        ・"/" => "/"
*        ・"/." => "/"
*        ・"////" => "/"
*        ・"/dir1" => "/dir1"
*        ・"/dir1///" => "/dir1"
*        ・"/dir1/././//" => "/dir1"
*        ・"/dir1/dir2/.." => "/dir1"
*
*        以下の場合にはエラーを返します。
*
*        ・先頭が"/"から開始していない。
*        ・ルートディレクトリよりも親のディレクトリを指定した。
*
* @param[out] outValue 正規化後のパス
* @param[out] outLength 正規化後のパスの文字数(ヌル終端文字を含まない)
* @param[in] pSrc 正規化前のパス
* @param[in] outBufLength 正規化後のパスを格納するバッファサイズ
* @param[in] isUncPreserved 先頭の連続した区切り文字を UNC とみなして保存するかどうか
*
* @return 関数の処理結果を返します。
*/
Result PathTool::Normalize(
           char* outValue,
           size_t* outLength,
           const char* pSrc,
           size_t outBufLength,
           bool isUncPreserved
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);
    NN_SDK_ASSERT_NOT_NULL(outLength);
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    if( (! IsSeparator(pSrc[0])) )
    {
        // 先頭が"/"から開始していません。
        NN_RESULT_THROW(nn::fs::ResultInvalidPathFormat());
    }

    bool isSkipNextSeparator = false;
    size_t srcIndex = 0;
    size_t totalLength = 0;
    while( !IsNul(pSrc[srcIndex]) )
    {
        if( IsSeparator(pSrc[srcIndex]) )
        {
            // 連続した"/"は無視します。
            while( IsSeparator(pSrc[++srcIndex]) )
            {
            }
            // 最後の文字が "/" であった場合は追加せずにループを終了します。
            if( IsNul(pSrc[srcIndex]) )
            {
                break;
            }

            // セパレータを追加します。
            if( !isSkipNextSeparator )
            {
                if( totalLength + 1 == outBufLength )
                {
                    outValue[totalLength] = StringTraits::NullCharactor;
                    *outLength = totalLength;
                    NN_RESULT_THROW(nn::fs::ResultTooLongPath());
                }
                outValue[totalLength++] = StringTraits::DirectorySeparator;

                // UNC の先頭にある連続した区切り文字はそのまま追加します。
                if( isUncPreserved && totalLength == 1 )
                {
                    while( totalLength < srcIndex )
                    {
                        if( totalLength + 1 == outBufLength )
                        {
                            outValue[totalLength] = StringTraits::NullCharactor;
                            *outLength = totalLength;
                            NN_RESULT_THROW(nn::fs::ResultTooLongPath());
                        }
                        outValue[totalLength] = StringTraits::DirectorySeparator;
                        ++totalLength;
                    }
                }
            }
            isSkipNextSeparator = false;
        }

        // "/"までの長さを取得します。
        size_t dirLength = 0;
        while( !IsSeparator(pSrc[srcIndex + dirLength])
            && !IsNul(pSrc[srcIndex + dirLength]) )
        {
            dirLength++;
        }

        if( IsCurrentDirectory(&pSrc[srcIndex]) )
        {
            // "."であれば次の"/"を無視します。
            isSkipNextSeparator = true;
        }
        else if( IsParentDirectory(&pSrc[srcIndex]) )
        {
            // ".."であれば親ディレクトリに戻ります。

            NN_SDK_ASSERT(outValue[totalLength - 1] == StringTraits::DirectorySeparator);
            NN_SDK_ASSERT(outValue[0] == StringTraits::DirectorySeparator);
            if( totalLength == 1 )
            {
                // ルートディレクトリに親ディレクトリは存在しません。
                NN_RESULT_THROW(nn::fs::ResultDirectoryUnobtainable());
            }

            // "/"が出てくるまで前に戻ります。
            totalLength -= 2;
            while( outValue[totalLength] != StringTraits::DirectorySeparator )
            {
                totalLength--;
            }

            // "/"の位置を指しています。
            NN_SDK_ASSERT(outValue[totalLength] == StringTraits::DirectorySeparator);
        }
        else
        {
            // ディレクトリパスを出力バッファに追加します。
            if( totalLength + dirLength + 1 > outBufLength )
            {
                size_t loopCount = outBufLength - 1 - totalLength;
                for( size_t i = 0; i < loopCount; ++i )
                {
                    outValue[totalLength++] = pSrc[srcIndex + i];
                }

                outValue[totalLength] = StringTraits::NullCharactor;
                *outLength = totalLength;

                NN_RESULT_THROW(nn::fs::ResultTooLongPath());
            }
            else
            {
                for( size_t i = 0; i < dirLength; ++i )
                {
                    outValue[totalLength++] = pSrc[srcIndex + i];
                }
            }
        }

        srcIndex += dirLength;
    }

    // 末尾に "." が来ていたら最後に追加した "/" を削除します
    if( isSkipNextSeparator )
    {
        --totalLength;
    }

    // ルートディレクトリであれば "/" だけを挿入します。
    if( (totalLength == 0) && (outBufLength > 1) )
    {
        totalLength = 1;
        outValue[0] = StringTraits::DirectorySeparator;
    }

    // 最後に "\0" を挿入します。
    if( outBufLength >= totalLength - 1 )
    {
        outValue[totalLength] = StringTraits::NullCharactor;

        bool isNormalized;
        NN_SDK_ASSERT(isUncPreserved || (IsNormalized(&isNormalized, outValue).IsSuccess() && isNormalized));
        NN_UNUSED(isNormalized);

        *outLength = totalLength;

        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_THROW(nn::fs::ResultTooLongPath());
    }
} // NOLINT(impl/function_size)

/**
* @brief パスが正規化済みかどうか判定します。
*
* @param[out] outNormalized 正規化済みかどうか
* @param[in] path パス
*
* @return 関数の処理結果を返します。
*
* @details 連続した区切り文字 ("//")、カレントディレクトリ (".") または親ディレクトリ ("..") を含む場合や
*          区切り文字で終わる場合、非正規化パスと判定します。
*          ドライブレター 1 文字とコロンで始まる Windows 形式のパスは
*          上記の場合であっても正規化パスとみなします。
*/
Result PathTool::IsNormalized(bool* outNormalized, const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outNormalized);
    NN_SDK_REQUIRES_NOT_NULL(path);

    enum class State
    {
        Initial,
        Other,
        DriveLetter,
        FirstSeparator,
        Separator,
        CurrentDirectory,
        ParentDirectory,
    };

    auto state = State::Initial;

    for( auto pChar = path; *pChar != StringTraits::NullCharactor; ++pChar )
    {
        switch( state )
        {
        case State::Initial:
            if( ('a' <= *pChar && *pChar <= 'z') || ('A' <= *pChar && *pChar <= 'Z') )
            {
                state = State::DriveLetter;
            }
            else if( *pChar == StringTraits::DirectorySeparator )
            {
                state = State::FirstSeparator;
            }
            else
            {
                return nn::fs::ResultInvalidPathFormat();
            }
            break;

        case State::Other:
            if( *pChar == StringTraits::DirectorySeparator )
            {
                state = State::Separator;
            }
            break;

        case State::DriveLetter:
            if( *pChar == ':' )
            {
                *outNormalized = true;
                NN_RESULT_SUCCESS;
            }
            else
            {
                return nn::fs::ResultInvalidPathFormat();
            }
            break;

        case State::FirstSeparator:
        case State::Separator:
            if( *pChar == StringTraits::DirectorySeparator )
            {
                *outNormalized = false;
                NN_RESULT_SUCCESS;
            }
            else if( *pChar == StringTraits::Dot )
            {
                state = State::CurrentDirectory;
            }
            else
            {
                state = State::Other;
            }
            break;

        case State::CurrentDirectory:
            if( *pChar == StringTraits::DirectorySeparator )
            {
                *outNormalized = false;
                NN_RESULT_SUCCESS;
            }
            else if( *pChar == StringTraits::Dot )
            {
                state = State::ParentDirectory;
            }
            else
            {
                state = State::Other;
            }
            break;

        case State::ParentDirectory:
            if( *pChar == StringTraits::DirectorySeparator )
            {
                *outNormalized = false;
                NN_RESULT_SUCCESS;
            }
            else
            {
                state = State::Other;
            }
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    switch( state )
    {
    case State::Initial:
    case State::DriveLetter:
        return nn::fs::ResultInvalidPathFormat();

    case State::FirstSeparator:
    case State::Other:
        *outNormalized = true;
        NN_RESULT_SUCCESS;

    case State::Separator:
    case State::CurrentDirectory:
    case State::ParentDirectory:
        *outNormalized = false;
        NN_RESULT_SUCCESS;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
} // NOLINT(impl/function_size)

/**
* @brief いずれかのパスが他方のサブパスかどうか判定します。
*
* @param[in] pLhs パス
* @param[in] pRhs パス
*
* @return 関数の処理結果を返します。
*
* @pre パスが正規化済み
*/
bool PathTool::IsSubpath(const char* pLhs, const char* pRhs) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pLhs);
    NN_SDK_REQUIRES_NOT_NULL(pRhs);

    // UNC と非 UNC パスの組み合わせはサブパスになりえません。
    if (pLhs[0] == '/' && pLhs[1] != '/')
    {
        if (pRhs[0] == '/' && pRhs[1] == '/')
        {
            return false;
        }
    }
    if (pRhs[0] == '/' && pRhs[1] != '/')
    {
        if (pLhs[0] == '/' && pLhs[1] == '/')
        {
            return false;
        }
    }

    // ルートディレクトリのみ特殊対応します。
    if (pLhs[0] == '/' && pLhs[1] == '\0')
    {
        if (pRhs[0] == '/' && pRhs[1] != '\0')
        {
            return true;
        }
    }
    if (pRhs[0] == '/' && pRhs[1] == '\0')
    {
        if (pLhs[0] == '/' && pLhs[1] != '\0')
        {
            return true;
        }
    }

    for( auto index = 0; ; ++index )
    {
        if( pLhs[index] == '\0' )
        {
            return pRhs[index] == '/';
        }
        else if( pRhs[index] == '\0' )
        {
            return pLhs[index] == '/';
        }
        else if( pLhs[index] != pRhs[index] )
        {
            return false;
        }
    }
}

}}

