﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_HipcServer.h>

#include <nn/eth/sfdl/eth.sfdl.h>
#include <nn/eth/eth_EthTypes.h>

#include "eth_Interface.h"
#include "eth_InterfaceSfObject.h"

namespace nn     {
namespace eth    {
namespace server {

nn::Result InterfaceSfObject::GetMediaList(const nn::sf::OutBuffer& mediaOut)
NN_NOEXCEPT
{
    if (mediaOut.GetSize() != sizeof(MediaList))
    {
        return nn::eth::ResultInvalidArgument();
    }

    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    if (m_pInterface == nullptr)
    {
        return nn::eth::ResultInterfaceRemoved();
    }

    return m_pInterface->GetMediaList(reinterpret_cast<MediaList*>(mediaOut.GetPointerUnsafe()));
}

nn::Result InterfaceSfObject::SetMediaType(MediaType mediaType)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    if (m_pInterface == nullptr)
    {
        return nn::eth::ResultInterfaceRemoved();
    }

    return m_pInterface->SetMediaType(mediaType);
}

nn::Result InterfaceSfObject::GetMediaType(nn::sf::Out<MediaType> mediaRequested, nn::sf::Out<MediaType> mediaCurrent, nn::sf::Out<uint32_t> eventCounter)
NN_NOEXCEPT
{
    nn::Result result;
    MediaType  requested;
    MediaType  current;
    uint32_t   counter;

    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    if (m_pInterface == nullptr)
    {
        return nn::eth::ResultInterfaceRemoved();
    }

    result = m_pInterface->GetMediaType(&requested, &current, &counter);

    if (result.IsSuccess())
    {
        mediaRequested.Set(requested);
        mediaCurrent.Set(current);
        eventCounter.Set(counter);
    }

    return result;
}

void InterfaceSfObject::Signal(nn::Result result)
NN_NOEXCEPT
{
    m_Result = result;
    m_SystemEvent.Signal();
}

nn::Result InterfaceSfObject::GetResult()
NN_NOEXCEPT
{
    return m_Result;
}

nn::Result InterfaceSfObject::Cancel()
NN_NOEXCEPT
{
    Signal(nn::eth::ResultEventCanceled());
    return ResultSuccess();
}

void InterfaceSfObject::Detach()
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    Signal(nn::eth::ResultInterfaceRemoved());
    m_pInterface = nullptr;
}

nn::Result InterfaceSfObject::Initialize(nn::sf::Out<nn::sf::NativeHandle> handleOut, const nn::sf::InBuffer& name)
NN_NOEXCEPT
{
    nn::Result  result;
    const char* pInterfaceName = name.GetPointerUnsafe();

    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    if (pInterfaceName == nullptr)
    {
        return nn::eth::ResultInvalidArgument();
    }

    if (m_pInterface != nullptr)
    {
        return nn::eth::ResultAlreadyInitialized();
    }

    if ((result = AttachToInterface(&m_pInterface, pInterfaceName, this)).IsSuccess())
    {
        handleOut.Set(nn::sf::NativeHandle(m_SystemEvent.GetReadableHandle(), false));
    }

    return result;
}

InterfaceSfObject::InterfaceSfObject()
NN_NOEXCEPT :
    m_Result(nn::ResultSuccess()),
    m_SystemEvent(nn::os::EventClearMode_AutoClear, true),
    m_Lock(false),
    m_pInterface(nullptr)
{

}

InterfaceSfObject::~InterfaceSfObject()
NN_NOEXCEPT
{
    if (m_pInterface != nullptr)
    {
        m_pInterface->DetachClient(this);
        m_pInterface = nullptr;
    }
}

}}}
