﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/util/util_IntrusiveList.h>

#include "eth_Interface.h"
#include "eth_InterfaceGroupSfObject.h"

namespace nn     {
namespace eth    {
namespace server {
namespace        {

nn::os::Mutex g_ClientListLock(false);
nn::util::IntrusiveList<InterfaceGroupSfObject, nn::util::IntrusiveListBaseNodeTraits<InterfaceGroupSfObject>> g_ClientList;

}

nn::Result InterfaceGroupSfObject::GetInterfaceList(const nn::sf::OutBuffer& interfaceListOut)
NN_NOEXCEPT
{
    if (interfaceListOut.GetSize() != sizeof(InterfaceList))
    {
        return nn::eth::ResultInvalidArgument();
    }

    return ListInterfaces(reinterpret_cast<InterfaceList*>(interfaceListOut.GetPointerUnsafe()));
}

nn::Result InterfaceGroupSfObject::GetInterfaceCount(nn::sf::Out<uint32_t> interfaceCount)
NN_NOEXCEPT
{
    uint32_t   count;
    nn::Result result;

    if ((result = InterfaceCount(&count)).IsSuccess())
    {
        interfaceCount.Set(count);
    }

    return result;
}

nn::Result InterfaceGroupSfObject::GetReadableHandle(nn::sf::Out<nn::sf::NativeHandle> handleOut)
NN_NOEXCEPT
{
    handleOut.Set(nn::sf::NativeHandle(m_SystemEvent.GetReadableHandle(), false));
    return ResultSuccess();
}

nn::Result InterfaceGroupSfObject::Signal(nn::Result result)
NN_NOEXCEPT
{
    m_Result = result;
    m_SystemEvent.Signal();
    return ResultSuccess();
}

nn::Result InterfaceGroupSfObject::GetResult()
NN_NOEXCEPT
{
    return m_Result;
}

nn::Result InterfaceGroupSfObject::Cancel()
NN_NOEXCEPT
{
    Signal(ResultEventCanceled());
    return ResultSuccess();
}

InterfaceGroupSfObject::InterfaceGroupSfObject()
NN_NOEXCEPT :
    m_Result(nn::ResultSuccess()),
    m_SystemEvent(nn::os::EventClearMode_AutoClear, true)
{
    uint32_t count;

    std::lock_guard<nn::os::Mutex> lock(g_ClientListLock);
    g_ClientList.push_front(*this);

    if (InterfaceCount(&count).IsSuccess() && count)
    {
        Signal(ResultSuccess());
    }
}

InterfaceGroupSfObject::~InterfaceGroupSfObject()
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ClientListLock);
    g_ClientList.erase(g_ClientList.iterator_to(*this));
}

void SignalGroup(nn::Result result)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ClientListLock);
    for (InterfaceGroupSfObject & attachedClient : g_ClientList)
    {
        attachedClient.Signal(result);
    }
}

}}}
