﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/psc.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/usb/usb_Host.h>
#include <nn/eth/eth_EthTypes.h>

#include "../eth_Worker.h"
#include "eth_UsbDeviceInfo.h"
#include "eth_UsbDevice.h"

namespace nn     {
namespace eth    {
namespace device {
namespace usb    {

nn::os::MultiWaitType     g_MultiWait;
nn::os::Mutex             g_DeviceListLock(false);

namespace {

const nn::psc::PmModuleId Dependencies[]  = { nn::psc::PmModuleId_Usb };
const uint32_t            DependencyCount = sizeof(Dependencies) / sizeof(Dependencies[0]);

nn::usb::Host             g_HsClient;
nn::psc::PmState          g_CurrentState = nn::psc::PmState_FullAwake;
nn::psc::PmModule         g_PmModule;
Worker                    g_UsbWorker;
uint32_t                  g_IfCount = 0;

}

void AttachDevices()
{
    const int InterfaceListMaxCount  = 5;
    nn::usb::InterfaceQueryOutput list[InterfaceListMaxCount];

    std::lock_guard<nn::os::Mutex> lock(g_DeviceListLock);

    for (UsbDeviceInfo & usbDeviceInfo : g_UsbDeviceInfoList)
    {
        int interfaceCount = 0;
        nn::usb::DeviceFilter devFilter = nn::usb::InvalidDeviceFilter;

        devFilter.idVendor   = usbDeviceInfo.m_IdVendor;
        devFilter.idProduct  = usbDeviceInfo.m_IdProduct;
        devFilter.matchFlags = nn::usb::DeviceFilterMatchFlags_Vendor | nn::usb::DeviceFilterMatchFlags_Product;

        g_HsClient.QueryAvailableInterfaces(&interfaceCount, list, sizeof(list), &devFilter);

        for (int i = 0; i < interfaceCount; i++)
        {
            if ((usbDeviceInfo.Initialize(&g_HsClient, &list[i].ifProfile)).IsSuccess())
            {
                g_IfCount++;
            }
        }
    }
}

void HandlePowerEvent()
NN_NOEXCEPT
{
    nn::psc::PmState   state;
    nn::psc::PmFlagSet flags;
    nn::Result         result;

    g_PmModule.GetEventPointer()->Clear();

    result = g_PmModule.GetRequest(&state, &flags);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    switch (state)
    {
    case nn::psc::PmState_FullAwake:
        // fallthrough...

    case nn::psc::PmState_MinimumAwake:
        // get updated list of attached interfaces,
        // re-attach if any adapters are present
        AttachDevices();
        g_CurrentState = state;
        break;

    case nn::psc::PmState_SleepReady:
        // release all attached adapters
        for (auto device  = UsbDevice::m_DeviceList.begin();
                  device != UsbDevice::m_DeviceList.end();
                  )
        {
            UsbDevice*       usbDevice = &(*(device++));
            FinalizeFunction func = usbDevice->GetFinalizeFunction();
            void*            arg  = usbDevice->GetFinalizeArgument();

            // - removes usbDevice from the list
            // - deallocates usbDevice
            func(usbDevice, arg);
            NN_ABORT_UNLESS(g_IfCount > 0);
            g_IfCount--;
        }
        g_CurrentState = state;
        break;

    default:
        break;
    }

    g_PmModule.Acknowledge(state, nn::ResultSuccess());
}

uint32_t GetInterfaceCount()
{
    return g_IfCount;
}

void EventThread()
NN_NOEXCEPT
{
    nn::usb::DeviceFilter       devFilter = nn::usb::InvalidDeviceFilter;
    nn::os::SystemEventType     newIfEvent;
    nn::os::MultiWaitHolderType newIfHolder;
    nn::os::MultiWaitHolderType powerEventHolder;
    nn::Result                  result;

    nn::os::InitializeMultiWait(&g_MultiWait);

    result = g_HsClient.Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = g_HsClient.CreateInterfaceAvailableEvent(&newIfEvent, nn::os::EventClearMode_ManualClear, 1, &devFilter);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = g_PmModule.Initialize(nn::psc::PmModuleId_Eth, Dependencies, DependencyCount, nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::InitializeMultiWaitHolder(&newIfHolder, &newIfEvent);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &newIfHolder);

    nn::os::InitializeMultiWaitHolder(&powerEventHolder, g_PmModule.GetEventPointer()->GetBase());
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &powerEventHolder);

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&g_MultiWait);

        if (holder == &newIfHolder)
        {
            nn::os::ClearSystemEvent(&newIfEvent);

            if (g_CurrentState == nn::psc::PmState_FullAwake || g_CurrentState == nn::psc::PmState_MinimumAwake)
            {
                AttachDevices();
            }
        }
        else if (holder == &powerEventHolder)
        {
            HandlePowerEvent();
        }
        else
        {
            UsbDevice*  usbDevice = reinterpret_cast<UsbDevice*>(nn::os::GetMultiWaitHolderUserData(holder));
            FinalizeFunction func = usbDevice->GetFinalizeFunction();
            void*             arg = usbDevice->GetFinalizeArgument();

            func(usbDevice, arg);

            NN_ABORT_UNLESS(g_IfCount > 0);
            g_IfCount--;
        }
    }
}

nn::Result Initialize()
NN_NOEXCEPT
{
    g_UsbWorker.Initialize(&EventThread,
                           NN_SYSTEM_THREAD_PRIORITY(socket, EthEvents),
                           NN_SYSTEM_THREAD_NAME(socket, EthEvents));
    return ResultSuccess();
}

}}}}
