﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/eth/sfdl/eth.sfdl.h>
#include <nn/eth/eth_EthTypes.h>
#include <nn/eth/eth_EthClient.h>
#include <nn/eth/server/eth_EthServer.h>

namespace nn     {
namespace eth    {
namespace client {
namespace        {

const int SessionHeapSize = 8 * 1024;

class InitExpHeap
{
public:
    InitExpHeap()
    NN_NOEXCEPT
    {
        nn::sf::ExpHeapStaticAllocator
                <SessionHeapSize, InterfaceHandler>::
                Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_InitExpHeap;

} // namespace

InterfaceHandler::InterfaceHandler()
NN_NOEXCEPT : m_Initialized(false)
{

}

InterfaceHandler::~InterfaceHandler()
NN_NOEXCEPT
{
    m_InterfaceImpl = nullptr;
}

nn::Result InterfaceHandler::Initialize(
        InterfaceGroupHandler* pGroupHandler,
        InterfaceInfo*         pInterfaceInfo,
        nn::os::EventClearMode eventClearMode)
NN_NOEXCEPT
{
    NN_UNUSED(pGroupHandler);

    nn::sf::NativeHandle sfHandle;
    nn::Result result;

    NN_ABORT_UNLESS(m_Initialized == false, "already initialized\n");

    result = nn::sf::CreateHipcProxyByName
                    <nn::eth::sf::IEthInterface, nn::sf::ExpHeapStaticAllocator
                    <SessionHeapSize, InterfaceHandler>::Policy>
                    (&m_InterfaceImpl, InterfacePortName);

    if (result.IsSuccess())
    {
        nn::sf::InBuffer interfaceName(
                            pInterfaceInfo->interfaceName,
                            sizeof(pInterfaceInfo->interfaceName));

        if ((result = m_InterfaceImpl->Initialize(&sfHandle, interfaceName)).IsSuccess())
        {
            m_InterfaceInfo = *pInterfaceInfo;
            m_SystemEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), eventClearMode);
            sfHandle.Detach();
            m_Initialized = true;
        }
        else
        {
            m_InterfaceImpl = nullptr;
        }
    }

    return result;
}

nn::Result InterfaceHandler::Finalize()
NN_NOEXCEPT
{
    m_InterfaceImpl = nullptr;
    nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
    m_Initialized   = false;
    return ResultSuccess();
}

nn::Result InterfaceHandler::GetResult()
NN_NOEXCEPT
{
    return m_InterfaceImpl->GetResult();
}

nn::Result InterfaceHandler::Cancel()
NN_NOEXCEPT
{
    return m_InterfaceImpl->Cancel();
}

void InterfaceHandler::ClearEvent()
NN_NOEXCEPT
{
    m_SystemEvent.Clear();
}

nn::os::SystemEvent* InterfaceHandler::GetSystemEventPointer()
NN_NOEXCEPT
{
    return &m_SystemEvent;
}

nn::Result InterfaceHandler::GetMediaList(MediaList* pMediaListOut)
NN_NOEXCEPT
{
    nn::sf::OutBuffer mediaList(reinterpret_cast<char*>(pMediaListOut), sizeof(MediaList));
    return m_InterfaceImpl->GetMediaList(mediaList);
}

nn::Result InterfaceHandler::SetMediaType(MediaType mediaType)
NN_NOEXCEPT
{
    return m_InterfaceImpl->SetMediaType(mediaType);
}

nn::Result InterfaceHandler::GetMediaType(MediaType* pMediaRequested, MediaType* pMediaCurrent, uint32_t* pEventCounter)
NN_NOEXCEPT
{
    return m_InterfaceImpl->GetMediaType(pMediaRequested, pMediaCurrent, pEventCounter);
}

InterfaceInfo* InterfaceHandler::GetInterfaceInfo()
NN_NOEXCEPT
{
    return &m_InterfaceInfo;
}

const char* InterfaceHandler::GetInterfaceName()
NN_NOEXCEPT
{
    return m_InterfaceInfo.interfaceName;
}

}}}
