﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/dt/dt_Result.h>
#include <nn/dt/dt_ResultPrivate.h>
#include "dt_FdtReaderUtil.h"

extern "C"
{
#include "libfdt.h"
}

namespace nn { namespace dt { namespace server {

nn::Result GetInterruptInfo(InterruptInfo* pOutInfo, const uint32_t* pCells, size_t cellCount)
{
    const uint32_t TriggerMask_EdgeLowToHigh   = 0x1;
    const uint32_t TriggerMask_EdgeHighToLow   = 0x2;
    const uint32_t TriggerMask_Edge            = TriggerMask_EdgeLowToHigh | TriggerMask_EdgeHighToLow;
    const uint32_t TriggerMask_LevelActiveHigh = 0x4;
    const uint32_t TriggerMask_LevelActiveLow  = 0x8;
    const uint32_t TriggerMask_Level           = TriggerMask_LevelActiveHigh | TriggerMask_LevelActiveLow;

    // Tegra のデバイスツリーでは、interrupts プロパティは 3つの cell から構成されていて
    // <interrupt-controller-number interrupt-number trigger-mask> となっている。
    if (cellCount != 3)
    {
        return nn::dt::ResultPropertyTypeError();
    }

    // 割り込み番号
    pOutInfo->name = fdt32_to_cpu(pCells[1]);

    auto triggerMask = fdt32_to_cpu(pCells[2]);

    // InterruptTriggerType
    if (triggerMask & TriggerMask_Edge)
    {
        pOutInfo->trigger = InterruptTriggerType_Edge;
    }
    else if (triggerMask & TriggerMask_Level)
    {
        pOutInfo->trigger = InterruptTriggerType_Level;
    }
    else
    {
        pOutInfo->trigger = InterruptTriggerType_Unspecified;
    }

    // InterruptPolarityType
    if (triggerMask & TriggerMask_LevelActiveHigh)
    {
        pOutInfo->polarity = InterruptPolarityType_ActiveHigh;
    }
    else if (triggerMask & TriggerMask_LevelActiveLow)
    {
        pOutInfo->polarity = InterruptPolarityType_ActiveLow;
    }
    else
    {
        pOutInfo->polarity = InterruptPolarityType_Unspecified;
    }

    return nn::ResultSuccess();
}

}}}
