﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/dt/dt_Result.h>
#include <nn/dt/dt_ResultPrivate.h>
#include <nn/dt/detail/dt_Log.h>
#include <nn/dt/detail/dt_PropertyApi.h>

#include <nn/result/result_HandlingUtility.h>

#include "dt_FdtReader.h"
#include "dt_FdtReaderUtil.h"

extern "C"
{
#include "libfdt.h"
}

namespace nn { namespace dt { namespace server {

namespace {
#if defined NN_SDK_BUILD_DEBUG || defined NN_SDK_BUILD_DEVELOP
    const char* GetFdtErrorName(int fdtError)
    {
        #define NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(name) case -name: return #name;

        switch (fdtError)
        {
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_NOTFOUND);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_EXISTS);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_NOSPACE);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_BADOFFSET);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_BADPATH);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_BADPHANDLE);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_BADSTATE);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_TRUNCATED);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_BADMAGIC);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_BADVERSION);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_BADSTRUCTURE);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_BADLAYOUT);
        NN_DETAIL_DT_CASE_RETURN_ERROR_NAME(FDT_ERR_INTERNAL);
        default:
            return "";
        }

        #undef NN_DETAIL_DT_CASE_RETURN_ERROR_NAME
    }
#endif

    nn::Result GetFdtResult(int fdtError)
    {
        #define NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(name, result) case -name: return result();

        switch (fdtError)
        {
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_NOTFOUND,     nn::dt::ResultLibFdtErrorNotFound);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_EXISTS,       nn::dt::ResultLibFdtErrorExists);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_NOSPACE,      nn::dt::ResultLibFdtErrorNoSpace);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_BADOFFSET,    nn::dt::ResultLibFdtErrorBadOffset);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_BADPATH,      nn::dt::ResultLibFdtErrorBadPath);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_BADPHANDLE,   nn::dt::ResultLibFdtErrorBadPHandle);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_BADSTATE,     nn::dt::ResultLibFdtErrorBadState);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_TRUNCATED,    nn::dt::ResultLibFdtErrorTruncated);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_BADMAGIC,     nn::dt::ResultLibFdtErrorBadMagic);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_BADVERSION,   nn::dt::ResultLibFdtErrorBadVersion);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_BADSTRUCTURE, nn::dt::ResultLibFdtErrorBadStructure);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_BADLAYOUT,    nn::dt::ResultLibFdtErrorBadLayout);
        NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT(FDT_ERR_INTERNAL,     nn::dt::ResultLibFdtErrorInternal);
        default:
            return nn::ResultSuccess();
        }

        #undef NN_DETAIL_DT_CASE_RETURN_ERROR_RESULT
    }

    NN_FORCEINLINE void ConvertEndian(char* buffer, size_t bufferSize, size_t elementSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(bufferSize % elementSize == 0);

        for (int i = 0; i < bufferSize / elementSize; i++)
        {
            char* ptr = buffer + elementSize * i;
            switch (elementSize)
            {
            case 1:
                break;
            case 2:
                *reinterpret_cast<uint16_t*>(ptr) = fdt16_to_cpu(*reinterpret_cast<fdt16_t*>(ptr));
                break;
            case 4:
                *reinterpret_cast<uint32_t*>(ptr) = fdt32_to_cpu(*reinterpret_cast<fdt32_t*>(ptr));
                break;
            case 8:
                *reinterpret_cast<uint64_t*>(ptr) = fdt64_to_cpu(*reinterpret_cast<fdt64_t*>(ptr));
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }

}

void FdtReader::Initialize(const nn::Bit8* pFdt, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFdt);
    m_pFdt = pFdt;
    m_FdtSize = size;
}

nn::Result FdtReader::GetNodePath(char* pOutBuffer, size_t bufferSize, Node* pNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pNode));

    auto ret = fdt_get_path(m_pFdt, pNode->GetOffset(), pOutBuffer, bufferSize);
    if (ret == -FDT_ERR_NOSPACE)
    {
        return nn::dt::ResultBufferTooSmall();
    }
    NN_RESULT_DO(CheckReturnedOffset(ret));

    return nn::ResultSuccess();
}

nn::Result FdtReader::DumpNode(Node* pNode, int depth) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pNode));
    PrintNode(m_pFdt, pNode, depth);
    return ResultSuccess();
}

nn::Result FdtReader::FindNodeByPath(Node* pOutNode, const char* path) NN_NOEXCEPT
{
    auto offset = fdt_path_offset(m_pFdt, path);
    if (offset == -FDT_ERR_BADPATH)
    {
        return nn::dt::ResultInvalidArgument();
    }
    if (offset == -FDT_ERR_NOTFOUND)
    {
        return nn::dt::ResultNodeNotExist();
    }
    NN_RESULT_DO(CheckReturnedOffset(offset));

    *pOutNode = Node(offset);
    return nn::ResultSuccess();
}

nn::Result FdtReader::FindNodeByPHandle(Node* pOutNode, PHandle phandle) NN_NOEXCEPT
{
    auto offset = fdt_node_offset_by_phandle(m_pFdt, phandle);
    if (offset == -FDT_ERR_BADPHANDLE)
    {
        return nn::dt::ResultInvalidArgument();
    }
    NN_RESULT_DO(CheckReturnedOffset(offset));

    *pOutNode = Node(offset);

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetCompatibleNodeCount(size_t* pOutCount, const char* compatible) NN_NOEXCEPT
{
    size_t count = 0;

    // compatible なノードの探索。fdt_node_offset_by_compatible() のコメント参照。
    auto offset = fdt_node_offset_by_compatible(m_pFdt, -1, compatible);
    while (IsValidNodeOffset(offset))
    {
        count++;
        offset = fdt_node_offset_by_compatible(m_pFdt, offset, compatible);
    }

    // 正常系では offset == -FDT_ERR_NOTFOUND となる
    if (offset != -FDT_ERR_NOTFOUND)
    {
        NN_RESULT_DO(CheckReturnedOffset(offset));
        NN_ABORT("Unexpected behavior of libfdt.\n");
    }

    *pOutCount = count;
    if (*pOutCount == 0)
    {
        return nn::dt::ResultNodeNotExist();
    }

    return nn::ResultSuccess();
}

nn::Result FdtReader::ListCompatibleNode(Node* pOutNodeList, size_t* pOutCount, size_t maxCount, const char* compatible) NN_NOEXCEPT
{
    size_t index = 0;

    // compatible なノードの探索。fdt_node_offset_by_compatible() のコメント参照。
    auto offset = fdt_node_offset_by_compatible(m_pFdt, -1, compatible);
    while (IsValidNodeOffset(offset))
    {
        if (index < maxCount)
        {
            Node node(offset);
            pOutNodeList[index] = node;
        }
        else
        {
            return nn::dt::ResultBufferTooSmall();
        }

        index++;
        offset = fdt_node_offset_by_compatible(m_pFdt, offset, compatible);
    }

    // 正常系では offset == -FDT_ERR_NOTFOUND となる
    if (offset != -FDT_ERR_NOTFOUND)
    {
        NN_RESULT_DO(CheckReturnedOffset(offset));
        NN_ABORT("Unexpected behavior of libfdt.\n");
    }

    *pOutCount = index;
    if (*pOutCount == 0)
    {
        return nn::dt::ResultNodeNotExist();
    }

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetParentNode(Node* pOutNode, Node* pCurrentNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pCurrentNode));

    auto parentOffset = fdt_parent_offset(m_pFdt, pCurrentNode->GetOffset());
    if (parentOffset == -FDT_ERR_NOTFOUND)
    {
        return nn::dt::ResultNodeNotExist();
    }
    NN_RESULT_DO(CheckReturnedOffset(parentOffset));

    *pOutNode = Node(parentOffset);
    return nn::ResultSuccess();
}

nn::Result FdtReader::GetChildNodeByName(Node* pOutNode, Node* pCurrentNode, const char* name) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pCurrentNode));

    int offset;
    fdt_for_each_subnode(offset, m_pFdt, pCurrentNode->GetOffset())
    {
        auto nodeName = fdt_get_name(m_pFdt, offset, nullptr);
        NN_SDK_ASSERT_NOT_NULL(nodeName);

        if (std::strcmp(name, nodeName) == 0)
        {
            *pOutNode = Node(offset);
            return nn::ResultSuccess();
        }
    }

    return nn::dt::ResultNodeNotExist();
}

nn::Result FdtReader::GetChildNodeCount(size_t* pOutCount, Node* pCurrentNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pCurrentNode));

    size_t count = 0;
    int offset;
    fdt_for_each_subnode(offset, m_pFdt, pCurrentNode->GetOffset())
    {
        count++;
    }

    *pOutCount = count;
    if (*pOutCount == 0)
    {
        return nn::dt::ResultNodeNotExist();
    }

    return nn::ResultSuccess();
}

nn::Result FdtReader::ListChildNode(Node* pOutNodeList, size_t* pOutCount, size_t maxCount, Node* pCurrentNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pCurrentNode));

    size_t index = 0;
    int offset;

    fdt_for_each_subnode(offset, m_pFdt, pCurrentNode->GetOffset())
    {
        if (index < maxCount)
        {
            Node node(offset);
            pOutNodeList[index] = node;
        }
        else
        {
            return nn::dt::ResultBufferTooSmall();
        }

        index++;
    }

    *pOutCount = index;
    if (*pOutCount == 0)
    {
        return nn::dt::ResultNodeNotExist();
    }

    return nn::ResultSuccess();
}


nn::Result FdtReader::GetFirstChildNode(Node* pOutNode, Node* pCurrentNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pCurrentNode));

    auto childOffset = fdt_first_subnode(m_pFdt, pCurrentNode->GetOffset());
    if (childOffset == -FDT_ERR_NOTFOUND)
    {
        return nn::dt::ResultNodeNotExist();
    }
    NN_RESULT_DO(CheckReturnedOffset(childOffset));

    *pOutNode = Node(childOffset);
    return nn::ResultSuccess();
}

nn::Result FdtReader::GetNextSiblingNode(Node* pOutNode, Node* pCurrentNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pCurrentNode));

    auto nextOffset = fdt_next_subnode(m_pFdt, pCurrentNode->GetOffset());
    if (nextOffset == -FDT_ERR_NOTFOUND)
    {
        return nn::dt::ResultNodeNotExist();
    }
    NN_RESULT_DO(CheckReturnedOffset(nextOffset));

    *pOutNode = Node(nextOffset);
    return nn::ResultSuccess();
}

nn::Result FdtReader::GetValueOfAddressCells(uint32_t* pOutValue, Node* pNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pNode));

    auto parent = fdt_parent_offset(m_pFdt, pNode->GetOffset());
    NN_RESULT_DO(CheckLibFdtError(parent));

    *pOutValue = fdt_address_cells(m_pFdt, parent);
    NN_RESULT_DO(CheckLibFdtError(*pOutValue));

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetValueOfSizeCells(uint32_t* pOutValue, Node* pNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pNode));

    auto parent = fdt_parent_offset(m_pFdt, pNode->GetOffset());
    NN_RESULT_DO(CheckLibFdtError(parent));

    *pOutValue = fdt_size_cells(m_pFdt, parent);
    NN_RESULT_DO(CheckLibFdtError(*pOutValue));

    return nn::ResultSuccess();
}

nn::Result FdtReader::IsPropertyExist(bool *pOutExist, Node* pNode, const char* name) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pNode));

    int lenp;
    auto ptr = fdt_getprop(m_pFdt, pNode->GetOffset(), name, &lenp);

    if (ptr == nullptr && lenp != -FDT_ERR_NOTFOUND)
    {
        NN_RESULT_DO(CheckLibFdtError(lenp));
        NN_ABORT("Unexpected behavior of libfdt.\n");
    }

    *pOutExist = (ptr != nullptr);

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetPropertySize(size_t *pOutSize, Node* pNode, const char* name) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pNode));

    int lenp;
    auto ptr = fdt_getprop(m_pFdt, pNode->GetOffset(), name, &lenp);

    if (ptr == nullptr)
    {
        if (lenp == -FDT_ERR_NOTFOUND)
        {
            return nn::dt::ResultPropertyNotExist();
        }
        NN_RESULT_DO(CheckLibFdtError(lenp));
        NN_ABORT("Unexpected behavior of libfdt.\n");
    }
    NN_ABORT_UNLESS(lenp >= 0);

    *pOutSize = lenp;

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetProperty(char* pOutValue, Node* pNode, const char* name, size_t elementSize) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pNode));

    NN_RESULT_DO(GetPropertyWithIndex(pOutValue, pNode, name, 0, elementSize));

    // プロパティが 2要素以上の配列であった場合にエラーになる点が GetPropertyWithIndex と異なる
    size_t size;
    NN_RESULT_DO(GetPropertySize(&size, pNode, name));

    if (size != elementSize)
    {
        return nn::dt::ResultPropertyTypeError();
    }

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetPropertyWithIndex(char* pOutValue, Node* pNode, const char* name, size_t index, size_t elementSize) NN_NOEXCEPT
{
    // 入力チェック
    NN_RESULT_DO(CheckInputNode(pNode));
    if (!detail::IsSupportedPropertySize(elementSize))
    {
        return nn::dt::ResultInvalidArgument();
    }

    // プロパティの先頭ポインタとサイズを取得
    int lenp;
    auto pProperty = reinterpret_cast<const char*>(fdt_getprop(m_pFdt, pNode->GetOffset(), name, &lenp));

    if (pProperty == nullptr)
    {
        if (lenp == -FDT_ERR_NOTFOUND)
        {
            return nn::dt::ResultPropertyNotExist();
        }
        NN_RESULT_DO(CheckLibFdtError(lenp));
        NN_ABORT("Unexpected behavior of libfdt.\n");
    }
    NN_ABORT_UNLESS(lenp >= 0);

    if (lenp % elementSize != 0)
    {
        return nn::dt::ResultPropertyTypeError();
    }
    if (lenp / elementSize <= index)
    {
        return nn::dt::ResultPropertyNotExist();
    }

    // オフセットを適用して、プロパティのデータをコピー
    std::memcpy(pOutValue, pProperty + elementSize * index, elementSize);

    ConvertEndian(pOutValue, elementSize, elementSize);

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetPropertyList(char* pOutList, size_t* pOutPropertySize, size_t maxCount, Node* pNode, const char* name, size_t elementSize) NN_NOEXCEPT
{
    const size_t bufferSize = elementSize * maxCount;

    // 入力チェック
    NN_RESULT_DO(CheckInputNode(pNode));
    if (!detail::IsSupportedPropertySize(elementSize))
    {
        return nn::dt::ResultInvalidArgument();
    }

    // プロパティの先頭ポインタとサイズを取得
    int lenp;
    auto pProperty = reinterpret_cast<const char*>(fdt_getprop(m_pFdt, pNode->GetOffset(), name, &lenp));

    if (pProperty == nullptr)
    {
        if (lenp == -FDT_ERR_NOTFOUND)
        {
            return nn::dt::ResultPropertyNotExist();
        }
        NN_RESULT_DO(CheckLibFdtError(lenp));
        NN_ABORT("Unexpected behavior of libfdt.\n");
    }
    NN_ABORT_UNLESS(lenp >= 0);

    if (lenp % elementSize != 0)
    {
        return nn::dt::ResultPropertyTypeError();
    }

    *pOutPropertySize = lenp;

    std::memcpy(pOutList, pProperty, bufferSize);

    ConvertEndian(pOutList, bufferSize, elementSize);

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetInterruptCount(size_t* pOutCount, Node* pNode) NN_NOEXCEPT
{
    int lenp;
    uint32_t interruptCells;

    NN_RESULT_DO(CheckInputNode(pNode));

    NN_RESULT_DO(GetValueOfInterruptCells(&interruptCells, pNode));
    const uint32_t interruptPropertySize = interruptCells * sizeof(uint32_t);

    // interrupts のサイズから、割り込み番号の数を計算
    fdt_getprop(m_pFdt, pNode->GetOffset(), "interrupts", &lenp);
    if (lenp == -FDT_ERR_NOTFOUND)
    {
        return nn::dt::ResultPropertyNotExist();
    }
    NN_RESULT_DO(CheckLibFdtError(lenp));

    if (lenp % interruptPropertySize != 0)
    {
        return nn::dt::ResultPropertyTypeError();
    }
    *pOutCount = lenp / interruptPropertySize;

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetInterruptList(InterruptInfo* pOutList, size_t* pOutActualCount, size_t maxCount, Node* pNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInputNode(pNode));

    int lenp;

    // 出力バッファサイズチェック
    size_t interruptCount;
    NN_RESULT_DO(GetInterruptCount(&interruptCount, pNode));

    size_t copyCount = std::min(interruptCount, maxCount);

    // interrupts プロパティ取得
    auto pInterrupts = reinterpret_cast<const uint32_t*>(fdt_getprop(m_pFdt, pNode->GetOffset(), "interrupts", &lenp));
    if (lenp == -FDT_ERR_NOTFOUND)
    {
        return nn::dt::ResultPropertyNotExist();
    }
    NN_RESULT_DO(CheckLibFdtError(lenp));

    uint32_t interruptCells;
    NN_RESULT_DO(GetValueOfInterruptCells(&interruptCells, pNode));

    for (int i = 0; i < copyCount; i++)
    {
        NN_RESULT_DO(GetInterruptInfo(&pOutList[i], &pInterrupts[interruptCells * i], interruptCells));
    }
    *pOutActualCount = interruptCount;

    return nn::ResultSuccess();
}

nn::Result FdtReader::GetRegisterAddressList(char* pOutList, size_t* pOutActualCount, size_t maxCount, Node* pNode, size_t elementSize) NN_NOEXCEPT
{
    return GetRegisterPropertyList(pOutList, pOutActualCount, maxCount, pNode, elementSize, RegisterPropertyType_Address);
}

nn::Result FdtReader::GetRegisterSizeList(char* pOutList, size_t* pOutActualCount, size_t maxCount, Node* pNode, size_t elementSize) NN_NOEXCEPT
{
    return GetRegisterPropertyList(pOutList, pOutActualCount, maxCount, pNode, elementSize, RegisterPropertyType_Size);
}

nn::Result FdtReader::GetRegisterPropertyRaw(const char** pPointerToProperty, size_t *pOutPropertySize, uint32_t* pOutAddressCells, uint32_t* pOutSizeCells, Node* pNode) NN_NOEXCEPT
{
    uint32_t addressCells;
    NN_RESULT_DO(GetValueOfAddressCells(&addressCells, pNode));

    uint32_t sizeCells;
    NN_RESULT_DO(GetValueOfSizeCells(&sizeCells, pNode));

    int lenp;
    auto pProperty = fdt_getprop(m_pFdt, pNode->GetOffset(), "reg", &lenp);

    if (pProperty == nullptr)
    {
        if (lenp == -FDT_ERR_NOTFOUND)
        {
            return nn::dt::ResultPropertyNotExist();
        }
        NN_RESULT_DO(CheckLibFdtError(lenp));
        NN_ABORT("Unexpected behavior of libfdt.\n");
    }
    NN_ABORT_UNLESS(lenp >= 0);

    const size_t pairSize = sizeof(uint32_t) * (addressCells + sizeCells);
    if (lenp % pairSize != 0)
    {
        return nn::dt::ResultFdtPropertyTypeError();
    }

    *pPointerToProperty = reinterpret_cast<const char*>(pProperty);
    *pOutPropertySize = lenp;
    *pOutAddressCells = addressCells;
    *pOutSizeCells = sizeCells;
    return nn::ResultSuccess();
}

nn::Result FdtReader::GetRegisterPropertyList(char* pOutList, size_t* pOutActualCount, size_t maxCount, Node* pNode, size_t elementSize, RegisterPropertyType type) NN_NOEXCEPT
{
    const char* pProperty;
    size_t propertySize;
    uint32_t addressCells;
    uint32_t sizeCells;

    NN_RESULT_DO(GetRegisterPropertyRaw(&pProperty, &propertySize, &addressCells, &sizeCells, pNode));
    size_t pairSize = sizeof(uint32_t) * (addressCells + sizeCells); // Byte size of <address, size> pair

    size_t expectedElementSize;
    switch (type)
    {
    case RegisterPropertyType_Address:
        expectedElementSize = sizeof(uint32_t) * addressCells;
        break;
    case RegisterPropertyType_Size:
        expectedElementSize = sizeof(uint32_t) * sizeCells;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    if (elementSize != expectedElementSize)
    {
        return nn::dt::ResultPropertyTypeError();
    }

    size_t copyCount = std::min(maxCount, propertySize / pairSize);

    for (int i = 0; i < copyCount; i++)
    {
        char* pDestination = pOutList + i * elementSize;
        size_t offset = GetOffsetInPropertyFrom(type, addressCells);
        std::memcpy(pDestination, pProperty + i * pairSize + offset, elementSize);
        ConvertEndian(pDestination, elementSize, elementSize);
    }

    *pOutActualCount = propertySize / pairSize;
    return nn::ResultSuccess();
}

nn::Result FdtReader::GetValueOfInterruptCells(uint32_t* pOutValue, Node* pNode) NN_NOEXCEPT
{
    int lenp;

    NN_RESULT_DO(CheckInputNode(pNode));

    // 祖先ノードも含めて interrupt-parent プロパティを検索
    auto offset = pNode->GetOffset();

    uint32_t interruptParentPHandle;
    while (true)
    {
        auto pProperty = fdt_getprop(m_pFdt, offset, "interrupt-parent", &lenp);
        if (pProperty != nullptr)
        {
            interruptParentPHandle = fdt32_to_cpu(*reinterpret_cast<const uint32_t*>(pProperty));
            break;
        }
        NN_ABORT_UNLESS(lenp == -FDT_ERR_NOTFOUND, "Unexpected behavior of libfdt.\n");

        offset = fdt_parent_offset(m_pFdt, offset);
        if (offset == -FDT_ERR_NOTFOUND || offset == -FDT_ERR_BADOFFSET)
        {
            return nn::dt::ResultPropertyNotExist();
        }
        NN_RESULT_DO(CheckReturnedOffset(offset));
    }

    // interrupt-parent のノードオフセット取得
    int interruptParentOffset = fdt_node_offset_by_phandle(m_pFdt, interruptParentPHandle);
    NN_RESULT_DO(CheckReturnedOffset(interruptParentOffset));

    // #interrupt-cells プロパティ取得
    const void* pInterruptCells = fdt_getprop(m_pFdt, interruptParentOffset, "#interrupt-cells", &lenp);
    if (lenp == -FDT_ERR_NOTFOUND)
    {
        return nn::dt::ResultPropertyNotExist();
    }
    NN_RESULT_DO(CheckLibFdtError(lenp));

    *pOutValue = fdt32_to_cpu(*reinterpret_cast<const int*>(pInterruptCells));
    return nn::ResultSuccess();
}

nn::Result FdtReader::CheckLibFdtError(int error) NN_NOEXCEPT
{
    if (error < 0)
    {
        NN_DETAIL_DT_WARN("libfdt returns %s (%d).\n", GetFdtErrorName(error), error);
        return GetFdtResult(error);
    }

    return nn::ResultSuccess();
}

nn::Result FdtReader::CheckReturnedOffset(int offset) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckLibFdtError(offset));
    NN_ABORT_UNLESS(IsValidNodeOffset(offset));

    return nn::ResultSuccess();
}

nn::Result FdtReader::CheckInputNode(Node* pNode) NN_NOEXCEPT
{
    if (pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    // 有効な node offset に対して fdt_get_name() は必ず成功する
    int lenp;
    if (fdt_get_name(m_pFdt, pNode->GetOffset(), &lenp) == nullptr && lenp == -FDT_ERR_BADOFFSET)
    {
        NN_DETAIL_DT_WARN("Node offset %zu is bad (m_FdtSize=%zu).\n", pNode->GetOffset(), m_FdtSize);
        return nn::dt::ResultInvalidArgument();
    }
    NN_RESULT_DO(CheckLibFdtError(lenp));

    return nn::ResultSuccess();
}

}}}
