﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os/os_InterruptEventCommon.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/dt/dt_Result.h>

#include "dt_FdtManager.h"
#include "dt_DeviceTreeInterface.h"

namespace nn { namespace dt { namespace server {

namespace {

NN_FORCEINLINE nn::Result CheckInBufferArgumentIsString(const nn::sf::InBuffer& buffer) NN_NOEXCEPT
{
    if (buffer.GetSize() <= 0 || buffer.GetPointerUnsafe()[buffer.GetSize() - 1] != '\0')
    {
        return nn::dt::ResultInvalidArgument();
    }
    return nn::ResultSuccess();
}

NN_FORCEINLINE nn::Result CheckOutBufferArgumentSize(const nn::sf::OutBuffer& buffer, size_t expectedSize) NN_NOEXCEPT
{
    if (buffer.GetSize() != expectedSize)
    {
        return nn::dt::ResultInvalidArgument();
    }
    return nn::ResultSuccess();
}

}

nn::Result DeviceTreeInterface::FindNodeByPath(nn::sf::Out<nn::dt::Node> outNode, const nn::sf::InBuffer& path) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInBufferArgumentIsString(path));

    Node node;
    auto result = GetFdtManager()->GetFdtReader()->FindNodeByPath(&node, path.GetPointerUnsafe());
    *outNode = node;
    return result;
}

nn::Result DeviceTreeInterface::FindNodeByPHandle(nn::sf::Out<nn::dt::Node> outNode, std::uint32_t phandle) NN_NOEXCEPT
{
    Node node;
    auto result = GetFdtManager()->GetFdtReader()->FindNodeByPHandle(&node, phandle);
    *outNode = node;
    return result;
}

nn::Result DeviceTreeInterface::GetCompatibleNodeCount(nn::sf::Out<std::uint64_t> outCount, const nn::sf::InBuffer& compatible) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInBufferArgumentIsString(compatible));

    size_t count;
    auto result = GetFdtManager()->GetFdtReader()->GetCompatibleNodeCount(&count, compatible.GetPointerUnsafe());
    *outCount = count;
    return result;
}

nn::Result DeviceTreeInterface::ListCompatibleNode(const nn::sf::OutBuffer& outNodeList, nn::sf::Out<std::uint64_t> outCount, std::uint64_t maxCount, const nn::sf::InBuffer& compatible) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInBufferArgumentIsString(compatible));
    NN_RESULT_DO(CheckOutBufferArgumentSize(outNodeList, sizeof(Node) * maxCount));

    size_t count;
    auto result = GetFdtManager()->GetFdtReader()->ListCompatibleNode(reinterpret_cast<Node*>(outNodeList.GetPointerUnsafe()), &count, maxCount, compatible.GetPointerUnsafe());
    *outCount = count;
    return result;
}

nn::Result DeviceTreeInterface::GetParentNode(nn::sf::Out<nn::dt::Node> outNode, nn::dt::Node currentNode) NN_NOEXCEPT
{
    Node node;
    auto result = GetFdtManager()->GetFdtReader()->GetParentNode(&node, &currentNode);
    *outNode = node;
    return result;
}

nn::Result DeviceTreeInterface::GetChildNodeByName(nn::sf::Out<nn::dt::Node> outNode, nn::dt::Node currentNode, const nn::sf::InBuffer& name) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInBufferArgumentIsString(name));

    Node node;
    auto result = GetFdtManager()->GetFdtReader()->GetChildNodeByName(&node, &currentNode, name.GetPointerUnsafe());
    *outNode = node;
    return result;
}

nn::Result DeviceTreeInterface::GetChildNodeCount(nn::sf::Out<std::uint64_t> outCount, nn::dt::Node currentNode) NN_NOEXCEPT
{
    size_t count;
    auto result = GetFdtManager()->GetFdtReader()->GetChildNodeCount(&count, &currentNode);
    *outCount = count;
    return result;
}

nn::Result DeviceTreeInterface::ListChildNode(const nn::sf::OutBuffer& outNodeList, nn::sf::Out<std::uint64_t> outCount, std::uint64_t maxCount, nn::dt::Node currentNode) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckOutBufferArgumentSize(outNodeList, sizeof(Node) * maxCount));

    size_t count;
    auto result = GetFdtManager()->GetFdtReader()->ListChildNode(reinterpret_cast<Node*>(outNodeList.GetPointerUnsafe()), &count, maxCount, &currentNode);
    *outCount = count;
    return result;
}

nn::Result DeviceTreeInterface::GetValueOfAddressCells(nn::sf::Out<std::uint32_t> outValue, nn::dt::Node node) NN_NOEXCEPT
{
    uint32_t value;
    auto result = GetFdtManager()->GetFdtReader()->GetValueOfAddressCells(&value, &node);
    *outValue = value;
    return result;
}

nn::Result DeviceTreeInterface::GetValueOfSizeCells(nn::sf::Out<std::uint32_t> outValue, nn::dt::Node node) NN_NOEXCEPT
{
    uint32_t value;
    auto result = GetFdtManager()->GetFdtReader()->GetValueOfSizeCells(&value, &node);
    *outValue = value;
    return result;
}

nn::Result DeviceTreeInterface::IsPropertyExist(nn::sf::Out<bool> outExist, nn::dt::Node node, const nn::sf::InBuffer& name) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInBufferArgumentIsString(name));

    bool exist;
    auto result = GetFdtManager()->GetFdtReader()->IsPropertyExist(&exist, &node, name.GetPointerUnsafe());
    *outExist = exist;
    return result;
}

nn::Result DeviceTreeInterface::GetPropertySize(nn::sf::Out<std::uint64_t> outSize, nn::dt::Node node, const nn::sf::InBuffer& name) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInBufferArgumentIsString(name));

    size_t size;
    auto result = GetFdtManager()->GetFdtReader()->GetPropertySize(&size, &node, name.GetPointerUnsafe());
    *outSize = size;
    return result;
}

nn::Result DeviceTreeInterface::GetProperty(const nn::sf::OutBuffer& outValue, nn::dt::Node node, const nn::sf::InBuffer& name, std::uint64_t elementSize) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInBufferArgumentIsString(name));
    NN_RESULT_DO(CheckOutBufferArgumentSize(outValue, elementSize));

    return GetFdtManager()->GetFdtReader()->GetProperty(outValue.GetPointerUnsafe(), &node, name.GetPointerUnsafe(), elementSize);
}

nn::Result DeviceTreeInterface::GetPropertyWithIndex(const nn::sf::OutBuffer& outValue, nn::dt::Node node, const nn::sf::InBuffer& name, std::uint64_t index, std::uint64_t elementSize) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInBufferArgumentIsString(name));
    NN_RESULT_DO(CheckOutBufferArgumentSize(outValue, elementSize));

    return GetFdtManager()->GetFdtReader()->GetPropertyWithIndex(outValue.GetPointerUnsafe(), &node, name.GetPointerUnsafe(), index, elementSize);
}

nn::Result DeviceTreeInterface::GetPropertyList(const nn::sf::OutBuffer& outArray, nn::sf::Out<std::uint64_t> outPropertySize, std::uint64_t maxCount, nn::dt::Node node, const nn::sf::InBuffer& name, std::uint64_t elementSize) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckInBufferArgumentIsString(name));
    NN_RESULT_DO(CheckOutBufferArgumentSize(outArray, elementSize * maxCount));

    size_t propertySize;
    auto result = GetFdtManager()->GetFdtReader()->GetPropertyList(outArray.GetPointerUnsafe(), &propertySize, maxCount, &node, name.GetPointerUnsafe(), elementSize);
    *outPropertySize = propertySize;
    return result;
}

nn::Result DeviceTreeInterface::GetInterruptCount(nn::sf::Out<std::uint64_t> outCount, nn::dt::Node node) NN_NOEXCEPT
{
    size_t count;
    auto result = GetFdtManager()->GetFdtReader()->GetInterruptCount(&count, &node);
    *outCount = count;
    return result;
}

nn::Result DeviceTreeInterface::GetInterruptList(const nn::sf::OutBuffer& outList, nn::sf::Out<std::uint64_t> outActualCount, std::uint64_t maxCount, nn::dt::Node node) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckOutBufferArgumentSize(outList, sizeof(nn::dt::InterruptInfo) * maxCount));

    size_t count;
    auto result = GetFdtManager()->GetFdtReader()->GetInterruptList(reinterpret_cast<InterruptInfo*>(outList.GetPointerUnsafe()), &count, maxCount, &node);
    *outActualCount = count;
    return result;
}

nn::Result DeviceTreeInterface::GetRegisterAddressList(const nn::sf::OutBuffer& outArray, nn::sf::Out<std::uint64_t> outActualCount, std::uint64_t maxCount, nn::dt::Node node, std::uint64_t elementSize) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckOutBufferArgumentSize(outArray, elementSize * maxCount));

    size_t actualCount;
    auto result = GetFdtManager()->GetFdtReader()->GetRegisterAddressList(outArray.GetPointerUnsafe(), &actualCount, maxCount, &node, elementSize);
    *outActualCount = actualCount;
    return result;
}

nn::Result DeviceTreeInterface::GetRegisterSizeList(const nn::sf::OutBuffer& outArray, nn::sf::Out<std::uint64_t> outActualCount, std::uint64_t maxCount, nn::dt::Node node, std::uint64_t elementSize) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckOutBufferArgumentSize(outArray, elementSize * maxCount));

    size_t actualCount;
    auto result = GetFdtManager()->GetFdtReader()->GetRegisterSizeList(outArray.GetPointerUnsafe(), &actualCount, maxCount, &node, elementSize);
    *outActualCount = actualCount;
    return result;
}

nn::Result DeviceTreeInterface::GetFirstChildNode(nn::sf::Out<nn::dt::Node> outNode, nn::dt::Node currentNode) NN_NOEXCEPT
{
    Node node;
    auto result = GetFdtManager()->GetFdtReader()->GetFirstChildNode(&node, &currentNode);
    *outNode = node;
    return result;
}

nn::Result DeviceTreeInterface::GetNextSiblingNode(nn::sf::Out<nn::dt::Node> outNode, nn::dt::Node currentNode) NN_NOEXCEPT
{
    Node node;
    auto result = GetFdtManager()->GetFdtReader()->GetNextSiblingNode(&node, &currentNode);
    *outNode = node;
    return result;
}

nn::Result DeviceTreeInterface::GetNodePath(const nn::sf::OutBuffer& outValue, nn::dt::Node node) NN_NOEXCEPT
{
    return GetFdtManager()->GetFdtReader()->GetNodePath(outValue.GetPointerUnsafe(), outValue.GetSize(), &node);
}

nn::Result DeviceTreeInterface::DumpNode(nn::dt::Node node, std::int32_t depth) NN_NOEXCEPT
{
    return GetFdtManager()->GetFdtReader()->DumpNode(&node, depth);
}

}}}  // namespace nn::dt
