﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/ro/detail/ro_RoExceptionInfo.h>
#include <nn/util/util_StringUtil.h>
#include "diag_SymbolTableParser-os.horizon.h"

namespace nn { namespace diag { namespace detail {

bool IsSystemSymbolImpl(uintptr_t address) NN_NOEXCEPT
{
    ro::detail::ExceptionInfo info;
    auto result = ro::detail::GetExceptionInfo(&info, reinterpret_cast<uintptr_t>(IsSystemSymbolImpl));
    NN_ABORT_UNLESS(result);
    return (info.moduleAddress <= address && address < info.moduleAddress + info.moduleSize);
}

uintptr_t GetNearestExportedSymbol(const char** pOutSymbolName, size_t* outSymbolSize, uintptr_t address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSymbolName);
    NN_SDK_REQUIRES_NOT_NULL(outSymbolSize);

    SymbolTableParser parser;
    if (!parser.Initialize(address))
    {
        return 0u;
    }

    int index = -1;
    uintptr_t addressMax = 0u;
    while (parser.MoveNext())
    {
        const auto symType = parser.GetType();
        if (!(symType == nn::ro::detail::Elf::STT_FUNC || symType == nn::ro::detail::Elf::STT_OBJECT))
        {
            continue;
        }

        const auto symAddress = parser.GetAddress();
        if (addressMax < symAddress && symAddress <= address)
        {
            index = parser.GetIndex();
            addressMax = symAddress;
        }
    }

    if (index >= 0)
    {
        parser.MoveTo(index);
        *pOutSymbolName = parser.GetName();
        *outSymbolSize = parser.GetSize();
    }

    return addressMax;
}

uintptr_t GetAddressFromSymbolName(uintptr_t address, const char* symbolName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(symbolName);

    SymbolTableParser parser;
    if (!parser.Initialize(address))
    {
        return 0u;
    }

    while (parser.MoveNext())
    {
        auto symType = parser.GetType();
        if (!(symType == nn::ro::detail::Elf::STT_FUNC || symType == nn::ro::detail::Elf::STT_OBJECT))
        {
            continue;
        }

        auto symName = parser.GetName();
        if (std::strcmp(symName, symbolName) == 0)
        {
            return parser.GetAddress();
        }
    }

    return 0u;
}

uintptr_t GetSymbolNameImpl(char* outNameBuffer, size_t nameBufferSize, uintptr_t address) NN_NOEXCEPT
{
    if (IsSystemSymbolImpl(address))
    {
        return 0u;
    }

    const char* symbolName;
    size_t symbolSize;
    auto symbolAddress = GetNearestExportedSymbol(&symbolName, &symbolSize, address);
    if (symbolAddress != 0u)
    {
        util::Strlcpy(outNameBuffer, symbolName, nameBufferSize);
    }
    return symbolAddress;
}

size_t GetSymbolSizeImpl(uintptr_t address) NN_NOEXCEPT
{
    const char* symbolName;
    size_t symbolSize;
    auto symbolAddress = GetNearestExportedSymbol(&symbolName, &symbolSize, address);

    if (symbolAddress != address)
    {
        return 0u;
    }

    return symbolSize;
}

}}} // nn::diag::detail
