﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace diag { namespace detail {

template <typename Holder, typename Context>
class ObserverManager
{
    NN_DISALLOW_COPY(ObserverManager);
    NN_DISALLOW_MOVE(ObserverManager);

public:
    ObserverManager() NN_NOEXCEPT :
        m_ObserverListHead(nullptr),
        m_pObserverListTail(&m_ObserverListHead),
        m_Mutex(false)
    {}

    ~ObserverManager() NN_NOEXCEPT
    {
        UnregisterAllObserver();
    }

    void RegisterObserver(Holder* observerHolder) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_Mutex); // TODO: Writer Lock にする

        NN_SDK_ASSERT(!observerHolder->isRegistered, "observerHolder is already registered");

        *m_pObserverListTail = observerHolder;
        m_pObserverListTail = &observerHolder->next;
        observerHolder->next = nullptr;
        observerHolder->isRegistered = true;
    }

    void UnregisterObserver(Holder* observerHolder) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_Mutex); // TODO: Writer Lock にする

        NN_SDK_ASSERT(observerHolder->isRegistered, "observerHolder is not registered");

        if (observerHolder == m_ObserverListHead)
        {
            m_ObserverListHead = observerHolder->next;

            if (&observerHolder->next == m_pObserverListTail)
            {
                m_pObserverListTail = &m_ObserverListHead;
            }
        }
        else
        {
            for (Holder* holder = m_ObserverListHead;
                holder != nullptr; holder = holder->next)
            {
                if (holder->next == observerHolder)
                {
                    holder->next = observerHolder->next;

                    if (&observerHolder->next == m_pObserverListTail)
                    {
                        m_pObserverListTail = &holder->next;
                    }

                    break;
                }
            }
        }

        observerHolder->isRegistered = false;
    }

    void InvokeAllObserver(const Context& context) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_Mutex); // TODO: Reader Lock にする

        for(Holder* holder = m_ObserverListHead;
            holder != nullptr; holder = holder->next)
        {
            holder->observer(context);
        }
    }

    void UnregisterAllObserver() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_Mutex); // TODO: Writer Lock にする

        for (Holder* holder = m_ObserverListHead;
            holder != nullptr; holder = holder->next)
        {
            holder->isRegistered = false;
        }

        m_ObserverListHead = nullptr;
        m_pObserverListTail = &m_ObserverListHead;
    }

private:
    Holder* m_ObserverListHead;
    Holder** m_pObserverListTail;
    os::Mutex m_Mutex;
};

}}} // nn::diag::detail
