﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <libunwind.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/svc/svc_Base.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace diag { namespace detail {

namespace {
    // unw_cursor_t, unw_context_t はとても大きいので、static に置いて排他的に利用する
    os::MutexType g_Mutex = NN_OS_MUTEX_INITIALIZER(false);
    unw_cursor_t  g_Cursor;
    unw_context_t g_Context;
}

namespace {
    bool CheckInstructionAddress(uintptr_t address) NN_NOEXCEPT
    {
        if (address == 0)
        {
            return false;
        }

        const size_t InstructionSize = 4;
        if (address & (InstructionSize - 1))
        {
            return false;
        }

        svc::MemoryInfo memoryInfo;
        svc::PageInfo   pageInfo;

        if (svc::QueryMemory(&memoryInfo, &pageInfo, address).IsFailure())
        {
            return false;
        }

        if (memoryInfo.permission != svc::MemoryPermission_ReadExecute)
        {
            return false;
        }

        return true;
    }

    bool CheckStackRange(uintptr_t address) NN_NOEXCEPT
    {
        if (address == 0)
        {
            return false;
        }

        if (address & (sizeof(uintptr_t) - 1))
        {
            return false;
        }

        uintptr_t   stack;
        size_t      stackSize;

        os::GetCurrentStackInfo(&stack, &stackSize);

        if (!(stack <= address && address < stack + stackSize))
        {
            return false;
        }

        return true;
    }
}

int GetBacktraceImpl(uintptr_t *pOutArray, int arrayCountMax) NN_NOEXCEPT
{
    os::LockMutex(&g_Mutex);
    NN_UTIL_SCOPE_EXIT { os::UnlockMutex(&g_Mutex); };

    if (unw_getcontext(&g_Context) != 0)
    {
        return 0;
    }

    if (unw_init_local(&g_Cursor, &g_Context) != 0)
    {
        return 0;
    }

    int currentCount = 0;

    while (currentCount < arrayCountMax)
    {
        if (!(unw_step(&g_Cursor) > 0))
        {
            break;
        }

        unw_word_t ip;
        if (!(unw_get_reg(&g_Cursor, UNW_REG_IP, &ip) == 0 && CheckInstructionAddress(ip)))
        {
            break;
        }

        unw_word_t sp;
        if (!(unw_get_reg(&g_Cursor, UNW_REG_SP, &sp) == 0 && CheckStackRange(sp)))
        {
            break;
        }

        unw_word_t fp;
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
        if (!(unw_get_reg(&g_Cursor, UNW_ARM64_FP, &fp) == 0 && CheckStackRange(fp)))
        {
            break;
        }
#elif defined(NN_BUILD_CONFIG_ADDRESS_32)
        if (!(unw_get_reg(&g_Cursor, UNW_ARM_R11, &fp) == 0 && CheckStackRange(fp)))
        {
            break;
        }
#else
        #error "未サポートのアドレスサイズです。"
#endif

        pOutArray[currentCount++] = static_cast<uintptr_t>(ip);
    }

    return currentCount;
}

}}} // nn::diag::detail
