﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/diag/diag_Backtrace.h>
#include <nn/os/os_Fiber.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/svc/svc_Base.h>

namespace nn { namespace diag {

namespace
{
    uintptr_t GetAddressValue(uintptr_t address)
    {
        if (address == 0)
        {
            return 0;
        }

        if (address & (sizeof(uintptr_t) - 1))
        {
            return 0;
        }

        return *reinterpret_cast<uintptr_t*>(address);
    }

    bool CheckInstructionAddress(uintptr_t address, const char* pMsg) NN_NOEXCEPT
    {
        if (address == 0)
        {
            NN_SDK_LOG("Failed to get backtrace. %s is null.\n", pMsg);
            return false;
        }

        const size_t InstructionSize = 4;
        if (address & (InstructionSize - 1))
        {
            NN_SDK_LOG("Failed to get backtrace. Invalid alignment of %s: %p.\n", pMsg, address);
            return false;
        }

        svc::MemoryInfo memoryInfo;
        svc::PageInfo   pageInfo;

        auto result = svc::QueryMemory(&memoryInfo, &pageInfo, address);
        if (result.IsFailure())
        {
            NN_SDK_LOG("Failed to get backtrace. Query memory failed: %p (%03d-%04d).\n",
                address, result.GetModule(), result.GetDescription());
            return false;
        }

        if (memoryInfo.permission != svc::MemoryPermission_ReadExecute)
        {
        #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
            char permR = (memoryInfo.permission & svc::MemoryPermission_Read)?     'r' : '-';
            char permW = (memoryInfo.permission & svc::MemoryPermission_Write)?    'w' : '-';
            char permX = (memoryInfo.permission & svc::MemoryPermission_Execute)?  'x' : '-';
            char permD = (memoryInfo.permission & svc::MemoryPermission_DontCare)? 'd' : '-';

            NN_SDK_LOG("Failed to get backtrace. Invalid memory permission of %s: %p (%c%c%c%c).\n", pMsg,
                address, permR, permW, permX, permD);
        #endif

            return false;
        }

        return true;
    }

    void GetStackInfo(uintptr_t* outStack, size_t* outStackSize) NN_NOEXCEPT
    {
        auto fiber = nn::os::GetCurrentFiber();
        if (fiber == nullptr)
        {
            auto thread = nn::os::GetCurrentThread();
            *outStack = reinterpret_cast<uintptr_t>(thread->_stack);
            *outStackSize = thread->_stackSize;
        }
        else
        {
            *outStack = reinterpret_cast<uintptr_t>(fiber->_stack);
            *outStackSize = fiber->_stackSize;
        }
    }

    bool CheckStackRange(uintptr_t address) NN_NOEXCEPT
    {
        if (address == 0)
        {
            // 正常系で null になるので、ログ出力しない。
            return false;
        }

        if (address & (sizeof(uintptr_t) - 1))
        {
            NN_SDK_LOG("Failed to get backtrace. Invalid alignment of stack (or frame) pointer: %p.\n", address);
            return false;
        }

        uintptr_t   stack;
        size_t      stackSize;

        GetStackInfo(&stack, &stackSize);

        if (!(stack <= address && address < stack + stackSize))
        {
            NN_SDK_LOG("Failed to get backtrace. Address out of stack (%p, %p-%p).\n", address, stack, stack + stackSize);
            return false;
        }

        return true;
    }
}

namespace detail
{
    int GetBacktraceImpl(uintptr_t *pOutArray, int arrayCountMax, uintptr_t fp, uintptr_t sp) NN_NOEXCEPT
    {
        if (!(CheckStackRange(fp) && CheckStackRange(sp)))
        {
            return 0;
        }

        int currentCount = 0;

        while (currentCount < arrayCountMax)
        {
            uintptr_t lr;

    #if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
            const uintptr_t v = GetAddressValue(fp);
            if (sp <= v && v < sp + nn::os::MemoryPageSize)
            {
                lr = GetAddressValue(fp + sizeof(fp));
                fp = GetAddressValue(fp);
            }
            else
            {
                lr = GetAddressValue(fp);
                fp = GetAddressValue(fp - sizeof(fp));
            }
    #elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
            lr = GetAddressValue(fp + sizeof(fp));
            fp = GetAddressValue(fp);
    #else
        #error "未サポートのアーキテクチャです。"
    #endif

            if (!(CheckStackRange(fp) && CheckInstructionAddress(lr, "Link register value on the stack")))
            {
                break;
            }

            pOutArray[currentCount++] = lr;
        }

        return currentCount;
    }

    int GetBacktraceImpl(uintptr_t *pOutArray, int arrayCountMax) NN_NOEXCEPT
    {
        uintptr_t fp, sp;

        asm volatile
        (
    #if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
            " mov %0, fp\n\t"
            " mov %1, sp\n\t" :
    #elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
            " mov %0, x29\n\t"
            " mov %1, sp\n\t" :
    #else
        #error "未サポートのアーキテクチャです。"
    #endif
            "=&r"(fp), "=&r"(sp) :
            :
            "memory"
        );

        // 末尾呼び出しの最適化が行われると、nn::diag::GetBacktrace() を呼び出した関数の
        // fp, lr がスタックフレームに保存されないため、戻り値を volatile 付きの定数に代入してから返す。
        volatile const auto addressCount = GetBacktraceImpl(pOutArray, arrayCountMax, fp, sp);

        return addressCount;
    }
}

int GetBacktrace(uintptr_t *pOutArray, int arrayCountMax, uintptr_t fp, uintptr_t sp, uintptr_t pc) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutArray);
    NN_SDK_REQUIRES_GREATER(arrayCountMax, 0);

    pOutArray[0] = pc;
    return detail::GetBacktraceImpl(pOutArray + 1, arrayCountMax - 1, fp, sp) + 1;
}

}} // nn::diag
