﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Macro.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Result.h>

#include <nn/dd/dd_PhysicalAddress.h>
#include <nn/dd/dd_Result.h>

namespace nn { namespace dd {
namespace detail {

inline void InitializePhysicalMemoryInfoImpl( PhysicalMemoryInfo* pOutInfo, const void* addr, size_t size ) NN_NOEXCEPT
{
    pOutInfo->physicalAddress    = 0;
    pOutInfo->virtualAddress     = reinterpret_cast<uintptr_t>(addr);
    pOutInfo->size               = 0;
    pOutInfo->_endVirtualAddress = reinterpret_cast<uintptr_t>(addr) + size;
}

inline Result QueryNextPhysicalMemoryInfoImpl( PhysicalMemoryInfo* pOutInfo ) NN_NOEXCEPT
{
    svc::PhysicalMemoryInfo info;

    uintptr_t nextVirtualAddress = pOutInfo->virtualAddress + pOutInfo->size;
    size_t    nextSize           = reinterpret_cast<size_t>(pOutInfo->_endVirtualAddress - nextVirtualAddress);

    if (nextVirtualAddress >= pOutInfo->_endVirtualAddress)
    {
        // 全領域のアドレス変換が完了
        return ResultEndOfQuery();
    }

    // SVC
    auto result = svc::QueryPhysicalAddress(&info, nextVirtualAddress);
    if (result.IsFailure())
    {
        // 現状 SVC が失敗する場合は以下のエラーと解釈する
        return ResultInvalidCurrentMemory();
    }

    uintptr_t offset = nextVirtualAddress - info.virtualAddress;
    pOutInfo->physicalAddress = info.physicalAddress + offset;
    pOutInfo->virtualAddress  = nextVirtualAddress;
    pOutInfo->size            = std::min<size_t>(info.size - offset, nextSize);

    return nn::ResultSuccess();
}

inline Result QuerySinglePhysicalAddressImpl( PhysicalAddress* pOutAddress, const void* addr, size_t size ) NN_NOEXCEPT
{
    PhysicalMemoryInfo pmi;

    InitializePhysicalMemoryInfoImpl(&pmi, addr, size);
    auto result = QueryNextPhysicalMemoryInfoImpl(&pmi);

    // ResultEndOfQuery が返る場合は assert（通常ないはず）
    NN_SDK_ASSERT(!ResultEndOfQuery::Includes(result), NN_TEXT("内部エラー：ResultEndOfQuery が返りました。"));

    if (result.IsFailure())
    {
        return result;
    }
    if (pmi.size < size)
    {
        // 単一のメモリ領域ではない
        return ResultNotSingleRegion();
    }

    *pOutAddress = pmi.physicalAddress;
    return nn::ResultSuccess();
}

}   // namespace detail
}}  // namespace nn::dd
