﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/dd/dd_Types.h>
#include <nn/dd/dd_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Result.h>

namespace nn { namespace dd {
namespace detail {

inline void StoreDataCacheImpl( const void* addr, size_t size ) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    // Aarch64 の場合は svc を呼ばずユーザーランドで Store する
    uintptr_t cacheTypeRegister = 0;
    uintptr_t cacheLineSize = 0;
    uintptr_t endAddr = reinterpret_cast<uintptr_t>(addr) + size;

    asm volatile ("mrs %0, ctr_el0" : "=r" (cacheTypeRegister));
    cacheLineSize = 4 << ((cacheTypeRegister >> 16) & 0xF);

    uintptr_t startAddr = reinterpret_cast<uintptr_t>(addr) & ~(cacheLineSize - 1);
    for (auto ptr=startAddr; ptr<endAddr; ptr += cacheLineSize)
    {
        asm volatile ("dc cvac, %0" : : "r" (ptr));
    }

    // Memory barrier
    asm volatile ("dsb sy" : : : "memory");
#else
    // TORIAEZU: nn::svc::StoreDataCache に置き換える予定（現時点では desc 未対応のためまだやらない）
    auto result = nn::svc::StoreProcessDataCache(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, reinterpret_cast<uintptr_t>( addr ), size);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
#endif
}

inline void FlushDataCacheImpl( const void* addr, size_t size ) NN_NOEXCEPT
{
    // TORIAEZU:
    //
    // size に応じてキャッシュ全体を Flush する svc::FlushEntireDataCache を呼ぶようにする
    // desc の修正を待って対応する予定
    //
    // バラつきはあるものの、size が 2MB～2.5MB を超えるあたりから全体を Flush した方が速い
    //
#if defined(NN_BUILD_CONFIG_CPU_ARM64)
    // Aarch64 の場合は svc を呼ばずユーザーランドで Flush する
    // ※ ここを変更する場合 nn::os にある FlushDataCacheImpl にも同様の修正を入れる必要がありそう
    uintptr_t cacheTypeRegister = 0;
    uintptr_t cacheLineSize = 0;
    uintptr_t endAddr = reinterpret_cast<uintptr_t>(addr) + size;

    asm volatile ("mrs %0, ctr_el0" : "=r" (cacheTypeRegister));
    cacheLineSize = 4 << ((cacheTypeRegister >> 16) & 0xF);

    uintptr_t startAddr = reinterpret_cast<uintptr_t>(addr) & ~(cacheLineSize - 1);
    for (auto ptr=startAddr; ptr<endAddr; ptr += cacheLineSize)
    {
        asm volatile ("dc civac, %0" : : "r" (ptr));
    }

    // Memory barrier
    asm volatile ("dsb sy" : : : "memory");
#else
    // TORIAEZU: nn::svc::FlushDataCache に置き換える予定（現時点では desc 未対応のためまだやらない）
    auto result = nn::svc::FlushProcessDataCache(nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, reinterpret_cast<uintptr_t>( addr ), size);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
#endif
}

inline void InvalidateDataCacheImpl( void* addr, size_t size ) NN_NOEXCEPT
{
    // 自プロセスの場合、Flush の方が速いため内部では Flush を呼ぶ
    FlushDataCacheImpl(addr, size);
}

inline Result InvalidateProcessDataCacheImpl( ProcessHandle handle, uint64_t addr, uint64_t size ) NN_NOEXCEPT
{
    auto result = nn::svc::InvalidateProcessDataCache(nn::svc::Handle(handle), addr, size);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultInvalidHandle )
        {
            return dd::ResultInvalidHandle();
        }
        NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
        {
            return dd::ResultInvalidMemoryState();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

inline Result StoreProcessDataCacheImpl( ProcessHandle handle, uint64_t addr, uint64_t size ) NN_NOEXCEPT
{
    auto result = nn::svc::StoreProcessDataCache(nn::svc::Handle(handle), addr, size);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultInvalidHandle )
        {
            return dd::ResultInvalidHandle();
        }
        NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
        {
            return dd::ResultInvalidMemoryState();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

inline Result FlushProcessDataCacheImpl( ProcessHandle handle, uint64_t addr, uint64_t size ) NN_NOEXCEPT
{
    auto result = nn::svc::FlushProcessDataCache(nn::svc::Handle(handle), addr, size);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultInvalidHandle )
        {
            return dd::ResultInvalidHandle();
        }
        NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
        {
            return dd::ResultInvalidMemoryState();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

}   // namespace detail
}}  // namespace nn::dd
