﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/dd/dd_DeviceAddressSpaceApi.h>
#include <nn/dd/dd_DeviceAddressSpaceTypes.h>
#include <nn/dd/dd_Types.h>
#include <nn/dd/dd_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_NativeHandleTypes.h>

#include "detail/dd_DeviceAddressSpaceImpl.h"

namespace nn { namespace dd {

//-----------------------------------------------------------------------------
// デバイスアドレス空間オブジェクトの作成
//
Result CreateDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, uint64_t address, uint64_t size) NN_NOEXCEPT
{
    NN_DD_ASSERT_PAGESIZE_ALIGNED(address);
    NN_DD_ASSERT_PAGESIZE_ALIGNED(size);
    NN_SDK_ASSERT(size > 0, NN_TEXT("size に 0 が指定されています。"));

    DeviceAddressSpaceHandle    handle;
    auto result = detail::DeviceAddressSpaceImpl::Create(&handle, address, size);
    if (!result.IsSuccess())
    {
        deviceSpace->_state = DeviceAddressSpaceType::State_NotInitialized;
        return result;
    }

    deviceSpace->_deviceHandle      = handle;
    deviceSpace->_isHandleManaged   = true;
    deviceSpace->_state = DeviceAddressSpaceType::State_Initialized;
    NN_RESULT_SUCCESS;
}

Result CreateDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, uint64_t size) NN_NOEXCEPT
{
    return CreateDeviceAddressSpace( deviceSpace, 0, size );
}


//--------------------------------------------------------------------------
//  デバイスアドレス空間オブジェクトへのハンドルのアタッチ
void AttachDeviceAddressSpaceHandle(DeviceAddressSpaceType* deviceSpace, nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT
{
    NN_ABORT_UNLESS( handle != nn::os::InvalidNativeHandle,
        NN_TEXT("handle が有効なハンドルではありません。") );

    deviceSpace->_deviceHandle      = handle;
    deviceSpace->_isHandleManaged   = managed;
    deviceSpace->_state = DeviceAddressSpaceType::State_Initialized;
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間オブジェクトの破棄
//
void DestroyDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace) NN_NOEXCEPT
{
    NN_SDK_ASSERT( deviceSpace->_state == DeviceAddressSpaceType::State_Initialized, NN_TEXT("deviceSpace オブジェクトが初期化状態ではありません。") );

    if (deviceSpace->_isHandleManaged)
    {
        auto deviceHandle = deviceSpace->_deviceHandle;
        detail::DeviceAddressSpaceImpl::Close( deviceHandle );
    }

    deviceSpace->_state = DeviceAddressSpaceType::State_NotInitialized;
    deviceSpace->_deviceHandle      = 0;
    deviceSpace->_isHandleManaged   = false;
}


//--------------------------------------------------------------------------
//  デバイスアドレス空間ハンドルの取得
nn::os::NativeHandle GetDeviceAddressSpaceHandle(DeviceAddressSpaceType* deviceSpace) NN_NOEXCEPT
{
    NN_SDK_ASSERT( deviceSpace->_state == DeviceAddressSpaceType::State_Initialized, NN_TEXT("deviceSpace オブジェクトが初期化状態ではありません。") );

    return deviceSpace->_deviceHandle;
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間へのマップ（Aligned 版）
//
Result MapDeviceAddressSpaceAligned(DeviceAddressSpaceType* deviceSpace, ProcessHandle processHandle, uint64_t processAddress, size_t size, DeviceVirtualAddress deviceAddress, nn::dd::MemoryPermission devicePermission) NN_NOEXCEPT
{
    NN_SDK_ASSERT( deviceSpace->_state == DeviceAddressSpaceType::State_Initialized, NN_TEXT("deviceSpace オブジェクトが初期化状態ではありません。") );

    NN_DD_ASSERT_PAGESIZE_ALIGNED(processAddress);
    NN_DD_ASSERT_PAGESIZE_ALIGNED(deviceAddress);
    NN_DD_ASSERT_PAGESIZE_ALIGNED(size);
    NN_SDK_ASSERT(processAddress + size > processAddress);
    NN_SDK_ASSERT(deviceAddress  + size > deviceAddress );
    NN_SDK_ASSERT(size > 0, NN_TEXT("size に 0 が指定されています。"));

    NN_SDK_ASSERT((processAddress & 0x3fffff) == (deviceAddress & 0x3fffff),
        NN_TEXT("processAddress と deviceAddress の下位 22bit が一致していません"));

    auto deviceHandle = deviceSpace->_deviceHandle;
    return detail::DeviceAddressSpaceImpl::MapAligned( deviceHandle,
                                                       processHandle,
                                                       processAddress,
                                                       size,
                                                       deviceAddress,
                                                       devicePermission );
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間へのマップ（NotAligned 版）
//
Result MapDeviceAddressSpaceNotAligned(DeviceAddressSpaceType* deviceSpace, ProcessHandle processHandle, uint64_t processAddress, size_t size, DeviceVirtualAddress deviceAddress, nn::dd::MemoryPermission devicePermission) NN_NOEXCEPT
{
    NN_SDK_ASSERT( deviceSpace->_state == DeviceAddressSpaceType::State_Initialized, NN_TEXT("deviceSpace オブジェクトが初期化状態ではありません。") );

    NN_DD_ASSERT_PAGESIZE_ALIGNED(processAddress);
    NN_DD_ASSERT_PAGESIZE_ALIGNED(deviceAddress);
    NN_DD_ASSERT_PAGESIZE_ALIGNED(size);
    NN_SDK_ASSERT(processAddress + size > processAddress);
    NN_SDK_ASSERT(deviceAddress  + size > deviceAddress );
    NN_SDK_ASSERT(size > 0, NN_TEXT("size に 0 が指定されています。"));

    auto deviceHandle = deviceSpace->_deviceHandle;
    return detail::DeviceAddressSpaceImpl::MapNotAligned( deviceHandle,
                                                          processHandle,
                                                          processAddress,
                                                          size,
                                                          deviceAddress,
                                                          devicePermission );
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間からのアンマップ（改訂版）
//
void UnmapDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, ProcessHandle processHandle, uint64_t processAddress, size_t size, DeviceVirtualAddress deviceAddress) NN_NOEXCEPT
{
    NN_SDK_ASSERT( deviceSpace->_state == DeviceAddressSpaceType::State_Initialized, NN_TEXT("deviceSpace オブジェクトが初期化状態ではありません。") );

    NN_DD_ASSERT_PAGESIZE_ALIGNED(processAddress);
    NN_DD_ASSERT_PAGESIZE_ALIGNED(deviceAddress);
    NN_DD_ASSERT_PAGESIZE_ALIGNED(size);
    NN_SDK_ASSERT(processAddress + size > processAddress);
    NN_SDK_ASSERT(deviceAddress  + size > deviceAddress );
    NN_SDK_ASSERT(size > 0, NN_TEXT("size に 0 が指定されています。"));

    auto deviceHandle  = deviceSpace->_deviceHandle;
    detail::DeviceAddressSpaceImpl::Unmap( deviceHandle,
                                           processHandle,
                                           processAddress,
                                           size,
                                           deviceAddress );
}


//-----------------------------------------------------------------------------
// DeviceAddressSpaceMapInfo 構造体の初期化（インクリメンタル版）
//
void InitializeDeviceAddressSpaceMapInfo(DeviceAddressSpaceMapInfo* info, DeviceAddressSpaceType* deviceSpace, ProcessHandle processHandle, uint64_t processAddress, size_t size, DeviceVirtualAddress deviceAddress, MemoryPermission devicePermission) NN_NOEXCEPT
{
    NN_SDK_ASSERT( deviceSpace->_state == DeviceAddressSpaceType::State_Initialized, NN_TEXT("deviceSpace オブジェクトが初期化状態ではありません。") );

    NN_DD_ASSERT_PAGESIZE_ALIGNED(processAddress);
    NN_DD_ASSERT_PAGESIZE_ALIGNED(deviceAddress);
    NN_DD_ASSERT_PAGESIZE_ALIGNED(size);
    NN_SDK_ASSERT(processAddress + size > processAddress);
    NN_SDK_ASSERT(deviceAddress  + size > deviceAddress );
    NN_SDK_ASSERT(size > 0, NN_TEXT("size に 0 が指定されています。"));

    info->_lastMappedSize       = 0;
    info->_processAddress       = processAddress;
    info->_size                 = size;
    info->_deviceStartAddress   = deviceAddress;
    info->_deviceEndAddress     = deviceAddress + size;
    info->_processHandle        = processHandle;
    info->_devicePermission     = devicePermission;
    info->_pDeviceAddressSpace  = deviceSpace;
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間へのマップ（インクリメンタル版）
//
Result MapNextDeviceAddressSpaceRegion(size_t* outMappedSize, DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT
{
    NN_SDK_ASSERT( info->_lastMappedSize == 0,
        NN_TEXT("一つ前にマップした領域がまだアンマップされていません。"));

    size_t mappedSize = 0;
    if (info->_deviceStartAddress < info->_deviceEndAddress)
    {
        auto result = detail::DeviceAddressSpaceImpl::MapPartially(
                                &mappedSize,
                                info->_pDeviceAddressSpace->_deviceHandle,
                                info->_processHandle,
                                info->_processAddress,
                                info->_size,
                                info->_deviceStartAddress,
                                info->_devicePermission );
        if (!result.IsSuccess())
        {
            return result;
        }
    }

    // マップに成功した場合はサイズ情報だけ設定する。
    // 次アドレスの加算は UnmapDeviceAddressSpaceRegion() で行なう。
    info->_lastMappedSize   = mappedSize;
    *outMappedSize          = mappedSize;
    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間からのアンマップ（インクリメンタル版）
//
void UnmapDeviceAddressSpaceRegion(DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT
{
    auto lastMappedSize = info->_lastMappedSize;

    NN_SDK_ASSERT( lastMappedSize > 0,
        NN_TEXT("事前に MapNextDeviceAddressSpaceRegion() が呼ばれていません。"));

    detail::DeviceAddressSpaceImpl::Unmap(
                                info->_pDeviceAddressSpace->_deviceHandle,
                                info->_processHandle,
                                info->_processAddress,
                                lastMappedSize,
                                info->_deviceStartAddress);

    info->_lastMappedSize       = 0;
    info->_processAddress       += lastMappedSize;
    info->_deviceStartAddress   += lastMappedSize;
}


//-----------------------------------------------------------------------------
// DeviceAddressSpaceMapInfo 内のプロセスアドレス取得（インクリメンタル版）
//
uint64_t GetMappedProcessAddress(DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT
{
    NN_SDK_ASSERT( info->_lastMappedSize > 0,
        NN_TEXT("事前に MapNextDeviceAddressSpaceRegion() が呼ばれていません。"));

    return info->_processAddress;
}

//-----------------------------------------------------------------------------
// DeviceAddressSpaceMapInfo 内のデバイスアドレス取得（インクリメンタル版）
//
DeviceVirtualAddress GetMappedDeviceVirtualAddress(DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT
{
    NN_SDK_ASSERT( info->_lastMappedSize > 0,
        NN_TEXT("事前に MapNextDeviceAddressSpaceRegion() が呼ばれていません。"));

    return info->_deviceStartAddress;
}

//-----------------------------------------------------------------------------
// DeviceAddressSpaceMapInfo 内のサイズ取得（インクリメンタル版）
//
size_t GetMappedSize(DeviceAddressSpaceMapInfo* info) NN_NOEXCEPT
{
    return info->_lastMappedSize;
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間オブジェクトをデバイスグループにアタッチ
//
Result AttachDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, DeviceName deviceName) NN_NOEXCEPT
{
    NN_SDK_ASSERT( deviceSpace->_state == DeviceAddressSpaceType::State_Initialized, NN_TEXT("deviceSpace オブジェクトが初期化状態ではありません。") );

    return detail::DeviceAddressSpaceImpl::Attach(deviceSpace, deviceName);
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間オブジェクトをデバイスグループからデタッチ
//
void DetachDeviceAddressSpace(DeviceAddressSpaceType* deviceSpace, DeviceName deviceName) NN_NOEXCEPT
{
    NN_SDK_ASSERT( deviceSpace->_state == DeviceAddressSpaceType::State_Initialized, NN_TEXT("deviceSpace オブジェクトが初期化状態ではありません。") );

    detail::DeviceAddressSpaceImpl::Detach(deviceSpace, deviceName);
}


//-----------------------------------------------------------------------------
//  ProcessHandle のクローズ
//
void CloseProcessHandle(ProcessHandle handle) NN_NOEXCEPT
{
    if (handle == nn::os::InvalidNativeHandle)
    {
        return;
    }

    detail::DeviceAddressSpaceImpl::Close( handle );
}

//-----------------------------------------------------------------------------

}}  // namespace nn::dd
