﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/os.h>
#include <nn/socket.h>
#include <cstdint>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <nn\socket\net\ethernet.h>
#include <nn\socket\socket_ApiPrivate.h>
#include <if_types.h>
#include "cfg_DumpRoute.h"


namespace nn { namespace bsdsocket { namespace cfg {

struct bits {
    u_long  b_mask;
    char    b_val;
} bits[] = {
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Up),   'U' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Gateway),  'G' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Host), 'H' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Reject),   'R' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Dynamic),  'D' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Modified), 'M' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Done), 'd' }, /* Completed -- for routing messages only */
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_XResolve), 'X' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Static),   'S' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Proto1),   '1' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Proto2),   '2' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Proto3),   '3' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_BlackHole),'B' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_Broadcast),'b' },
    { static_cast<u_long>(nn::socket::RtFlag::Rtf_LlInfo),   'L' },
    { 0 , 0 }
};


/*
* kvm(3) bindings for every needed symbol
*/

enum
{
    MAXHOSTNAMELEN = 256,
    CTL_NET = 4,
};



static int Wflag = 1;
static int Aflag = 0;
static int numeric_addr = 1;

#define roundup(x, y) ((((x) + ((y) - 1))/(y)) * (y)) /* to any y */

/* column widths; each followed by one space */
#define WID_IF_DEFAULT(af)  ((af) == nn::socket::Family::Af_Inet6 ? 8 : (Wflag ? 10 : 8))

typedef union {
    long    dummy;  /* Helps align structure. */
    nn::socket::SockAddr u_sa;
    u_short u_data[128];
} sa_u;




char ifmap[16][nn::socket::IfNamSiz];
static int ifmap_size;


static int wid_dst;
static int wid_gw;
static int wid_flags;
static int wid_pksent;
static int wid_mtu;
static int wid_if;
static int wid_expire;


static void
p_rtentry_sysctl(nn::socket::RtMsgHdr *rtm);

static void
p_sockaddr(nn::socket::SockAddr *sa, nn::socket::SockAddr *mask, nn::socket::RtFlag flags, int width);

static const char *
fmt_sockaddr(nn::socket::SockAddr *sa, nn::socket::SockAddr *mask, nn::socket::RtFlag flags);

char *
routename(nn::socket::InAddrT in);

char *
netname(nn::socket::InAddrT in, nn::socket::InAddrT mask);

static void
p_flags(nn::socket::RtFlag f, const char *format);

static void
domask(char *dst, nn::socket::InAddrT addr, u_long mask);

static const char *
fmt_flags(int f);

char *
link_ntoa(const nn::socket::SockAddrDl *sdl);

char *
ether_ntoa (const nn::socket::EtherAddr *addr);

static void
size_cols(nn::socket::Family ef, struct radix_node *rn)
{
    NN_UNUSED(rn);
    wid_dst = 18;
    wid_gw = 18;
    wid_flags = 6;
    wid_pksent = 8;
    wid_mtu = 6;
    wid_if = WID_IF_DEFAULT(ef);
    wid_expire = 6;

    /*if (Wflag && rn != NULL)
    size_cols_tree(rn);*/
}

/*
* Print address family header before a section of the routing table.
*/
void
pr_family(nn::socket::Family af1)
{
    const char *afname;

    switch (static_cast<int>(af1)) {
    case static_cast<int>(nn::socket::Family::Af_Inet):
        afname = "Internet";
        break;
#ifdef INET6
    case static_cast<int>(nn::socket::Family::Af_Inet6):
        afname = "Internet6";
        break;
#endif /*INET6*/
    case static_cast<int>(nn::socket::FamilyPrivate::Af_Ipx):
        afname = "IPX";
        break;
    case static_cast<int>(nn::socket::FamilyPrivate::Af_Iso):
        afname = "ISO";
        break;
    case static_cast<int>(nn::socket::FamilyPrivate::Af_AppleTalk):
        afname = "AppleTalk";
        break;
    case static_cast<int>(nn::socket::FamilyPrivate::Af_Ccitt):
        afname = "X.25";
        break;
    case static_cast<int>(nn::socket::FamilyPrivate::Af_NetGraph):
        afname = "Netgraph";
        break;
    default:
        afname = NULL;
        break;
    }
    if (afname)
        printf("\n%s:\n", afname);
    else
        printf("\nProtocol Family %d:\n", static_cast<int>(af1));
}

void
pr_rthdr(nn::socket::Family af1)
{
    NN_UNUSED(af1);
    if (Aflag)
        printf("%-8.8s ","Address");
    if (Wflag) {
        printf("%-*.*s %-*.*s %-*.*s %*.*s %*.*s %*.*s %*s\n",
            wid_dst,    wid_dst,    "Destination",
            wid_gw,     wid_gw,     "Gateway",
            wid_flags,  wid_flags,  "Flags",
            wid_pksent, wid_pksent, "Use",
            wid_mtu,    wid_mtu,    "Mtu",
            wid_if,     wid_if,     "Netif",
            wid_expire,         "Expire");
    } else {
        printf("%-*.*s %-*.*s %-*.*s  %*.*s %*s\n",
            wid_dst,    wid_dst,    "Destination",
            wid_gw,     wid_gw,     "Gateway",
            wid_flags,  wid_flags,  "Flags",
            wid_if,     wid_if,     "Netif",
            wid_expire,         "Expire");
    }
}

void DumpRoute()
{
    nn::socket::Family af = nn::socket::Family::Af_Inet;
    int fibnum = 0;

    size_t needed;
    int mib[7];
    char *buf, *next, *lim;
    nn::socket::RtMsgHdr *rtm;
    nn::socket::SockAddr *sa;
    nn::socket::Family fam = nn::socket::Family::Af_Unspec;

    // should never have this many interfaces
    nn::socket::IfReq ibuf[16];
    nn::socket::IfConf ifc;
    nn::socket::IfReq *ifrp, *ifend;

    printf("Routing Tables");


    int sock = nn::socket::SocketExempt(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    if(!sock)
    {
        printf("\n Socket Creation Error! Interface names will be missing.\n");
    }

    ifc.ifc_len = sizeof(ibuf);
    ifc.ifc_buf = (caddr_t)ibuf;

    if (!sock || ((nn::socket::Ioctl(sock, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocGIfConf), (char *)&ifc, sizeof(ifc)) < 0) ||
        (static_cast<size_t>(ifc.ifc_len) < sizeof(struct ifreq))))
    {
        printf("\n SIOCGIFCONF Ioctl Error! Interface names will be missing.\n");
    }

    ifrp = ibuf;
    ifend = (nn::socket::IfReq *)((char *)ibuf + ifc.ifc_len);
    ifmap_size = 0;
    while (ifrp < ifend)
    {
        size_t n;
       // printf("ifrp->ifr_name  = %s\n",ifrp->ifr_name);
        if (!sock || nn::socket::Ioctl(sock, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocGIfIndex), ifrp,sizeof(struct ifreq)) != 0)
        {
            nn::socket::Errno lastErrno = nn::socket::GetLastError();
            if(lastErrno != nn::socket::Errno::ENxIo)
            {
                printf("SIOCGIFCONF Ioctl Error: %d\n",lastErrno );
            }
        }
        else
        {
            if(ifrp->ifr_index >= 16)
            {
                printf("\nError! Interface index out of bounds! (>16)\n");
            }
            else
            {
                strcpy(ifmap[ifrp->ifr_index],ifrp->ifr_name);
                ++ifmap_size;
            }
        }
        /* Bump interface config pointer */
        n = ifrp->ifr_addr.sa_len + sizeof(ifrp->ifr_name);
        if (n < sizeof(*ifrp)) n = sizeof(*ifrp);
        ifrp = (nn::socket::IfReq *)((char *)ifrp + n);
    }

    if(sock)
    {
        nn::socket::Shutdown(sock, nn::socket::ShutdownMethod::Shut_Rd);
    }
    /*
    * Retrieve interface list at first
    * since we need #ifindex -> if_xname match
    */
    /*if (getifaddrs(&ifap) != 0)
    err(EX_OSERR, "getifaddrs");

    for (ifa = ifap; ifa; ifa = ifa->ifa_next) {

    if (ifa->ifa_addr->sa_family != nn::socket::Family::Af_Link)
    continue;

    sdl = (nn::socket::SockAddrDl *)ifa->ifa_addr;
    ifindex = sdl->sdl_index;

    if (ifindex >= ifmap_size) {
    size = roundup(ifindex + 1, 32) *
    sizeof(struct ifmap_entry);
    if ((ifmap = realloc(ifmap, size)) == NULL)
    errx(2, "realloc(%d) failed", size);
    memset(&ifmap[ifmap_size], 0,
    size - ifmap_size *
    sizeof(struct ifmap_entry));

    ifmap_size = roundup(ifindex + 1, 32);
    }

    if (*ifmap[ifindex].ifname != '\0')
    continue;

    strlcpy(ifmap[ifindex].ifname, ifa->ifa_name, nn::socket::IfNamSiz);
    }

    freeifaddrs(ifap);*/

    mib[0] = CTL_NET;
    mib[1] = static_cast<int>(nn::socket::Family::Pf_Route);
    mib[2] = 0;
    mib[3] = static_cast<int>(af);
    mib[4] = nn::socket::Net_Rt_Dump;
    mib[5] = 0;
    mib[6] = fibnum;
    if (nn::socket::Sysctl(mib, sizeof(mib) / sizeof(int), NULL, &needed, NULL, 0) < 0)
        printf("Error sysctl: net.route.0.%d.dump.%d estimate", static_cast<int>(af),
        fibnum);
    if ((buf = (char *)malloc(needed)) == NULL)
        printf("Error malloc(%lu)", (unsigned long)needed);
    if (nn::socket::Sysctl(mib, sizeof(mib) / sizeof(int), buf, &needed, NULL, 0) < 0)
        printf("Error sysctl: net.route.0.%d.dump.%d", static_cast<int>(af), fibnum);
    lim  = buf + needed;
    for (next = buf; next < lim; next += rtm->rtm_msglen) {
        rtm = (nn::socket::RtMsgHdr *)next;
        if (rtm->rtm_version != nn::socket::Rtm_Version)
            continue;
        /*
        * Peek inside header to determine AF
        */
        sa = (nn::socket::SockAddr *)(rtm + 1);
        if (fam != sa->sa_family) {
            fam = sa->sa_family;
            size_cols(fam, NULL);
            pr_family(fam);
            pr_rthdr(fam);
        }
        p_rtentry_sysctl(rtm);
    }
    free(buf);
}

static void
p_rtentry_sysctl(nn::socket::RtMsgHdr *rtm)
{
    nn::socket::SockAddr *sa = (nn::socket::SockAddr *)(rtm + 1);
    char buffer[128];
    char prettyname[128];
    sa_u addr, mask, gw;
    unsigned int l;

    /*#define	GETSA(_s, _f)	{ \
    memset(&(_s),0, sizeof(_s)); \
    if (rtm->rtm_addrs & _f) { \
    l = roundup(sa->sa_len, sizeof(long)); \
    memcpy(&(_s), sa, (l > sizeof(_s)) ? sizeof(_s) : l); \
    sa = (nn::socket::SockAddr *)((char *)sa + l); \
    } \
    }*/
    //GETSA(addr, nn::socket::RtaFlag::Rta_Dst);
    memset(&(addr),0, sizeof(addr));
    if ((rtm->rtm_addrs & nn::socket::RtaFlag::Rta_Dst) != nn::socket::RtaFlag::Rta_None)
    {
        //roundup
        l = roundup(sa->sa_len, sizeof(long));
        memcpy(&(addr), sa, (l > sizeof(addr)) ? sizeof(addr) : l);
        sa = (nn::socket::SockAddr *)((char *)sa + l);
    }

    //GETSA(gw, nn::socket::RtaFlag::Rta_Gateway);
    memset(&(gw),0, sizeof(gw));
    if ((rtm->rtm_addrs & nn::socket::RtaFlag::Rta_Gateway) != nn::socket::RtaFlag::Rta_None)
    {
        //roundup
        l = roundup(sa->sa_len, sizeof(long));
        memcpy(&(gw), sa, (l > sizeof(gw)) ? sizeof(gw) : l);
        sa = (nn::socket::SockAddr *)((char *)sa + l);
    }

    //GETSA(mask, nn::socket::RtaFlag::Rta_NetMask);
    memset(&(mask),0, sizeof(mask));
    if ((rtm->rtm_addrs & nn::socket::RtaFlag::Rta_NetMask) != nn::socket::RtaFlag::Rta_None)
    {
        //roundup
        l = roundup(sa->sa_len, sizeof(long));
        memcpy(&(mask), sa, (l > sizeof(mask)) ? sizeof(mask) : l);
        sa = (nn::socket::SockAddr *)((char *)sa + l);
    }

    p_sockaddr(&addr.u_sa, &mask.u_sa, rtm->rtm_flags, wid_dst);
    p_sockaddr(&gw.u_sa, NULL, nn::socket::RtFlag::Rtf_Host, wid_gw);

    snprintf(buffer, sizeof(buffer), "%%-%d.%ds ", wid_flags, wid_flags);
    p_flags(rtm->rtm_flags, buffer);
    if (Wflag) {
        printf("%*u ", wid_pksent, rtm->rtm_rmx.rmx_pksent);

        if (rtm->rtm_rmx.rmx_mtu != 0)
            printf("%*u ", wid_mtu, rtm->rtm_rmx.rmx_mtu);
        else
            printf("%*s ", wid_mtu, "");
    }

    memset(prettyname, 0, sizeof(prettyname));
    if (rtm->rtm_index < 16 && ifmap_size != 0) {
        strlcpy(prettyname, ifmap[rtm->rtm_index],
            sizeof(prettyname));
        if (*prettyname == '\0')
            strlcpy(prettyname, "---", sizeof(prettyname));
    }
    else
    {
        snprintf(prettyname,sizeof(prettyname),"%d", rtm->rtm_index);
    }

    printf("%*.*s", wid_if, wid_if, prettyname);
    if (rtm->rtm_rmx.rmx_expire) {
        time_t expire_time;
        int upTimeSeconds = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetSeconds();
        if ((expire_time =
            rtm->rtm_rmx.rmx_expire - upTimeSeconds) > 0)
            printf(" %*d", wid_expire, (int)expire_time);
    }

    putchar('\n');
}

static void
p_sockaddr(nn::socket::SockAddr *sa, nn::socket::SockAddr *mask, nn::socket::RtFlag flags, int width)
{
    const char *cp;
    cp = fmt_sockaddr(sa, mask, flags);

    if (width < 0 )
        printf("%s ", cp);
    else {
        if (numeric_addr)
            printf("%-*s ", width, cp);
        else
            printf("%-*.*s ", width, width, cp);
    }
}

static void
p_flags(nn::socket::RtFlag f, const char *format)
{
    printf(format, fmt_flags(static_cast<int>(f)));
}

static const char *
fmt_flags(int f)
{
    static char name[33];
    char *flags;
    struct bits *p = bits;

    for (flags = name; p->b_mask; p++)
        if (p->b_mask & f)
            *flags++ = p->b_val;
    *flags = '\0';
    return (name);
}

static const char *
fmt_sockaddr(nn::socket::SockAddr *sa, nn::socket::SockAddr *mask, nn::socket::RtFlag flags)
{

    static char workbuf[128];
    const char *cp;

    if (sa == NULL)
        return ("null");

    switch(sa->sa_family) {
    case nn::socket::Family::Af_Inet:
        {
            nn::socket::SockAddrIn *sockin = (nn::socket::SockAddrIn *)sa;

            if ((sockin->sin_addr.S_addr == nn::socket::InAddr_Any) &&
                mask &&
                nn::socket::InetNtohl(((nn::socket::SockAddrIn *)mask)->sin_addr.S_addr)
                ==0L)
                cp = "default" ;
            else if ((flags & nn::socket::RtFlag::Rtf_Host) != nn::socket::RtFlag::Rtf_None)
            {
                cp = routename(sockin->sin_addr.S_addr);
            }
            else if (mask)
            {
                cp = netname(sockin->sin_addr.S_addr,
                    ((nn::socket::SockAddrIn *)mask)->sin_addr.S_addr);
            }
            else
            {
                cp = netname(sockin->sin_addr.S_addr, nn::socket::InAddr_Any);
            }
            break;
        }

    case nn::socket::Family::Af_Link:
        {
            nn::socket::SockAddrDl *sdl = (nn::socket::SockAddrDl *)sa;

            if (sdl->sdl_nlen == 0 && sdl->sdl_alen == 0 &&
                sdl->sdl_slen == 0) {
                    (void) sprintf(workbuf, "link#%d", sdl->sdl_index);
                    cp = workbuf;
            } else
                switch (sdl->sdl_type) {

                case IFT_ETHER:
                case IFT_L2VLAN:
                case IFT_BRIDGE:
                    if (sdl->sdl_alen == nn::socket::Ether_Addr_Len) {
                        cp = ether_ntoa((nn::socket::EtherAddr *)
                            (sdl->sdl_data + sdl->sdl_nlen));
                        break;
                    }
                    /* FALLTHROUGH */
                default:
                    cp = link_ntoa(sdl);
                    break;
            }
            break;
        }
    default:
        {
            u_char *s = (u_char *)sa->sa_data, *slim;
            char *cq, *cqlim;

            cq = workbuf;
            slim =  sa->sa_len + (u_char *) sa;
            cqlim = cq + sizeof(workbuf) - 6;
            cq += sprintf(cq, "(%d)", static_cast<int>(sa->sa_family));
            while (s < slim && cq < cqlim) {
                cq += sprintf(cq, " %02x", *s++);
                if (s < slim)
                    cq += sprintf(cq, "%02x", *s++);
            }
            cp = workbuf;
        }
    }

    return (cp);
}

char *
routename(nn::socket::InAddrT in)
{
    char *cp;
    static char line[MAXHOSTNAMELEN];

    cp = 0;
    /*if (!numeric_addr) {
    hp = nn::socket::GetHostByAddr(&in, sizeof (nn::socket::InAddr), nn::socket::Family::Af_Inet);
    if (hp) {
    cp = hp->h_name;
    trimdomain(cp, strlen(cp));
    }
    }*/
    if (cp) {
        strlcpy(line, cp, sizeof(line));
    } else {
#define C(x)    ((x) & 0xff)
        in = nn::socket::InetNtohl(in);
        sprintf(line, "%u.%u.%u.%u",
            C(in >> 24), C(in >> 16), C(in >> 8), C(in));
    }
    return (line);
}

/*
* Return the name of the network whose address is given.
*/
char *
netname(nn::socket::InAddrT in, nn::socket::InAddrT mask)
{
    char *cp = 0;
    static char line[MAXHOSTNAMELEN];
    nn::socket::InAddrT i;

    /* It is ok to supply host address. */
    in &= mask;

    i = nn::socket::InetNtohl(in);
    /*if (!numeric_addr && i) {
    np = getnetbyaddr(i >> NSHIFT(nn::socket::InetNtohl(mask)), nn::socket::Family::Af_Inet);
    if (np != NULL) {
    cp = np->n_name;
    trimdomain(cp, strlen(cp));
    }
    }*/

    if (cp != NULL) {
        strlcpy(line, cp, sizeof(line));
    } else {
        nn::socket::InetNtop(nn::socket::Family::Af_Inet, &in, line, sizeof(line) - 1);
    }
    domask(line + strlen(line), i, nn::socket::InetNtohl(mask));
    return (line);
}

static void
domask(char *dst, nn::socket::InAddrT addr, u_long mask)
{
    NN_UNUSED(addr);

    int b, i;

    if (mask == 0 /*|| (!numeric_addr && NSHIFT(mask) != 0)*/) {
        *dst = '\0';
        return;
    }
    i = 0;
    for (b = 0; b < 32; b++)
        if (mask & (1 << b)) {
            int bb;

            i = b;
            for (bb = b + 1; bb < 32; bb++)
                if (!(mask & (1 << bb))) {
                    i = -1; /* noncontig */
                    break;
                }
                break;
        }
        if (i == -1)
            sprintf(dst, "&0x%lx", mask);
        else
            sprintf(dst, "/%d", 32 - i);
}

static const char hexlist[] = "0123456789abcdef";

char *
link_ntoa(const nn::socket::SockAddrDl *sdl)
{
    static char obuf[64];
    char *out = obuf;
    int i;
    u_char *in = (u_char *)nn::socket::LlAddr(sdl);
    u_char *inlim = in + sdl->sdl_alen;
    int firsttime = 1;

    if (sdl->sdl_nlen) {
        bcopy(sdl->sdl_data, obuf, sdl->sdl_nlen);
        out += sdl->sdl_nlen;
        if (sdl->sdl_alen)
            *out++ = ':';
    }
    while (in < inlim) {
        if (firsttime)
            firsttime = 0;
        else
            *out++ = '.';
        i = *in++;
        if (i > 0xf) {
            out[1] = hexlist[i & 0xf];
            i >>= 4;
            out[0] = hexlist[i];
            out += 2;
        } else
            *out++ = hexlist[i];
    }
    *out = 0;
    return (obuf);
}

char *
    ether_ntoa_r (const nn::socket::EtherAddr *addr, char * buf)
{
    snprintf(buf, 18, "%02x:%02x:%02x:%02x:%02x:%02x",
        addr->octet[0], addr->octet[1],
        addr->octet[2], addr->octet[3],
        addr->octet[4], addr->octet[5]);
    return buf;
}

/*
* Convert Ethernet address to standard hex-digits-and-colons printable form.
*/
char *
    ether_ntoa (const nn::socket::EtherAddr *addr)
{
    static char buf[18];
    return ether_ntoa_r(addr, buf);
}

}}}
#undef MAXHOSTNAMELEN
