﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_KMemoryManager.h"
#include "../kern_KTransferMemory.h"
#include "../kern_KProcess.h"
#include "../kern_KHandleTable.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"
#include "../kern_KScopedResourceLimitTester.h"

namespace nn { namespace kern { namespace svc {
namespace {
Result SvcCreateTransferMemory(nn::svc::Handle* pOut, uintptr_t addr, size_t size, nn::svc::MemoryPermission myProtect)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    switch (myProtect)
    {
    case nn::svc::MemoryPermission_None:
    case nn::svc::MemoryPermission_Read:
    case nn::svc::MemoryPermission_ReadWrite:
        break;
    default:
        return nn::svc::ResultInvalidNewMemoryPermission();
    }

    Result result;
    KProcess& process = GetCurrentProcess();

    KScopedResourceLimitTester transferMemoryTester(&process, nn::svc::LimitableResource_TransferMemoryCountMax);
    if (transferMemoryTester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    KTransferMemory* pTransferMemory = KTransferMemory::Create();
    if (!pTransferMemory)
    {
        return nn::svc::ResultOutOfResource();
    }

    KScopedAutoObject<KTransferMemory> autoCloser(pTransferMemory);

    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    result = pTransferMemory->Initialize(addr, size, myProtect);
    if (result.IsFailure())
    {
        return result;
    }

    transferMemoryTester.Accepted();

    result = KTransferMemory::Register(pTransferMemory);
    if (result.IsSuccess())
    {
        result = GetCurrentProcess().GetHandleTable().Add(pOut, pTransferMemory);
    }

    return result;
}

Result SvcMapTransferMemory(nn::svc::Handle handle, uintptr_t addr, size_t size, nn::svc::MemoryPermission ownerPermission)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    switch (ownerPermission)
    {
    case nn::svc::MemoryPermission_None:
    case nn::svc::MemoryPermission_Read:
    case nn::svc::MemoryPermission_ReadWrite:
        break;
    default:
        return nn::svc::ResultInvalidState();
    }

    KTransferMemory* pTransferMemory = GetCurrentProcess().GetHandleTable().GetObject<KTransferMemory>(handle);

    if (!pTransferMemory)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KTransferMemory> autoCloser(pTransferMemory);

    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInRange(addr, size, KMemoryState_Transfered))
    {
        return nn::svc::ResultInvalidRegion();
    }

    return pTransferMemory->Map(addr, size, ownerPermission);
}

Result SvcUnmapTransferMemory(nn::svc::Handle handle, uintptr_t addr, size_t size)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KTransferMemory* pTransferMemory = GetCurrentProcess().GetHandleTable().GetObject<KTransferMemory>(handle);

    if (!pTransferMemory)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KTransferMemory> autoCloser(pTransferMemory);

    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInRange(addr, size, KMemoryState_Transfered))
    {
        return nn::svc::ResultInvalidRegion();
    }

    return pTransferMemory->Unmap(addr, size);
}
}
#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcCreateTransferMemory32(nn::svc::Handle* pOut, uintptr_t addr, size_t size, nn::svc::MemoryPermission myProtect)
{
    Result result = SvcCreateTransferMemory(pOut, addr, size, myProtect);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapTransferMemory32(nn::svc::Handle handle, uintptr_t addr, size_t size, nn::svc::MemoryPermission ownerPermission)
{
    Result result = SvcMapTransferMemory(handle, addr, size, ownerPermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapTransferMemory32(nn::svc::Handle handle, uintptr_t addr, size_t size)
{
    Result result = SvcUnmapTransferMemory(handle, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcCreateTransferMemory64(nn::svc::Handle* pOut, uintptr_t addr, size_t size, nn::svc::MemoryPermission myProtect)
{
    Result result = SvcCreateTransferMemory(pOut, addr, size, myProtect);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapTransferMemory64(nn::svc::Handle handle, uintptr_t addr, size_t size, nn::svc::MemoryPermission ownerPermission)
{
    Result result = SvcMapTransferMemory(handle, addr, size, ownerPermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapTransferMemory64(nn::svc::Handle handle, uintptr_t addr, size_t size)
{
    Result result = SvcUnmapTransferMemory(handle, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcCreateTransferMemory64From32(nn::svc::Handle* pOut, uintptr_t addr, size_t size, nn::svc::MemoryPermission myProtect)
{
    Result result = SvcCreateTransferMemory(pOut, addr, size, myProtect);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapTransferMemory64From32(nn::svc::Handle handle, uintptr_t addr, size_t size, nn::svc::MemoryPermission ownerPermission)
{
    Result result = SvcMapTransferMemory(handle, addr, size, ownerPermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapTransferMemory64From32(nn::svc::Handle handle, uintptr_t addr, size_t size)
{
    Result result = SvcUnmapTransferMemory(handle, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
