﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../kern_Platform.h"
#include "../kern_Kernel.h"
#include "../kern_Utility.h"
#include "../kern_KResourceLimit.h"
#include "../kern_KProcess.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"

namespace nn { namespace kern { namespace svc {
namespace {
Result SvcCreateResourceLimit(nn::svc::Handle* pOut)
{
    KResourceLimit* pResourceLimit = KResourceLimit::Create();
    if (!pResourceLimit)
    {
        return nn::svc::ResultOutOfResource();
    }

    KScopedAutoObject<KResourceLimit> autoCloser(pResourceLimit);

    pResourceLimit->Initialize();

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    Result result = KResourceLimit::Register(pResourceLimit);

    if (result.IsSuccess())
    {
        result = handleTable.Add(pOut, pResourceLimit);
    }

    return result;
}

Result SvcSetResourceLimitLimitValue(nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource, int64_t value)
{
    if (resource >= nn::svc::LimitableResource_Num)
    {
        return nn::svc::ResultInvalidEnum();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KResourceLimit* pResourceLimit = handleTable.GetObject<KResourceLimit>(resourceLimit);
    if (!pResourceLimit)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KResourceLimit> autoCloser(pResourceLimit);

    return pResourceLimit->SetLimitValue(resource, value);
}

Result SvcGetResourceLimitLimitValue(int64_t* pValue, nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource)
{
    if (resource >= nn::svc::LimitableResource_Num)
    {
        return nn::svc::ResultInvalidEnum();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KResourceLimit* pResourceLimit = handleTable.GetObject<KResourceLimit>(resourceLimit);
    if (!pResourceLimit)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KResourceLimit> autoCloser(pResourceLimit);

    *pValue = pResourceLimit->GetLimitValue(resource);

    return ResultSuccess();
}

Result SvcGetResourceLimitCurrentValue(int64_t* pValue, nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource)
{
    if (resource >= nn::svc::LimitableResource_Num)
    {
        return nn::svc::ResultInvalidEnum();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KResourceLimit* pResourceLimit = handleTable.GetObject<KResourceLimit>(resourceLimit);
    if (!pResourceLimit)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KResourceLimit> autoCloser(pResourceLimit);

    *pValue = pResourceLimit->GetCurrentValue(resource);

    return ResultSuccess();
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcCreateResourceLimit32(nn::svc::Handle* pOut)
{
    Result result = SvcCreateResourceLimit(pOut);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcSetResourceLimitLimitValue32(nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource, int64_t value)
{
    Result result = SvcSetResourceLimitLimitValue(resourceLimit, resource, value);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcGetResourceLimitLimitValue32(int64_t* pValue, nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource)
{
    Result result = SvcGetResourceLimitLimitValue(pValue, resourceLimit, resource);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcGetResourceLimitCurrentValue32(int64_t* pValue, nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource)
{
    Result result = SvcGetResourceLimitCurrentValue(pValue, resourceLimit, resource);
    ClearSvcOutRegistersReturnResult();
    return result;
}

#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcCreateResourceLimit64(nn::svc::Handle* pOut)
{
    Result result = SvcCreateResourceLimit(pOut);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcSetResourceLimitLimitValue64(nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource, int64_t value)
{
    Result result = SvcSetResourceLimitLimitValue(resourceLimit, resource, value);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcGetResourceLimitLimitValue64(int64_t* pValue, nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource)
{
    Result result = SvcGetResourceLimitLimitValue(pValue, resourceLimit, resource);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcGetResourceLimitCurrentValue64(int64_t* pValue, nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource)
{
    Result result = SvcGetResourceLimitCurrentValue(pValue, resourceLimit, resource);
    ClearSvcOutRegistersReturnResult();
    return result;
}

#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcCreateResourceLimit64From32(nn::svc::Handle* pOut)
{
    Result result = SvcCreateResourceLimit(pOut);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcSetResourceLimitLimitValue64From32(nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource, int64_t value)
{
    Result result = SvcSetResourceLimitLimitValue(resourceLimit, resource, value);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcGetResourceLimitLimitValue64From32(int64_t* pValue, nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource)
{
    Result result = SvcGetResourceLimitLimitValue(pValue, resourceLimit, resource);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcGetResourceLimitCurrentValue64From32(int64_t* pValue, nn::svc::Handle resourceLimit, nn::svc::LimitableResource resource)
{
    Result result = SvcGetResourceLimitCurrentValue(pValue, resourceLimit, resource);
    ClearSvcOutRegistersReturnResult();
    return result;
}

#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
