﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_KProcess.h"
#include "../kern_KTaggedAddress.h"
#include "../kern_KPageTableBase.h"
#include "../kern_Utility.h"
#include "../kern_CPUSelect.h"
#include "kern_MemoryMap.h"
#include "../kern_KCodeMemory.h"
#include "../kern_KUnsafeMemory.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"
#include <cstring>

namespace nn { namespace kern { namespace svc {

namespace {
Result SvcSetHeapSize(uintptr_t* pOut, size_t size)
{
    if ((size % HEAP_SIZE_ALIGN) != 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (size >= NN_KERN_P_ADDR_MAIN_MEMORY_SIZE)
    {
        return nn::svc::ResultInvalidSize();
    }

    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();

    KProcessAddress addr;

    Result result = pageTable.SetHeapSize(&addr, size);
    *pOut = GetAsInteger(addr);
    return result;
}

Result SvcSetMemoryPermission(uintptr_t addr, size_t size, nn::svc::MemoryPermission permission)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    switch (permission)
    {
    case nn::svc::MemoryPermission_None:
    case nn::svc::MemoryPermission_Read:
    case nn::svc::MemoryPermission_ReadWrite:
        break;
    default:
        return nn::svc::ResultInvalidNewMemoryPermission();
    }

    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    return pageTable.SetMemoryPermission(addr, size, permission);
}

Result SvcSetProcessMemoryPermission(nn::svc::Handle process, uint64_t addr, uint64_t size, nn::svc::MemoryPermission permission)
{
    if (addr != static_cast<uintptr_t>(addr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (size != static_cast<size_t>(size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    switch (permission)
    {
    case nn::svc::MemoryPermission_None:
    case nn::svc::MemoryPermission_Read:
    case nn::svc::MemoryPermission_ReadWrite:
    case nn::svc::MemoryPermission_ReadExecute:
        break;
    default:
        return nn::svc::ResultInvalidNewMemoryPermission();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, process);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);
    KProcessPageTable& pageTable = pProcess->GetPageTable();

    if (!pageTable.IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    return pageTable.SetProcessMemoryPermission(addr, size, permission);
}

Result SvcMapMemory(uintptr_t toAddr, uintptr_t fromAddr, size_t size)
{
    if ((toAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((fromAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(toAddr < toAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(fromAddr < fromAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!pageTable.IsInRange(toAddr, size, KMemoryState_Stack))
    {
        return nn::svc::ResultInvalidRegion();
    }

    return pageTable.MapMemory(toAddr, fromAddr, size);
}

Result SvcUnmapMemory(uintptr_t toAddr, uintptr_t fromAddr, size_t size)
{
    if ((toAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((fromAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(toAddr < toAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(fromAddr < fromAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!pageTable.IsInRange(toAddr, size, KMemoryState_Stack))
    {
        return nn::svc::ResultInvalidRegion();
    }

    return pageTable.UnmapMemory(toAddr, fromAddr, size);
}

Result SvcQueryProcessMemory(nn::svc::MemoryInfo* pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle hProcess, uint64_t addr)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, hProcess);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }
    if (addr != static_cast<uintptr_t>(addr))
    {
        pBlockInfo->baseAddress = 0x100000000ull;
        pBlockInfo->size        = 0ull - pBlockInfo->baseAddress;
        pBlockInfo->state       = nn::svc::MemoryState_Inaccessible;
        pBlockInfo->permission  = nn::svc::MemoryPermission_None;
        pBlockInfo->attribute   = static_cast<nn::svc::MemoryAttribute>(0);
        pBlockInfo->ipcCount    = 0;
        pBlockInfo->deviceCount = 0;
        pPageInfo->flags = 0;
        return ResultSuccess();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);

    KMemoryInfo mi = {};
    Result result = pProcess->GetPageTable().QueryInfo(&mi, pPageInfo, addr);
    if (result.IsFailure())
    {
        return result;
    }

    *pBlockInfo = mi.ConvertToSvcMemoryInfo();
    return ResultSuccess();
}

Result SvcQueryMemory(nn::svc::MemoryInfo* pBlockInfo, nn::svc::PageInfo* pPageInfo, uintptr_t addr)
{
    return SvcQueryProcessMemory(pBlockInfo, pPageInfo, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS, addr);
}

Result SvcQueryPhysicalAddress(nn::svc::PhysicalMemoryInfo* pBlockInfo, uintptr_t addr)
{
    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInRange(addr, 1))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    return pageTable.QueryPhysicalAddress(pBlockInfo, addr);
}

Result SvcQueryIoMapping(uintptr_t* pOut, uint64_t physAddr, size_t size)
{
    if (size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }

    if (!(physAddr < physAddr + size))
    {
        return nn::svc::ResultNotFound();
    }

    KProcessAddress addr;
    Result result = GetCurrentProcess().GetPageTable().QueryIoMapping(&addr, KPhysicalAddress(physAddr), size);
    *pOut = GetAsInteger(addr);
    return result;
}

Result SvcSetMemoryAttribute(uintptr_t addr, size_t size, Bit32 mask, Bit32 attribute)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    const Bit32 supported = nn::svc::MemoryAttribute_Uncached;
    if ((mask | attribute) != mask)
    {
        return nn::svc::ResultInvalidCombination();
    }
    if ((mask | attribute | supported) != supported)
    {
        return nn::svc::ResultInvalidCombination();
    }

    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    return pageTable.SetMemoryAttribute(addr, size, mask, attribute);
}

Result SvcMapPhysicalMemory(uintptr_t addr, size_t size)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidRegion();
    }

    KProcess& process = GetCurrentProcess();

    if (process.GetExtraResourceSize() == 0)
    {
        return nn::svc::ResultInvalidState();
    }

    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInReservedRegion(addr, size))
    {
        return nn::svc::ResultInvalidRegion();
    }
    return pageTable.MapPhysicalMemory(addr, size);
}

Result SvcUnmapPhysicalMemory(uintptr_t addr, size_t size)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidRegion();
    }

    KProcess& process = GetCurrentProcess();

    if (process.GetExtraResourceSize() == 0)
    {
        return nn::svc::ResultInvalidState();
    }

    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInReservedRegion(addr, size))
    {
        return nn::svc::ResultInvalidRegion();
    }
    return pageTable.UnmapPhysicalMemory(addr, size);
}


Result SvcCreateCodeMemory(nn::svc::Handle* pOut, uintptr_t addr, size_t size)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KCodeMemory* pCodeMemory = KCodeMemory::Create();
    if (pCodeMemory == nullptr)
    {
        return nn::svc::ResultOutOfResource();
    }
    KScopedAutoObject<KCodeMemory> autoCloser(pCodeMemory);

    KProcess& process = GetCurrentProcess();
    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    Result result = pCodeMemory->Initialize(addr, size);
    if (result.IsFailure())
    {
        return result;
    }

    result = KCodeMemory::Register(pCodeMemory);
    if (result.IsSuccess())
    {
        result = GetCurrentProcess().GetHandleTable().Add(pOut, pCodeMemory);
    }

    return result;
}

Result SvcControlCodeMemory(nn::svc::Handle handle,
        nn::svc::CodeMemoryOperation operation,
        uint64_t addr, uint64_t size, nn::svc::MemoryPermission permission)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KCodeMemory* pCodeMemory = GetCurrentProcess().GetHandleTable().GetObject<KCodeMemory>(handle);
    if (!pCodeMemory)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KCodeMemory> autoCloser(pCodeMemory);

    if (GetCurrentProcessPointer() == pCodeMemory->GetOwner())
    {
        return nn::svc::ResultInvalidHandle();
    }

    Result result;

    switch (operation)
    {
    case nn::svc::CodeMemoryOperation_Map:
        {
            if (!GetCurrentProcess().GetPageTable().IsInRange(addr, size, KMemoryState_CodeOut))
            {
                return nn::svc::ResultInvalidRegion();
            }
            switch (permission)
            {
            case nn::svc::MemoryPermission_ReadWrite:
                break;
            default:
                return nn::svc::ResultInvalidNewMemoryPermission();
            }
            result = pCodeMemory->Map(addr, size);
        }
        break;
    case nn::svc::CodeMemoryOperation_Unmap:
        {
            if (!GetCurrentProcess().GetPageTable().IsInRange(addr, size, KMemoryState_CodeOut))
            {
                return nn::svc::ResultInvalidRegion();
            }
            switch (permission)
            {
            case nn::svc::MemoryPermission_None:
                break;
            default:
                return nn::svc::ResultInvalidNewMemoryPermission();
            }
            result = pCodeMemory->Unmap(addr, size);
        }
        break;
    case nn::svc::CodeMemoryOperation_MapToOwner:
        {
            if (!pCodeMemory->GetOwner()->GetPageTable().IsInRange(addr, size, KMemoryState_GeneratedCode))
            {
                return nn::svc::ResultInvalidRegion();
            }
            switch (permission)
            {
            case nn::svc::MemoryPermission_Read:
            case nn::svc::MemoryPermission_ReadExecute:
                break;
            default:
                return nn::svc::ResultInvalidNewMemoryPermission();
            }
            result = pCodeMemory->MapToOwner(addr, size, permission);
        }
        break;
    case nn::svc::CodeMemoryOperation_UnmapFromOwner:
        {
            if (!pCodeMemory->GetOwner()->GetPageTable().IsInRange(addr, size, KMemoryState_GeneratedCode))
            {
                return nn::svc::ResultInvalidRegion();
            }
            switch (permission)
            {
            case nn::svc::MemoryPermission_None:
                break;
            default:
                return nn::svc::ResultInvalidNewMemoryPermission();
            }
            result = pCodeMemory->UnmapFromOwner(addr, size);
        }
        break;
    default:
        return nn::svc::ResultInvalidEnum();
    }

    return result;
}

Result SvcSetUnsafeLimit(size_t limitSize)
{
    if (limitSize % NN_KERN_FINEST_PAGE_SIZE != 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (limitSize > Kernel::GetKernelHeapManager().GetSize(KMemoryManager::Region_Unsafe))
    {
        return nn::svc::ResultOutOfRange();
    }
    return Kernel::GetUnsafeMemory().SetLimitSize(limitSize);
}

Result SvcMapPhysicalMemoryUnsafe(uintptr_t addr, size_t size)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcess& process = GetCurrentProcess();

    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInAslrRegion(addr, size))
    {
        return nn::svc::ResultInvalidRegion();
    }
    return pageTable.MapPhysicalMemoryUnsafe(addr, size);
}

Result SvcUnmapPhysicalMemoryUnsafe(uintptr_t addr, size_t size)
{
    if ((addr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcess& process = GetCurrentProcess();

    KProcessPageTable& pageTable = process.GetPageTable();

    if (!pageTable.IsInAslrRegion(addr, size))
    {
        return nn::svc::ResultInvalidRegion();
    }
    return pageTable.UnmapPhysicalMemoryUnsafe(addr, size);
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcSetHeapSize32( uintptr_t* pOut, size_t size)
{
    Result result = SvcSetHeapSize(pOut, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetMemoryPermission32(uintptr_t addr, size_t size, nn::svc::MemoryPermission permission)
{
    Result result = SvcSetMemoryPermission(addr, size, permission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetProcessMemoryPermission32(nn::svc::Handle process, uint64_t addr, uint64_t size, nn::svc::MemoryPermission permission)
{
    Result result = SvcSetProcessMemoryPermission(process, addr, size, permission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapMemory32(uintptr_t toAddr, uintptr_t fromAddr, size_t size)
{
    Result result = SvcMapMemory(toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapMemory32(uintptr_t toAddr, uintptr_t fromAddr, size_t size)
{
    Result result = SvcUnmapMemory(toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryMemory32( KUserPointer<nn::svc::ilp32::MemoryInfo*> pBlockInfo, nn::svc::PageInfo* pPageInfo, uintptr_t addr )
{
    nn::svc::MemoryInfo blockInfo = {0};

    Result result = SvcQueryMemory(&blockInfo, pPageInfo, addr);
    if (result.IsSuccess())
    {
        result = pBlockInfo.CopyFrom(&blockInfo);
    }

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryProcessMemory32( KUserPointer<nn::svc::ilp32::MemoryInfo*> pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle hProcess, uint64_t addr )
{
    nn::svc::MemoryInfo blockInfo = {0};

    Result result = SvcQueryProcessMemory(&blockInfo, pPageInfo, hProcess, addr);
    if (result.IsSuccess())
    {
        result = pBlockInfo.CopyFrom(&blockInfo);
    }

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryPhysicalAddress32(nn::svc::ilp32::PhysicalMemoryInfo* pBlockInfo, uintptr_t addr)
{
    Result result = SvcQueryPhysicalAddress(pBlockInfo, addr);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryIoMapping32(uintptr_t* pOut, uint64_t physAddr, size_t size)
{
    Result result = SvcQueryIoMapping(pOut, physAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetMemoryAttribute32(uintptr_t addr, size_t size, Bit32 mask, Bit32 attribute)
{
    Result result = SvcSetMemoryAttribute(addr, size, mask, attribute);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapPhysicalMemory32(uintptr_t addr, size_t size)
{
    Result result = SvcMapPhysicalMemory(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapPhysicalMemory32(uintptr_t addr, size_t size)
{
    Result result = SvcUnmapPhysicalMemory(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCreateCodeMemory32(nn::svc::Handle* pOut, uintptr_t addr, size_t size)
{
    Result result = SvcCreateCodeMemory(pOut, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcControlCodeMemory32(nn::svc::Handle handle,
        nn::svc::CodeMemoryOperation operation,
        uint64_t address, uint64_t size, nn::svc::MemoryPermission permission)
{
    Result result = SvcControlCodeMemory(handle, operation, address, size, permission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetUnsafeLimit32(size_t limitSize)
{
    Result result = SvcSetUnsafeLimit(limitSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapPhysicalMemoryUnsafe32(uintptr_t addr, size_t size)
{
    Result result = SvcMapPhysicalMemoryUnsafe(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapPhysicalMemoryUnsafe32(uintptr_t addr, size_t size)
{
    Result result = SvcUnmapPhysicalMemoryUnsafe(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcSetHeapSize64( uintptr_t* pOut, size_t size)
{
    Result result = SvcSetHeapSize(pOut, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetMemoryPermission64(uintptr_t addr, size_t size, nn::svc::MemoryPermission permission)
{
    Result result = SvcSetMemoryPermission(addr, size, permission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetProcessMemoryPermission64(nn::svc::Handle process, uint64_t addr, uint64_t size, nn::svc::MemoryPermission permission)
{
    Result result = SvcSetProcessMemoryPermission(process, addr, size, permission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapMemory64(uintptr_t toAddr, uint64_t fromAddr, size_t size)
{
    Result result = SvcMapMemory(toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapMemory64(uintptr_t toAddr, uint64_t fromAddr, size_t size)
{
    Result result = SvcUnmapMemory(toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryMemory64( KUserPointer<nn::svc::lp64::MemoryInfo*> pBlockInfo, nn::svc::PageInfo* pPageInfo, uintptr_t addr )
{
    nn::svc::MemoryInfo blockInfo = {};

    Result result = SvcQueryMemory(&blockInfo, pPageInfo, addr);
    if (result.IsSuccess())
    {
        result = pBlockInfo.CopyFrom(&blockInfo);
    }

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryProcessMemory64( KUserPointer<nn::svc::lp64::MemoryInfo*> pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle hProcess, uint64_t addr )
{
    nn::svc::MemoryInfo blockInfo = {};

    Result result = SvcQueryProcessMemory(&blockInfo, pPageInfo, hProcess, addr);
    if (result.IsSuccess())
    {
        result = pBlockInfo.CopyFrom(&blockInfo);
    }

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryPhysicalAddress64(nn::svc::lp64::PhysicalMemoryInfo* pBlockInfo, uintptr_t addr)
{
    Result result = SvcQueryPhysicalAddress(pBlockInfo, addr);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryIoMapping64(uintptr_t* pOut, uint64_t physAddr, size_t size)
{
    Result result = SvcQueryIoMapping(pOut, physAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetMemoryAttribute64(uintptr_t addr, size_t size, Bit32 mask, Bit32 attribute)
{
    Result result = SvcSetMemoryAttribute(addr, size, mask, attribute);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapPhysicalMemory64(uintptr_t addr, size_t size)
{
    Result result = SvcMapPhysicalMemory(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapPhysicalMemory64(uintptr_t addr, size_t size)
{
    Result result = SvcUnmapPhysicalMemory(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCreateCodeMemory64(nn::svc::Handle* pOut, uintptr_t addr, size_t size)
{
    Result result = SvcCreateCodeMemory(pOut, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcControlCodeMemory64(nn::svc::Handle handle,
        nn::svc::CodeMemoryOperation operation,
        uint64_t address, uint64_t size, nn::svc::MemoryPermission permission)
{
    Result result = SvcControlCodeMemory(handle, operation, address, size, permission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetUnsafeLimit64(size_t limitSize)
{
    Result result = SvcSetUnsafeLimit(limitSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapPhysicalMemoryUnsafe64(uintptr_t addr, size_t size)
{
    Result result = SvcMapPhysicalMemoryUnsafe(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapPhysicalMemoryUnsafe64(uintptr_t addr, size_t size)
{
    Result result = SvcUnmapPhysicalMemoryUnsafe(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcSetHeapSize64From32( uintptr_t* pOut, size_t size)
{
    Result result = SvcSetHeapSize(pOut, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetMemoryPermission64From32(uintptr_t addr, size_t size, nn::svc::MemoryPermission permission)
{
    Result result = SvcSetMemoryPermission(addr, size, permission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetProcessMemoryPermission64From32(nn::svc::Handle process, uint64_t addr, uint64_t size, nn::svc::MemoryPermission permission)
{
    Result result = SvcSetProcessMemoryPermission(process, addr, size, permission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapMemory64From32(uintptr_t toAddr, uint64_t fromAddr, size_t size)
{
    Result result = SvcMapMemory(toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapMemory64From32(uintptr_t toAddr, uint64_t fromAddr, size_t size)
{
    Result result = SvcUnmapMemory(toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryMemory64From32( KUserPointer<nn::svc::ilp32::MemoryInfo*> pBlockInfo, nn::svc::PageInfo* pPageInfo, uintptr_t addr )
{
    nn::svc::MemoryInfo blockInfo = {};

    Result result = SvcQueryMemory(&blockInfo, pPageInfo, addr);
    if (result.IsSuccess())
    {
        nn::svc::ilp32::MemoryInfo blockInfo32 = {};
        blockInfo32.baseAddress = blockInfo.baseAddress;
        blockInfo32.size        = blockInfo.size;
        blockInfo32.permission  = blockInfo.permission;
        blockInfo32.attribute   = blockInfo.attribute;
        blockInfo32.state       = blockInfo.state;
        blockInfo32.ipcCount    = blockInfo.ipcCount;
        blockInfo32.deviceCount = blockInfo.deviceCount;

        result = pBlockInfo.CopyFrom(&blockInfo32);
    }

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryProcessMemory64From32( KUserPointer<nn::svc::ilp32::MemoryInfo*> pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle hProcess, uint64_t addr )
{
    nn::svc::MemoryInfo blockInfo = {};

    Result result = SvcQueryProcessMemory(&blockInfo, pPageInfo, static_cast<nn::svc::Handle>(hProcess), addr);
    if (result.IsSuccess())
    {
        nn::svc::ilp32::MemoryInfo blockInfo32 = {};
        blockInfo32.baseAddress = blockInfo.baseAddress;
        blockInfo32.size        = blockInfo.size;
        blockInfo32.permission  = blockInfo.permission;
        blockInfo32.attribute   = blockInfo.attribute;
        blockInfo32.state       = blockInfo.state;
        blockInfo32.ipcCount    = blockInfo.ipcCount;
        blockInfo32.deviceCount = blockInfo.deviceCount;

        result = pBlockInfo.CopyFrom(&blockInfo32);
    }

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryPhysicalAddress64From32(nn::svc::ilp32::PhysicalMemoryInfo* pBlockInfo, uintptr_t addr)
{
    nn::svc::PhysicalMemoryInfo blockInfo = {};

    Result result = SvcQueryPhysicalAddress(&blockInfo, addr);

    pBlockInfo->physicalAddress = blockInfo.physicalAddress;
    pBlockInfo->virtualAddress = blockInfo.virtualAddress;
    pBlockInfo->size = blockInfo.size;

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryIoMapping64From32(uintptr_t* pOut, uint64_t physAddr, size_t size)
{
    Result result = SvcQueryIoMapping(pOut, physAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetMemoryAttribute64From32(uintptr_t addr, size_t size, Bit32 mask, Bit32 attribute)
{
    Result result = SvcSetMemoryAttribute(addr, size, mask, attribute);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapPhysicalMemory64From32(uintptr_t addr, size_t size)
{
    Result result = SvcMapPhysicalMemory(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapPhysicalMemory64From32(uintptr_t addr, size_t size)
{
    Result result = SvcUnmapPhysicalMemory(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcCreateCodeMemory64From32(nn::svc::Handle* pOut, uintptr_t addr, size_t size)
{
    Result result = SvcCreateCodeMemory(pOut, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcControlCodeMemory64From32(nn::svc::Handle handle,
        nn::svc::CodeMemoryOperation operation,
        uint64_t address, uint64_t size, nn::svc::MemoryPermission permission)
{
    Result result = SvcControlCodeMemory(handle, operation, address, size, permission);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetUnsafeLimit64From32(size_t limitSize)
{
    Result result = SvcSetUnsafeLimit(limitSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapPhysicalMemoryUnsafe64From32(uintptr_t addr, size_t size)
{
    Result result = SvcMapPhysicalMemoryUnsafe(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapPhysicalMemoryUnsafe64From32(uintptr_t addr, size_t size)
{
    Result result = SvcUnmapPhysicalMemoryUnsafe(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
