﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_Kernel.h"
#include "kern_WorkThread.h"
#include "kern_KEvent.h"
#include "kern_KThread.h"
#include "kern_KSynchronization.h"
#include "kern_KMemoryManager.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_KResourceLimit.h"
#include "kern_Utility.h"

namespace nn { namespace kern {

    void KWorkThreadManager::Initialize(Bit32 id, int32_t priority)
    {
        NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_ThreadCountMax, 1));
        m_Id        = id;
        m_pTaskHead = NULL;
        m_pTaskTail = NULL;
        m_IsActive  = false;

        m_pThread = KThread::Create();
        NN_KERN_ABORT_UNLESS(m_pThread);
        NN_KERN_ABORT_IF_FAILED(InitializeThreadForKernel(m_pThread, &WorkThread, reinterpret_cast<uintptr_t>(this), priority, KCPU::NUM_CORE - 1));
        KThread::Register(m_pThread);
        m_pThread->Run();
    }


    // ワーカースレッドのリストからひとつタスクを取得します。
    KWorkerTask* KWorkThreadManager::GetTask()
    {
        // リンクリストの最初の要素を取得します
        KWorkerTask* head = m_pTaskHead;
        if (head)
        {
            if (head == m_pTaskTail)
            {
                // headがリンクリストの末端だった場合
                m_pTaskHead = m_pTaskTail = 0;
            }
            else
            {
                // headがリンクリストの中間だった場合
                // リンクリストをひとつ辿ります
                m_pTaskHead = head->GetNextTask();
            }

            // nullを代入しタスク間のリンクリストを切ります。
            //
            // CHECK: リンクリストの末端に対し この処理は不要です。上のスコープに移動しても問題ありません。
            head->SetNextTask(0);
        }
        return head;
    }

    void KWorkThreadManager::WorkThread(uintptr_t param)
    {
        reinterpret_cast<KWorkThreadManager*>(param)->WorkThreadBody();
    }

    void KWorkThreadManager::WorkThreadBody()
    {
        NN_LOG("    WorkThread ID=%lld, Core=%d, State=%d, Priority=%d\n",
            m_pThread->GetId(),
            m_pThread->GetRunningProcessor(),
            m_pThread->GetState(),
            m_pThread->GetPriority() );

        for(;;)
        {
            KWorkerTask* pTask;

            {
                KScopedSchedulingLock locker;

                pTask = GetTask();

                if( pTask == NULL )
                {
                    // 処理するタスクがなくなったら寝る
                    m_IsActive = false;
                    m_pThread->SetState(KThread::STATE_WAIT);
                    continue;
                }

                m_IsActive = true;
            }

            pTask->DoTask();
        }
    }

    void KWorkThreadManager::AddTask(KWorkerTask* pTask)
    {
        KScopedSchedulingLock locker;

        // すでに登録済みのタスクを再登録していないか確認
        NN_KERN_ASSERT(pTask->GetNextTask() == 0);

        if( m_pTaskTail )
        {
            // すでに登録済みのタスクがある場合、リンクリストの最後につなぎます。
            m_pTaskTail->SetNextTask(pTask);
            m_pTaskTail = pTask;
        }
        else
        {
            // 最初に登録するタスクの場合
            m_pTaskTail = pTask;
            m_pTaskHead = pTask;

            // ワーカースレッドを実行状態にします。
            if( ! m_IsActive )
            {
                m_pThread->SetState(KThread::STATE_RUNNABLE);
            }
        }
    }

    void KWorkThreadManager::AddTask(Bit32 id, KWorkerTask* pTask)
    {
        NN_KERN_ASSERT(id < WORK_THREAD_COUNT);
        Kernel::GetWorkThreadManager(id).AddTask(pTask);
    }
}}

