﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KProcess.h"
#include "kern_KThread.h"
#include "kern_MemoryMap.h"
#include "kern_Trace.h"

namespace nn { namespace kern {

#if defined NN_KERN_ENABLE_KERNEL_TRACE

namespace {
    const int TraceUnitSize    = 64;

    const int TraceControlSize = 64;
    const int TraceMagic       = 0x01020304UL;
    const int TraceVersion     = 0x00010000UL;

    const int TransportControlSize        = 64;
    const uint64_t TransportControlHeader = 0x0102030405060708ULL;
    const uint32_t TransportVersion       = 0x00010000UL;

    // at least a MB for buffer
    const size_t MinSize = 1024 * 1024;
    const uint64_t UnknownId = ~0;

    void* g_pTraceMem                   = NULL;
    TraceControlHeader* g_pTraceControl = NULL;

    bool g_InitializedTrace = false;
    volatile bool g_Enabled = false;

    KSimpleLock g_LockCore[KCPU::NUM_CORE];

    static_assert(sizeof(TraceRecord) == TraceUnitSize, "size of TraceRecord must be the size of one unit (64 bytes)");

    inline bool TraceFilterRecord(uint32_t traceDataType)
    {
        uint32_t dataTypeGroup = (traceDataType & 0xFF00) >> 8;
        uint64_t dataTypeBit   = 1ull << (traceDataType & 0x00FF);

        if(g_pTraceControl->m_DataTypeMask[dataTypeGroup] & dataTypeBit)
        {
            return false;
        }

        return true;
    }

    uintptr_t TraceGetSpace(uint32_t* outSequence)
    {
        TraceTransportControl* pControl = reinterpret_cast<TraceTransportControl*>( g_pTraceMem );
        size_t total = TraceUnitSize;

        uint32_t start = pControl->m_Start;
        uint32_t end   = pControl->m_End;
        uint32_t size  = end - start;
        uint32_t latest;

        for (;;)
        {
            TraceSpace newSpace = pControl->m_Space;
            TraceSpace prevSpace = newSpace;
            latest = newSpace.m_Latest + total;

            if( latest >  size )
            {
                latest = total;
            }

            ++newSpace.m_Sequence;
            newSpace.m_Latest = latest;

            if( Interlocked::CompareAndSwapWeak(&pControl->m_Space, prevSpace, newSpace) == 0 )
            {
               *outSequence = newSpace.m_Sequence;
               return ( pControl->m_Start + latest ) - total;
            }
        }
    }
}

bool InitializeTransportLayer(void* pMemory, size_t length)
{
    if( length < MinSize || (length & (TraceUnitSize - 1)) || (reinterpret_cast<uintptr_t>(pMemory) & (TraceUnitSize - 1)) )
    {
            return false;
    }
    memset(pMemory, 0, length);   // zero pad bytes and trace record memory
    TraceTransportControl* pControl = reinterpret_cast<TraceTransportControl*>( pMemory );
    pControl->m_Header = TransportControlHeader;
    pControl->m_Version = TransportVersion;
    pControl->m_UnitSize = TraceUnitSize;
    pControl->m_Space.m_Sequence = 0;
    pControl->m_Space.m_Latest = 0;
    pControl->m_Start = reinterpret_cast<uintptr_t>( pControl ) + TransportControlSize;
    pControl->m_End = reinterpret_cast<uintptr_t>( pControl ) + length;
    return true;
}

void Trace(uint16_t traceDataType, uint64_t arg1, uint64_t arg2, uint64_t arg3, uint64_t arg4)
{
    if ( !g_Enabled || !g_InitializedTrace || TraceFilterRecord(traceDataType))
    {
        return;
    }

    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_LockCore[GetCurrentCpuNo()]);

    if ( !g_Enabled )
    {
        return;
    }

    uint32_t sequence;
    uintptr_t pDest = TraceGetSpace(&sequence);

    TraceRecord record;

    record.m_Header.m_Sequence = sequence;
    record.m_Header.m_Type = traceDataType;
    record.m_Header.m_Cid  = GetCurrentCpuNo();
    record.m_Header.m_Tid  = GetCurrentThread().GetId();
    record.m_Header.m_Timestamp = Kernel::GetHardwareTimer().GetTick();
    record.m_Header.m_Pid = UnknownId;
    if( GetCurrentProcessPointer() )
    {
        record.m_Header.m_Pid = GetCurrentProcessPointer()->GetId();
    }

    record.m_Data[0] = arg1;
    record.m_Data[1] = arg2;
    record.m_Data[2] = arg3;
    record.m_Data[3] = arg4;

    memcpy(reinterpret_cast<void*>(pDest), &record, TraceUnitSize);
}

void InitializeTrace()
{
    NN_KERN_ASSERT(sizeof(TraceControlHeader) < TraceControlSize);

    KVirtualAddress traceMemAddr = NN_KERN_V_TRACE_BUFFER;
    int traceSize = NN_KERN_V_TRACE_BUFFER_SIZE;
    NN_LOG("[Trace Events] Trace Buffer: %p size:%x\n", traceMemAddr, traceSize);

    g_pTraceControl = GetTypedPointer<TraceControlHeader>( traceMemAddr );

    g_pTraceControl->m_Magic = TraceMagic;
    g_pTraceControl->m_Version = TraceVersion;
    g_pTraceControl->m_Length = TraceControlSize;
    g_pTraceControl->m_IsKernel64bit = 0;
    g_pTraceControl->m_Flags = 0;
    if( sizeof(void*) == sizeof(uint64_t) )
    {
        g_pTraceControl->m_IsKernel64bit = 1;
    }

    std::memset(g_pTraceControl->m_DataTypeMask, 0xff, sizeof(g_pTraceControl->m_DataTypeMask));

    g_pTraceMem = GetTypedPointer<char>( traceMemAddr ) + TraceControlSize;
    bool rv = InitializeTransportLayer(g_pTraceMem, traceSize - TraceControlSize);

    NN_UNUSED(rv);
    NN_KERN_ASSERT(rv == true);
    g_InitializedTrace = true;

    KernelTraceStart();
}

void KernelTraceStart()
{
    g_Enabled = true;
}

void KernelTraceStop()
{
    g_Enabled = false;
    //make sure that all trace calls have finished
    for ( int i = 0; i < KCPU::NUM_CORE; ++i )
    {
        KDisableInterrupt di;
        KScopedSimpleLock lock(&g_LockCore[i]);
    }
}

#endif //NN_KERN_ENABLE_KERNEL_TRACE

}}
