﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_Version.h>
#include <nn/svc/svc_Kernel.h>

#include "kern_Platform.h"
#include "init/kern_InitFunctions.h"
#include "kern_CPUSelect.h"
#include "kern_InitialProcess.h"
#include "kern_KProcess.h"
#include "kern_KSynchronization.h"
#include "kern_KTaggedAddress.h"
#include "kern_KThread.h"
#include "kern_Kernel.h"
#include "kern_Utility.h"
#include "kern_DebugString.h"
#include "kern_SystemControl.h"
#include "kern_KPageTableManager.h"
#include "kern_KDeviceAddressSpace.h"
#include "kern_DebugSelect.h"
#include "kern_DpcManager.h"
#include "kern_KMemoryLayout.h"
#include "kern_KHeapArrange.h"
#include <cstring>

#include "kern_Main.h"

using namespace nn::kern;

/*!
    @brief     カーネルのエントリーポイント

    この関数は複数のコアで同時に実行されます。

*/
extern "C" void KernelMain(int32_t coreNo)
{
    // 1 コアローカル領域上のオブジェクトを構築
    Kernel::Initialize(coreNo);

    Kernel::SetState(Kernel::STATE_INITIALIZING);

    KThread& mainThread = Kernel::GetMainThread(coreNo);
    KThread& idleThread = Kernel::GetIdleThread(coreNo);
    IndicateDebugPoint(1);

    // コアごとに必要な処理
    {
        // 2 メインスレッドの構築
        void* svStackBottom     = GetUntypedPointer(KMemoryLayout::GetMainStackBottom(coreNo));
        void* svIdleStackBottom = GetUntypedPointer(KMemoryLayout::GetIdleStackBottom(coreNo));

        KAutoObject::Create(&mainThread);
        mainThread.Initialize(
            NULL,                                                   // f
            0,                                                      // param
            svStackBottom,                                          // svStackBottom
            0,                                                      // userStackBottom
            1,                                                      // prio
            coreNo,
            NULL,
            KThread::TYPE_MAIN );

        KAutoObject::Create(&idleThread);
        idleThread.Initialize(
            NULL,                                                   // f
            0,                                                      // param
            svIdleStackBottom,                                      // svStackBottom
            0,                                                      // userStackBottom
            KThread::IdleThreadPriority,                            // prio
            coreNo,
            NULL,
            KThread::TYPE_MAIN );

        // 3 カレントスレッドの設定
        SetCurrentThread(&mainThread);

        // 4 カレントプロセスの設定
        SetCurrentProcess(NULL);

        // 5 割り込み初期化
        Kernel::GetInterruptManager().Initialize(coreNo);

        // 6 ハードウェアタイマーの初期化
        Kernel::GetHardwareTimer().Initialize(coreNo);

        // 7 スケジューラ初期化
        GetCoreScheduler().Initialize(&idleThread);
    }

    if( coreNo == 0 )
    {
        KSystemControl::SystemInitialize0();
    }

    // すべてのコアの実行をいったん同期します
    KCPU::SynchronizeAllCore();
    IndicateDebugPoint(2);

#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
    KCPU::EnablePerformanceCounter();
#endif

    // コアごとでない処理
    if( coreNo == 0 )
    {
        KHeapArrange arrange;
        KSystemControl::GetHeapArrange(&arrange);
        Kernel::GetKernelHeapManager().Initialize(arrange);
    }
    if( coreNo == KCPU::NUM_CORE - 1 )
    {
        // 8-2 同期マネージャ初期化
        Kernel::GetSynchronization().Initialize();
    }

    // すべてのコアの実行をいったん同期します
    KCPU::SynchronizeAllCore();
    IndicateDebugPoint(3);

    size_t initialProcessSize = 0;
    // コアごとでない処理
    if( coreNo == 0 )
    {
        // 9-1 初期プロセスイメージ領域をメインメモリにコピー
        //   初期プロセスイメージが格納されていた領域の上に SlabHeap が作られるので
        //   SlabHeap の初期化より前にコピーしなければならない
        //   コピー先メモリは KernelHeap 上に確保されるので
        //   KernelHeap の初期化より後にコピーしなければならない
        initialProcessSize = CopyProcessImagesToMainMemory();

        NN_LOG("HorizonKernel (Ver.%d.%d SDK %d.%d.%d r%d)\n",
                NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR,
                NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO,
                NN_SDK_VERSION_RELSTEP);

        NN_LOG("KernelRegion    %p - %p\n", KMemoryLayout::GetKernelRegionBegin(), KMemoryLayout::GetKernelRegionEnd() - 1);
        NN_LOG("    Code        %p - %p\n", KMemoryLayout::GetCodeRegionBegin(), KMemoryLayout::GetCodeRegionEnd() - 1);
        NN_LOG("    Misc        %p - %p\n", KMemoryLayout::GetMiscRegionBegin(), KMemoryLayout::GetMiscRegionEnd() - 1);
        NN_LOG("    Stack       %p - %p\n", KMemoryLayout::GetStackRegionBegin(), KMemoryLayout::GetStackRegionEnd() - 1);
        NN_LOG("    Slab        %p - %p\n", KMemoryLayout::GetSlabRegionBegin(), KMemoryLayout::GetSlabRegionEnd() - 1);
        NN_LOG("LinearRegion    %p - %p\n", KMemoryLayout::GetLinearRegionBegin(), KMemoryLayout::GetLinearRegionEnd() - 1);
        NN_LOG("LinearRegionPhy %p - %p\n", KMemoryLayout::GetLinearRegionPhysicalBegin(), KMemoryLayout::GetLinearRegionPhysicalEnd() - 1);
        NN_LOG("    SlabPhys    %p - %p\n", KMemoryLayout::GetSlabRegionPhysicalBegin(), KMemoryLayout::GetSlabRegionPhysicalEnd() - 1);

        // 11-1 各スラブアロケータ初期化
        size_t slabSize;
        nn::kern::init::InitializeAllocators(&slabSize);
        slabSize = RoundUp(slabSize, NN_KERN_FINEST_PAGE_SIZE);
        KVirtualAddress ptHeap = KMemoryLayout::ToLinearVirtualAddress(KMemoryLayout::GetPtHeapRegionPhysicalBegin());
        size_t ptSize = KMemoryLayout::GetPtHeapRegionPhysicalSize();
        NN_UNUSED(ptSize);
        Kernel::InitializeResourceManager(ptHeap, KMemoryLayout::GetPtHeapRegionPhysicalSize());

#if defined(NN_KERN_V_ADDR_SLAB_HEAP)
        NN_KERN_RELEASE_LOG("SlabHeap           %p-%p  %6d KB\n",  KMemoryLayout::GetSlabRegionBegin(), KMemoryLayout::GetSlabRegionEnd() - 1, slabSize / 1024);
#endif
        NN_KERN_RELEASE_LOG("PT Heap            %p-%p  %6d KB\n",  ptHeap, ptHeap + ptSize - 1, ptSize / 1024);
#if defined(NN_KERN_V_ADDR_CODE_MAIN_EX)
        NN_KERN_RELEASE_LOG("EX                 %p-%p  %6d KB\n",  NN_KERN_V_ADDR_CODE_MAIN_EX,   NN_KERN_V_ADDR_CODE_MAIN_EX_END - 1, NN_KERN_V_ADDR_CODE_MAIN_EX_SIZE / 1024);
#endif
#if defined(NN_KERN_V_ADDR_CODE_MAIN_RO)
        NN_KERN_RELEASE_LOG("RO                 %p-%p  %6d KB\n",  NN_KERN_V_ADDR_CODE_MAIN_RO,   NN_KERN_V_ADDR_CODE_MAIN_RO_END - 1, NN_KERN_V_ADDR_CODE_MAIN_RO_SIZE / 1024);
#endif
#if defined(NN_KERN_V_ADDR_CODE_MAIN_RW)
        NN_KERN_RELEASE_LOG("RW                 %p-%p  %6d KB\n",  NN_KERN_V_ADDR_CODE_MAIN_RW,   NN_KERN_V_ADDR_CODE_MAIN_RW_END - 1, NN_KERN_V_ADDR_CODE_MAIN_RW_SIZE / 1024);
#endif
#if defined(NN_KERN_V_ADDR_CODE_MAIN_ZI)
        NN_KERN_RELEASE_LOG("ZI                 %p-%p  %6d KB\n",  NN_KERN_V_ADDR_CODE_MAIN_ZI,   NN_KERN_V_ADDR_CODE_MAIN_ZI_END - 1, NN_KERN_V_ADDR_CODE_MAIN_ZI_SIZE / 1024);
#endif
#if defined(NN_KERN_V_ADDR_CODE_MAIN)
        NN_KERN_RELEASE_LOG("CODE               %p-%p  %6d KB\n",  NN_KERN_V_ADDR_CODE_MAIN, NN_KERN_V_ADDR_CODE_MAIN_END - 1, NN_KERN_V_ADDR_CODE_MAIN_SIZE / 1024);
#endif
#if defined(NN_KERN_P_ADDR_KERNEL_REGION)
        NN_KERN_RELEASE_LOG("Kernel Region Phys %p-%p  %6d KB\n",  KMemoryLayout::GetKernelRegionPhysicalBegin(), KMemoryLayout::GetKernelRegionPhysicalEnd() - 1, KMemoryLayout::GetKernelRegionPhysicalSize() / 1024);
#endif

    }

    // 12 カーネルページテーブル再構築
    KPageTable::Initialize(coreNo);
    Kernel::GetKernelPageTable().Initialize(coreNo);
    IndicateDebugPoint(4);

    // すべてのコアの実行をいったん同期します
    KCPU::SynchronizeAllCore();

    // コアごとに必要な処理
    {
        // 13 CPU 固有の初期化
        KCPU::Initialize0(coreNo);
    }

    {
        // 14 メインスレッドをコンテナへ登録
        KCPU::SynchronizeAllCore();
        for (int32_t i = 0; i < KCPU::NUM_CORE; i++)
        {
            if (i == coreNo)
            {
                KThread::Register(&mainThread);
            }
            KCPU::SynchronizeAllCore();
        }

        KCPU::SynchronizeAllCore();
        for (int32_t i = 0; i < KCPU::NUM_CORE; i++)
        {
            if (i == coreNo)
            {
                KThread::Register(&idleThread);
            }
            KCPU::SynchronizeAllCore();
        }
    }

    {
        // 15 割り込み処理スレッドの構築
        Kernel::GetInterruptTaskManager().Initialize();

        // 16 割り込みの有効化
        KInterruptManager::EnableInterrupt();

        // 17 スケジューラ有効化
        GetCoreScheduler().Activate();

        KCPU::Initialize1(coreNo);

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        if (coreNo == KCPU::NUM_CORE - 1)
        {
            // 63 nn::svc::LowestThreadPriority
            DpcManager::Initialize(63);
        }
        else
#endif
        {
            // 59 = (16 + 12) + 31 nn::os::LowestThreadPriority
            DpcManager::Initialize(59);
        }
    }

    // すべてのコアの実行をいったん同期します
    KCPU::SynchronizeAllCore();
    IndicateDebugPoint(5);

    // コアごとでない処理
    if( coreNo == 0 )
    {
        // 18 ワークスレッドを作成
        nn::Bit32 id;

        // 優先度低のワークスレッド
        id = KWorkThreadManager::LOW_PRIO_WORK_THREAD;
        Kernel::GetWorkThreadManager(id).Initialize(id, 11);

        // プロセス間共有ページを初期化
        KSystemControl::SystemInitialize1();

        KDeviceAddressSpace::Initialize();

        // 現在のカーネルメモリ使用量 - 初期プロセスのコピー領域
        size_t cur = Kernel::Kernel::GetSystemResourceLimit().GetCurrentValue(nn::svc::LimitableResource_PhysicalMemoryMax);
        cur -= initialProcessSize;

        // 19 初期プロセスを構築＆開始
        LoadAndRunProcessesFromMainMemory();

        Kernel::SetState(Kernel::STATE_NORMAL);

        NN_KERN_RELEASE_LOG("Kernel Memory Total %6ld KB\n", cur / 1024);
        NN_KERN_RELEASE_LOG("    Kernel Region   %6ld KB\n", static_cast<size_t>(KMemoryLayout::GetKernelRegionPhysicalSize() / 1024));
        NN_KERN_RELEASE_LOG("    Reserved        %6ld KB\n", static_cast<size_t>((NN_KERN_P_ADDR_RESERVED_LO_SIZE + NN_KERN_P_ADDR_RESERVED_HI_SIZE) / 1024));
        NN_KERN_RELEASE_LOG("\n");
    }

    // すべてのコアの実行をいったん同期します
    KCPU::SynchronizeAllCore();
    IndicateDebugPoint(6);

    NN_LOG("    IdleThread ID=%lld, Core=%d, State=%d\n",
            idleThread.GetId(),
            idleThread.GetRunningProcessor(),
            idleThread.GetState());

#if defined NN_KERN_ENABLE_KTRACE_DUMP
    KTrace::BindStopInterruptHandler(coreNo);
    KCPU::SynchronizeAllCore();
#endif

    // 20 プロセスをロードし終わったので最低優先度に。
    NN_LOG("goto idle. [core%d] %d ms\n", coreNo,
        static_cast<uint32_t>((nn::svc::Tick((KHardwareTimer::GetTick())) * 1000) / nn::svc::Tick::GetTicksPerSecond()));
    GetCurrentThread().SetPriorityForIdleThread();

    IndicateDebugPoint(7);

    mainThread.Open();
    mainThread.Exit();
}   // NOLINT(readability/fn_size)

//extern "C" void __auto_semihosting(void) __attribute__ ((alias("KernelMain")));
#if defined NN_BUILD_CONFIG_COMPILER_GCC || defined NN_BUILD_CONFIG_COMPILER_CLANG
extern "C" {
    void *__dso_handle = &__dso_handle;
}
#endif
