﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_Kernel.h"
#include "kern_CPUSelect.h"
#include "kern_KMemoryManager.h"
#include "kern_KScheduler.h"
#include "kern_KSynchronization.h"
#include "kern_PageTableSelect.h"
#include "kern_InterruptManagerSelect.h"
#include "kern_KInterruptTaskManager.h"
#include "kern_HardwareTimerSelect.h"
#include "kern_New.h"
#include "kern_InterlockedSelect.h"
#include "kern_KPageTableManager.h"
#include "kern_KResourceLimit.h"
#include "kern_KPageGroup.h"
#include "kern_KUnsafeMemory.h"
#include "kern_Utility.h"
#include "kern_KMemoryLayout.h"

namespace nn { namespace kern {
Kernel::State                       Kernel::s_State;
KKernelPageTable                    Kernel::s_KernelPageTable;
KMemoryManager                      Kernel::s_KernelHeapManager;
KPageTableManager                   Kernel::s_PageTableManager;
KMemoryBlockResourceManager         Kernel::s_AppMemoryBlockManager;
KMemoryBlockResourceManager         Kernel::s_SysMemoryBlockManager;
KBlockInfoManager                   Kernel::s_BlockInfoManager;
KSynchronization                    Kernel::s_Synchronization;
KResourceLimit                      Kernel::s_SystemResourceLimit;
KUnsafeMemory                       Kernel::s_UnsafeMemory;
KWorkThreadManager                  Kernel::s_WorkThreadManager[KWorkThreadManager::WORK_THREAD_COUNT];
InterlockedVariable<int64_t>        Kernel::s_NumPageClear;
InterlockedVariable<int64_t>        Kernel::s_NumIpc;
KThread                             Kernel::s_MainThread[KCPU::NUM_CORE];
KThread                             Kernel::s_IdleThread[KCPU::NUM_CORE];

void Kernel::Initialize(int32_t coreNo)
{
    CoreLocalRegion& clr = *GetTypedPointer<Kernel::CoreLocalRegion>(KMemoryLayout::GetCoreLocalRegionBegin());
    new(&clr) CoreLocalRegion();

    KCPU::SetCurrentValue(reinterpret_cast<uintptr_t>(&clr));
    clr.currentSet.pCurrentThread           = nullptr;
    clr.currentSet.pCurrentProcess          = nullptr;
    clr.currentSet.pScheduler               = &clr.scheduler;
    clr.currentSet.pInterruptTaskManager    = &clr.interruptTaskManager;
    clr.currentSet.coreNo                   = coreNo;
    clr.currentSet.exceptionStackEnd        = GetUntypedPointer(KMemoryLayout::GetExceptionStackBottom(coreNo));

    clr.numSoftwareInterrupts   = 0;
    clr.numHardwareInterrupts   = 0;
    clr.numSystemCall           = 0;
    clr.numProcessSwitch        = 0;
    clr.numThreadSwitch         = 0;
    clr.numFpuSwitch            = 0;

    for (int i = 0; i < KCPU::NUM_PERFORMANCE_COUNTER; i++)
    {
        clr.prevCounter[i] = 0;
    }
}

void Kernel::InitializeResourceManager(KVirtualAddress buffer, size_t bufferSize)
{
    NN_KERN_ABORT_UNLESS((GetAsInteger(buffer) % NN_KERN_FINEST_PAGE_SIZE) == 0);
    NN_KERN_ABORT_UNLESS((bufferSize % NN_KERN_FINEST_PAGE_SIZE) == 0);
    size_t appMemoryBlockSize = NN_KERN_SLAB_OBJ_NUM_APP_MEMORY_BLOCK * sizeof(KMemoryBlock);
    size_t sysMemoryBlockSize = NN_KERN_SLAB_OBJ_NUM_SYS_MEMORY_BLOCK * sizeof(KMemoryBlock);
    size_t blockInfoSize      = NN_KERN_SLAB_OBJ_NUM_BLOCK_INFO       * sizeof(KBlockInfo);

    NN_KERN_ABORT_UNLESS(bufferSize > RoundUp(appMemoryBlockSize + sysMemoryBlockSize + blockInfoSize, NN_KERN_FINEST_PAGE_SIZE));
    size_t pageTableRegionSize = bufferSize - RoundUp(appMemoryBlockSize + sysMemoryBlockSize + blockInfoSize, NN_KERN_FINEST_PAGE_SIZE);
    size_t refCounterSize = RoundUp(KPageTableManager::CalcReferenceCounterBufferSize(pageTableRegionSize), NN_KERN_FINEST_PAGE_SIZE);
    NN_KERN_ABORT_UNLESS(pageTableRegionSize > refCounterSize);

    pageTableRegionSize -= refCounterSize;

    s_AppMemoryBlockManager.Initialize(buffer + pageTableRegionSize, appMemoryBlockSize);
    s_SysMemoryBlockManager.Initialize(buffer + pageTableRegionSize + appMemoryBlockSize, sysMemoryBlockSize);
    s_BlockInfoManager.Initialize(buffer + pageTableRegionSize + appMemoryBlockSize + sysMemoryBlockSize, blockInfoSize);
    s_PageTableManager.Initialize(buffer, pageTableRegionSize,
            GetTypedPointer<uint16_t>(buffer + pageTableRegionSize + appMemoryBlockSize + sysMemoryBlockSize + blockInfoSize));
}

}}

