﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_Assert.h"

namespace nn { namespace kern {

//#if 0 // 作業途中で一旦uintptr_tとしてコンパイル
#if defined(NN_SDK_BUILD_DEBUG) && NN_KERN_HAS_MMU

    // デバッグ版のみ、不正なコードはコンパイルエラーを出す。
    template <typename Tag, bool Dereferable = false>
    class KTaggedAddress
    {
        static_assert(sizeof(void*) == sizeof(uintptr_t), "");
    public:
        typedef intptr_t difference_type;

        KTaggedAddress() {}
        NN_IMPLICIT KTaggedAddress(uintptr_t addr) : m_addr(addr) {} // TODO: 一通り通ったらここをexplicitに
        template <typename T> explicit KTaggedAddress(T* p) : m_addr(reinterpret_cast<uintptr_t>(p)) {}
        KTaggedAddress operator=(KTaggedAddress other)
        {
            m_addr = other.m_addr;
            return *this;
        }

        template <typename Tag_, bool Dereferable_> friend uintptr_t GetAsInteger(KTaggedAddress<Tag_, Dereferable_> paddr);
        template <typename T, typename Tag_> friend T* GetTypedPointer(KTaggedAddress<Tag_, true> paddr);
        template <typename Tag_> friend void* GetUntypedPointer(KTaggedAddress<Tag_, true> paddr);

        template <typename Int> KTaggedAddress operator+(Int offset) const { return m_addr + offset; }
        template <typename Int> KTaggedAddress operator-(Int offset) const { return m_addr - offset; }
        template <typename Int> KTaggedAddress& operator+=(Int offset)
        {
            m_addr += offset;
            return *this;
        }
        template <typename Int> KTaggedAddress& operator-=(Int offset)
        {
            m_addr -= offset;
            return *this;
        }

        // アドレス＋アドレス は宣言されない。

        uintptr_t operator&(uintptr_t mask) const { return m_addr & mask; }
        uintptr_t operator|(uintptr_t mask) const { return m_addr | mask; }
        uintptr_t operator>>(int n) const { return m_addr >> n; }
        uintptr_t operator<<(int n) const { return m_addr << n; }

        template <typename Int> size_t operator/(Int size) const { return m_addr / size; }
        template <typename Int> difference_type operator%(Int align) const { return m_addr % align; }

        difference_type operator-(KTaggedAddress other) const { return m_addr - other.m_addr; }

        bool operator==(KTaggedAddress other) const { return m_addr == other.m_addr; }
        bool operator!=(KTaggedAddress other) const { return !(*this == other); }
        bool operator==(uintptr_t other) const { return m_addr == other; }
        bool operator!=(uintptr_t other) const { return !(*this == other); }
        bool operator< (KTaggedAddress other) const { return m_addr < other.m_addr; }
        bool operator> (KTaggedAddress other) const { return other < *this; }
        bool operator<=(KTaggedAddress other) const { return !(*this > other); }
        bool operator>=(KTaggedAddress other) const { return !(*this < other); }

        // アドレスが有効かどうかは、アドレスの数値を見るだけではわからないため、
        // operator bool と operator! を宣言してはいけない。
        // ゼロとの比較が特別な意味を持つかどうかは、アーキテクチャによる。

        // アドレス上の型はわからないため、operator* も宣言されない。
        // T* GetTypedPointer<T>(KTaggedAddress) を使う。

    private:
        uintptr_t m_addr;
    };

    static_assert(sizeof(KTaggedAddress<void, true>) == sizeof(uintptr_t), "");

    template <typename Tag, bool Dereferable> uintptr_t GetAsInteger(KTaggedAddress<Tag, Dereferable> paddr) { return paddr.m_addr; }
    template <typename T, typename Tag> T* GetTypedPointer(KTaggedAddress<Tag, true> paddr) { return reinterpret_cast<T*>(paddr.m_addr); }
    template <typename Tag> void* GetUntypedPointer(KTaggedAddress<Tag, true> paddr) { return reinterpret_cast<void*>(paddr.m_addr); }

    struct KPhysicalAddressTag
    {
    };
    typedef KTaggedAddress<KPhysicalAddressTag> KPhysicalAddress;
    static_assert(sizeof(KPhysicalAddress) == sizeof(uintptr_t), "");

    struct KVirtualAddressTag
    {
    };
    typedef KTaggedAddress<KVirtualAddressTag, true> KVirtualAddress;
    static_assert(sizeof(KVirtualAddress) == sizeof(uintptr_t), "");

    struct KProcessAddressTag
    {
    };
    typedef KTaggedAddress<KProcessAddressTag, true> KProcessAddress; // TODO: ここ本当に Derefableがtrueでいいのか
    static_assert(sizeof(KProcessAddress) == sizeof(uintptr_t), "");

#else
    // デバッグ版以外(Developmentはどうする？)では、パフォーマンスを重視し、
    // 十分な最適化がかからない場合を考慮して、uintptr_tのtypedefにする。
    // 万一最適化がこれでも弱すぎる場合、GetAsInteger などの関数をマクロにする。

    typedef uintptr_t KPhysicalAddress, KVirtualAddress, KProcessAddress;
    inline uintptr_t GetAsInteger(uintptr_t addr) { return addr; }
    template <typename T> inline T* GetTypedPointer(uintptr_t addr) { return reinterpret_cast<T*>(addr); }
    inline void* GetUntypedPointer(uintptr_t addr) { return reinterpret_cast<void*>(addr); }

#endif
    template <typename T> inline T Null() { return T(0); }

}}

