﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_KSynchronizationObject.h"
#include "kern_Kernel.h"
#include "kern_KSynchronization.h"
#include "kern_KScheduler.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_KThread.h"
#include "kern_KProcess.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KSynchronizationObject);


// 同期基本メソッドは protected で用意する

KSynchronizationObject::KSynchronizationObject()
: m_waitList()
{
    NN_KERN_THIS_ASSERT();
}

KSynchronizationObject::~KSynchronizationObject()
{
    NN_KERN_THIS_ASSERT();
}

void KSynchronizationObject::Finalize()
{
    NN_KERN_THIS_ASSERT();

#if defined NN_SDK_BUILD_DEVELOP || defined NN_SDK_BUILD_DEBUG
    {
        KScopedSchedulingLock locker;
        for (KSynchronizationObject::ThreadIterator itr = GetWaitQueueBeginIterator(); itr != GetWaitQueueEndIterator(); ++itr)
        {
            KThread* pThread = itr.GetCurrentItem();
            NN_WARNING(false, "finalize %p when %p(ID=%lld) waiting.\n", this, pThread, pThread->GetId());
        }
        NN_KERN_ASSERT(m_waitList.IsEmpty());
    }
#endif

    OnFinalizeSynchronization();
    KAutoObject::Finalize();
}

/*!
    @brief      この同期オブジェクトがシグナル状態になった際に呼び出されます。

                派生クラスの Signal イベントから呼ばれます。
*/
void KSynchronizationObject::NotifyAvailable()
{
    NN_KERN_THIS_ASSERT();

    // この同期オブジェクトがシグナル状態になったことを伝えます
    Kernel::GetSynchronization().OnAvailable(this);
}

/*!
    @brief      この同期オブジェクトが閉じられる際に呼び出されます。

*/
void KSynchronizationObject::NotifyAbort(Result reason)
{
    NN_KERN_THIS_ASSERT();
    Kernel::GetSynchronization().OnAbort(this, reason);
}

/*!
    @brief      待機スレッドリストに追加します。

                必ずスケジューラーをロックした状態で呼び出す必要があります。

    @param[in]  thread   待機リストに追加するスレッド

    @return     必ず true を返します。
*/
KSynchronizationObject::ThreadIterator
KSynchronizationObject::RegisterWaitingThread(KThread *thread)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    return m_waitList.Insert(m_waitList.GetEndIter(), thread);
}

/*!
    @brief      待機スレッドリストから指定のスレッドを削除します。

                必ずスケジューラーをロックした状態で呼び出す必要があります。

    @param[in]  itr   待機リストのイテレーター

    @return     削除したスレッドの次のイテレーターを返します。
*/
KSynchronizationObject::ThreadIterator
KSynchronizationObject::UnregisterWaitingThread(KSynchronizationObject::ThreadIterator itr)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(!m_waitList.IsEmpty());

    return m_waitList.Erase(itr);
}

/*!
    @brief      待機スレッドリストの先頭を示すイテレーターを取得します。

                必ずスケジューラーをロックした状態で呼び出す必要があります。

    @return     先頭の要素を示すイテレーターを返します。
*/
KSynchronizationObject::ThreadIterator
KSynchronizationObject::GetWaitQueueBeginIterator()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    return m_waitList.GetBeginIter();
}

/*!
    @brief      待機スレッドリストの終了を示すイテレーターを取得します。

                必ずスケジューラーをロックした状態で呼び出す必要があります。

    @return     終了を示すイテレーターを返します。
*/
KSynchronizationObject::ThreadIterator
KSynchronizationObject::GetWaitQueueEndIterator()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    return m_waitList.GetEndIter();
}

void KSynchronizationObject::DumpWaiter()
{
    NN_KERN_THIS_ASSERT();

#if defined NN_KERN_FOR_DEVELOPMENT
    {
        KScopedSchedulingLock locker;
        NN_KERN_RELEASE_LOG("Waiter Thread\n");
        bool hasWaiter = false;
        for (KSynchronizationObject::ThreadIterator itr = GetWaitQueueBeginIterator(); itr != GetWaitQueueEndIterator(); ++itr)
        {
            KThread* pThread = itr.GetCurrentItem();
            KProcess* pProcess = pThread->GetParentPointer();
            if (pProcess)
            {
                NN_KERN_RELEASE_LOG("    %p ID=%ld PID=%ld (%s)\n", pThread, pThread->GetId(), pProcess->GetId(), pProcess->GetName());
            }
            else
            {
                NN_KERN_RELEASE_LOG("    %p ID=%ld Kernel\n", pThread, pThread->GetId());
            }
            hasWaiter = true;
        }
        if (!hasWaiter)
        {
            NN_KERN_RELEASE_LOG("    Nothing\n");
        }
    }
#endif
}

}}
