﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Kernel.h"
#include "kern_KScheduler.h"
#include "kern_HardwareTimerSelect.h"
#include "kern_KThread.h"

namespace nn { namespace kern {

/*! @file

    @brief      KScopedSchedulingLockAndSleep クラス定義です。

    このインスタンスは、Sleep前の条件チェックという特定状況下で利用されます。

*/


class KScopedSchedulingLockAndSleep
{
private:
    int64_t       m_Tick;             //!< スリープから起き上がる時刻(Tick)
    KThread*      m_pThread;          //!< スリープ対象のスレッド
    KHardwareTimer* m_pTimer;

public:
    /*!
        @brief      スケジューリングロックとスリープ前処理をします。

        この呼び元は、スケジュールをロックした状態で、本当にスリープに入るべきか
        診断する処理を想定しています。

        @param[in]      thread      休眠対象のスレッド
        @param[in]      timeoutTick 起床時間(tick) 0以下の場合は起床しない

    */
    KScopedSchedulingLockAndSleep(KHardwareTimer** ppTimer, KThread* pThread, int64_t timeoutTick)
    : m_pThread(pThread)
    {
        KScheduler::s_SchedulerMutex.Lock();

        m_Tick = timeoutTick;
        if (timeoutTick > 0)
        {
            m_pTimer = &Kernel::GetHardwareTimer();
        }
        else
        {
            m_pTimer = nullptr;
        }
        *ppTimer = m_pTimer;
    }
    ~KScopedSchedulingLockAndSleep()
    {
        if (m_Tick > 0)
        {
            m_pTimer->RegisterTaskAbsolute(m_pThread, m_Tick);
        }

        // もし コンストラクタ～デストラクタ間で m_Thread がWAITING状態になっていたら、
        // 以下の関数を呼んだ瞬間に眠りに落ちます。
        KScheduler::s_SchedulerMutex.UnlockAndScheduleAll();
    }
    void Cancel()
    {
        m_Tick = 0;
    }
};

}}

