﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_KPageGroup.h"
#include "kern_Result.h"
#include "kern_KMemoryBlock.h"
#include "kern_KLightMutex.h"
#include "kern_KMemoryBlockManager.h"
#include "kern_PageTableBodySelect.h"
#include "kern_KMemoryManager.h"
#include "kern_KMemoryLayout.h"

namespace nn { namespace kern {
struct KPageProperty
{
    KMemoryPermission   permission;
    bool                isIo;
    bool                isUncached;
};

class KScopedPageTableUpdater;
class KPageTableBase
{
public:
    static const size_t PageSize = NN_KERN_FINEST_PAGE_SIZE;

public:
    Result MapMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size);
    Result UnmapMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size);
    Result SetMemoryPermission(KProcessAddress addr, size_t size, nn::svc::MemoryPermission permission);
    Result SetProcessMemoryPermission(KProcessAddress addr, size_t size, nn::svc::MemoryPermission permission);
    Result SetMemoryAttribute(KProcessAddress addr, size_t size, Bit32 mask, Bit32 attribute);
    Result SetHeapSize(KProcessAddress* pOut, size_t size);
    Result SetMaxHeapSize(size_t size);
    Result MapPageGroup(KProcessAddress addr, const KPageGroup& pg, KMemoryState state, KMemoryPermission permission);
    Result UnmapPageGroup(KProcessAddress addr, const KPageGroup& pg, KMemoryState state);
    Result MapPageGroup(
            KProcessAddress*    pAddr,
            const KPageGroup&   pg,
            KProcessAddress     regionBegin,
            size_t              regionNumPages,
            KMemoryState        state,
            KMemoryPermission   permission);
    Result MakePageGroupAndOpen(KPageGroup* pOut, KProcessAddress addr, size_t numPages, Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute) const;
    Result MakePageGroupContiguousAndOpen(KPageGroup* pOut, KProcessAddress addr, size_t numPages, Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute) const;
    Result MapCodeMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size);
    Result UnmapCodeMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size);
    Result QueryInfo(KMemoryInfo* pBlock, nn::svc::PageInfo* pPage, KProcessAddress addr) const;
    Result QueryPhysicalAddress(nn::svc::PhysicalMemoryInfo* pBlockInfo, KProcessAddress addr);
    Result QueryIoMapping(KProcessAddress* pAddr, KPhysicalAddress ioAddr, size_t ioSize);
    Result MapIo(KPhysicalAddress physAddr, size_t size, KMemoryPermission permission);
    Result MapStatic(KPhysicalAddress paddr, size_t size, KMemoryPermission permission);

    Result MapPages(KProcessAddress* pAddr, size_t numPages, KMemoryState state, KMemoryPermission permission)
    {
        return MapPages(pAddr, numPages, PageSize, Null<KPhysicalAddress>(), false, GetMapRegionBegin(state), GetMapRegionSize(state) / PageSize, state, permission);
    }
    Result MapPages(KProcessAddress* pAddr, size_t numPages, size_t align, KPhysicalAddress physAddr, KMemoryState state, KMemoryPermission permission)
    {
        return MapPages(pAddr, numPages, align, physAddr, true, GetMapRegionBegin(state), GetMapRegionSize(state) / PageSize, state, permission);
    }
    Result MapPages(KProcessAddress* pAddr, size_t numPages, size_t align, KPhysicalAddress physAddr, KProcessAddress regionBegin, size_t regionNumPages, KMemoryState state, KMemoryPermission permission)
    {
        return MapPages(pAddr, numPages, align, physAddr, true, regionBegin, regionNumPages, state, permission);
    }

    Result MapPages(KProcessAddress addr, size_t numPages, KMemoryState state, KMemoryPermission permission);

    Result UnmapPages(KProcessAddress addr, size_t numPages, KMemoryState state);

    Result InvalidateProcessDataCache(KProcessAddress addr, size_t size);

    Result ReadDebugMemory(void* buf, KProcessAddress addr, size_t size);
    Result WriteDebugMemory(KProcessAddress addr, const void* buf, size_t size);

    Result LockForDeviceAddressSpace(KPageGroup* pOut, KProcessAddress addr, size_t size, KMemoryPermission permission, bool aligned);
    Result UnlockForDeviceAddressSpace(KProcessAddress addr, size_t size);
    Result LockForTransferMemory(KPageGroup* pOut, KProcessAddress addr, size_t size, KMemoryPermission permission);
    Result UnlockForTransferMemory(KProcessAddress addr, size_t size, const KPageGroup& pg);
    Result LockForIpcUserBuffer(KProcessAddress addr, size_t size);
    Result UnlockForIpcUserBuffer(KProcessAddress addr, size_t size);
    Result LockForCodeMemory(KPageGroup* pOut, KProcessAddress addr, size_t size);
    Result UnlockForCodeMemory(KProcessAddress addr, size_t size, const KPageGroup& pg);

    Result MapTransferMemory(KProcessAddress addr, const KPageGroup& pg);
    Result UnmapTransferMemory(KProcessAddress addr, const KPageGroup& pg);

    Result CopyMemoryFromUser(KProcessAddress toAddr, size_t size,
            Bit32 toStateMask, Bit32 toState, KMemoryPermission toTestPermission,
            Bit32 toAttributeMask, Bit32 toAttribute,
            KProcessAddress fromAddr);

    Result CopyMemoryToUser(KProcessAddress toAddr, size_t size,
            KProcessAddress fromAddr,
            Bit32 fromStateMask, Bit32 fromState, KMemoryPermission fromTestPermission,
            Bit32 fromAttributeMask, Bit32 fromAttribute);

    Result CopyMemoryFromKernel(KProcessAddress toAddr, size_t size,
            Bit32 toStateMask, Bit32 toState, KMemoryPermission toTestPermission,
            Bit32 toAttributeMask, Bit32 toAttribute,
            KProcessAddress fromAddr);

    Result CopyMemoryToKernel(KProcessAddress toAddr, size_t size,
            KProcessAddress fromAddr,
            Bit32 fromStateMask, Bit32 fromState, KMemoryPermission fromTestPermission,
            Bit32 fromAttributeMask, Bit32 fromAttribute);


    Result SetupForIpc(KProcessAddress* pToAddr, size_t size,
            KProcessAddress fromAddr, KPageTableBase* pFromTable,
            KMemoryPermission testPermission, KMemoryState toState,
            bool isSend);
    Result CleanupForIpcServer(KProcessAddress addr, size_t size, KMemoryState toState);
    Result CleanupForIpcClient(KProcessAddress addr, size_t size, KMemoryState toState);


    Result CheckMemoryBlock(KProcessAddress addr, size_t size, Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute) const;

    Result MapPhysicalMemory(KProcessAddress addr, size_t size);
    Result UnmapPhysicalMemory(KProcessAddress addr, size_t size);

    Result MapPhysicalMemoryUnsafe(KProcessAddress addr, size_t size);
    Result UnmapPhysicalMemoryUnsafe(KProcessAddress addr, size_t size);

    void DumpMemoryBlocksLocked() const
    {
        m_MemoryBlocks.Dump();
    }

    void DumpMemoryBlocks() const
    {
        KScopedLightLock locker(&m_Mutex);
        DumpMemoryBlocksLocked();
    }

    void DumpPageTable() const
    {
        KScopedLightLock locker(&m_Mutex);
        m_Table.Dump(GetAsInteger(m_AddressBegin), m_AddressEnd - m_AddressBegin);
    }

    size_t CountPageTables() const
    {
        KScopedLightLock locker(&m_Mutex);
        return m_Table.CountPageTables();
    }

public:
    typedef KPageTableBody::TraverseContext TraverseContext;
    typedef KPageTableBody::TraverseData TraverseData;
    bool TraverseBegin(TraverseData* pOut, TraverseContext* pContext, KProcessAddress addr) const
    {
        KScopedLightLock locker(&m_Mutex);
        return m_Table.TraverseBegin(pOut, pContext, addr);
    }
    bool TraverseNext(TraverseData* pOut, TraverseContext* pContext) const
    {
        KScopedLightLock locker(&m_Mutex);
        return m_Table.TraverseNext(pOut, pContext);
    }

public:
    bool GetPhysicalAddress(KPhysicalAddress* pOut, KProcessAddress va) const
    {
        return m_Table.GetPhysicalAddress(pOut, va);
    }

    static bool IsLinearMapPhysicalAddress(KPhysicalAddress pa)
    {
        return KMemoryLayout::InLinearRegionPhysical(pa);
    }

    static bool IsLinearMapVirtualAddress(KVirtualAddress va)
    {
        return KMemoryLayout::InLinearRegion(va);
    }

    static KVirtualAddress GetLinearMapVirtualAddress(KPhysicalAddress pa)
    {
        NN_KERN_ASSERTMSG(IsLinearMapPhysicalAddress(pa), "not implemented paddr=%p", GetAsInteger(pa));
        return KMemoryLayout::ToLinearVirtualAddress(pa);
    }

    static KPhysicalAddress GetLinearMapPhysicalAddress(KVirtualAddress va)
    {
        NN_KERN_ASSERTMSG(IsLinearMapVirtualAddress(va), "not implemented vaddr=%p", GetAsInteger(va));
        return KMemoryLayout::ToLinearPhysicalAddress(va);
    }

    static KVirtualAddress GetHeapVirtualAddress(KPhysicalAddress pa)
    {
        NN_KERN_ASSERTMSG(IsHeapPhysicalAddress(pa), "not implemented paddr=0x%08x", GetAsInteger(pa));
        return GetLinearMapVirtualAddress(pa);
    }
    static KPhysicalAddress GetHeapPhysicalAddress(KVirtualAddress va)
    {
        NN_KERN_ASSERTMSG(IsHeapVirtualAddress(va), "not implemented paddr=0x%08x", GetAsInteger(va));
        return GetLinearMapPhysicalAddress(va);
    }
    static bool IsHeapVirtualAddress(KVirtualAddress va)
    {
        return KMemoryLayout::InHeapRegion(va);
    }
    static bool IsHeapPhysicalAddress(KPhysicalAddress pa)
    {
        return KMemoryLayout::InHeapRegionPhysical(pa);
    }

    static KVirtualAddress GetPageTableVirtualAddress(KPhysicalAddress pa)
    {
        NN_KERN_ASSERTMSG(IsLinearMapPhysicalAddress(pa),
                "not implemented paddr=%p", GetAsInteger(pa));
        return GetLinearMapVirtualAddress(pa);
    }
    static KPhysicalAddress GetPageTablePhysicalAddress(KVirtualAddress va)
    {
        NN_KERN_ASSERTMSG(IsLinearMapVirtualAddress(va),
                "not implemented vaddr=%p", GetAsInteger(va));
        return GetLinearMapPhysicalAddress(va);
    }
    bool IsInRange(KProcessAddress addr, size_t size) const
    {
        return (m_AddressBegin <= addr && addr < addr + size && addr + size - 1 <= m_AddressEnd - 1);
    }
    bool IsInRange(KProcessAddress addr) const
    {
        return m_AddressBegin <= addr && addr <= m_AddressEnd - 1;
    }
    bool IsInAslrRegion(KProcessAddress addr, size_t size) const
    {
        KProcessAddress regionBegin = GetAslrRegionBegin();
        size_t regionSize = GetAslrRegionSize();
        if (!(regionBegin <= addr && addr < addr + size && addr + size - 1 <= regionBegin + regionSize - 1))
        {
            return false;
        }
        if (!(addr + size <= m_HeapRegionBegin || m_HeapRegionEnd <= addr))
        {
            return false;
        }
        if (!(addr + size <= m_ReservedRegionBegin || m_ReservedRegionEnd <= addr))
        {
            return false;
        }
        return true;
    }
    bool IsInRange(KProcessAddress addr, size_t size, KMemoryState state) const
    {
        KProcessAddress regionBegin = GetMapRegionBegin(state);
        size_t regionSize = GetMapRegionSize(state);
        switch (state)
        {
        case KMemoryState_Free:
        case KMemoryState_Kernel:
            return (regionBegin <= addr && addr < addr + size && addr + size - 1 <= regionBegin + regionSize - 1);
        case KMemoryState_Io:
        case KMemoryState_Static:
        case KMemoryState_Code:
        case KMemoryState_CodeData:
        case KMemoryState_AliasCode:
        case KMemoryState_AliasCodeData:
        case KMemoryState_ThreadLocal:
        case KMemoryState_Shared:
        case KMemoryState_Transfered:
        case KMemoryState_SharedTransfered:
        case KMemoryState_SharedCode:
        case KMemoryState_Stack:
        case KMemoryState_GeneratedCode:
        case KMemoryState_CodeOut:
            {
                if (!(regionBegin <= addr && addr < addr + size && addr + size - 1 <= regionBegin + regionSize - 1))
                {
                    return false;
                }
                if (!(addr + size <= m_HeapRegionBegin || m_HeapRegionEnd <= addr))
                {
                    return false;
                }
                if (!(addr + size <= m_ReservedRegionBegin || m_ReservedRegionEnd <= addr))
                {
                    return false;
                }
                return true;
            }
            break;
        case KMemoryState_Normal:
            {
                if (!(regionBegin <= addr && addr < addr + size && addr + size - 1 <= regionBegin + regionSize - 1))
                {
                    return false;
                }
                if (!(addr + size <= m_ReservedRegionBegin || m_ReservedRegionEnd <= addr))
                {
                    return false;
                }
                return true;
            }
        case KMemoryState_Ipc:
        case KMemoryState_NonSecureIpc:
        case KMemoryState_NonDeviceIpc:
            {
                if (!(regionBegin <= addr && addr < addr + size && addr + size - 1 <= regionBegin + regionSize - 1))
                {
                    return false;
                }
                if (!(addr + size <= m_HeapRegionBegin || m_HeapRegionEnd <= addr))
                {
                    return false;
                }
                return true;
            }
            break;
        default:
                return false;
        }
    }

    bool IsInReservedRegion(KProcessAddress addr, size_t size) const
    {
        return IsInRange(addr, size) && (m_ReservedRegionBegin <= addr && addr < addr + size && addr + size - 1 <= m_ReservedRegionEnd - 1);
    }

protected:
    enum FillMemoryPattern
    {
        FillMemoryPattern_Zero      = 0,
        FillMemoryPattern_Stack     = 'X',
        FillMemoryPattern_Ipc       = 'Y',
        FillMemoryPattern_Allocate  = 'Z',
    };
    enum PageTableOperation
    {
        PageTableOperation_Map,
        PageTableOperation_MapRelocate,
        PageTableOperation_AllocateAndMap,
        PageTableOperation_Unmap,
        PageTableOperation_ChangeProperties,
        PageTableOperation_ChangePropertiesWithBreakBeforeMake,
    };
    KMemoryBlockResourceManager* GetMemoryBlockResourceManager() const { return m_pMemoryBlockManager; }
    Result InitializeForProcess(nn::svc::CreateProcessParameterFlag addressSapceType, bool enableAslr, bool fromBack, void* pTable, KProcessAddress begin, KProcessAddress end, KMemoryManager::Region region, KProcessAddress codeRegionAddr, size_t codeRegionSize, KMemoryBlockResourceManager* pMemoryBlockManager, KBlockInfoManager* pBlockInfoManager);
    Result InitializeForKernel(bool is64Bit, void* pTable, KProcessAddress begin, KProcessAddress end);
    void Finalize();

    virtual Result OperateImpl(
            void**              ppUpdateData,
            KProcessAddress     vaddr,
            size_t              numPages,
            KPhysicalAddress    paddr,
            bool                paIsValid,
            const KPageProperty property,
            PageTableOperation  operation
            ) = 0;

    virtual Result OperateImpl(void** ppUpdateData,
            KProcessAddress vaddr, size_t numPages, const KPageGroup& pg,
            const KPageProperty property, PageTableOperation operation) = 0;

    virtual bool GetStaticMap(KProcessAddress* pOut, KPhysicalAddress paddr, size_t size) = 0;

    bool IsPagesInRange(KProcessAddress addr, size_t numPages) const
    {
        return (m_AddressBegin <= addr &&  numPages <= (m_AddressEnd - m_AddressBegin) / PageSize && addr + numPages * PageSize - 1 <= m_AddressEnd - 1);
    }
    Result QueryInfoImpl(KMemoryInfo* pBlock, nn::svc::PageInfo* pPage, KProcessAddress addr) const;

    void Update(KMemoryBlockManagerUpdateBuffer* pBuffer, KProcessAddress addr, size_t numPages, KMemoryState state, KMemoryPermission permission, KMemoryAttribute attribute)
    {
        m_MemoryBlocks.Update(pBuffer, addr, numPages, state, permission, attribute);
    }

    virtual void PageTableUpdaterFinalize(void** ppData) = 0;

    bool IsKernel() const { return m_IsKernel; }
    bool IsAslr() const { return m_EnableAslr; }
    bool IsLockedByMe() const { return m_Mutex.IsLockedByMe(); }
    KPageTableBody& GetPageTable() { return m_Table; }
    KProcessAddress GetAddressBegin() const { return m_AddressBegin; }
    KProcessAddress GetAddressEnd() const { return m_AddressEnd; }
    KProcessAddress GetReservedRegionEnd()   const { return m_ReservedRegionEnd; }
    KProcessAddress GetHeapRegionEnd()       const { return m_HeapRegionEnd; }

    virtual size_t GetPageSize(int szc) const
    {
        switch (szc)
        {
        case 0: return PageSize;
        default:
                {
                    NN_KERNEL_PANIC("unknown szc %d", szc);
                }
                return PageSize;
        }
    }

    virtual int GetPageSizeCountMax() const { return 0; }

public:
    KBlockInfoManager* GetBlockInfoManager() const { return m_pBlockInfoManager; }
    Bit32 GetAllocateOption()                const { return m_AllocateOption; }
    KProcessAddress GetReservedRegionBegin() const { return m_ReservedRegionBegin; }
    size_t GetReservedRegionSize()           const { return GetReservedRegionEnd() - GetReservedRegionBegin(); }
    KProcessAddress GetHeapRegionBegin()     const { return m_HeapRegionBegin; }
    size_t GetHeapRegionSize()               const { return GetHeapRegionEnd() - GetHeapRegionBegin(); }
    size_t GetNormalMemorySize() const
    {
        KScopedLightLock locker(&m_Mutex);
        return (m_HeapEnd - m_HeapRegionBegin) + m_MapPhysicalMemorySize;
    }
    size_t GetCodeSize() const;
    size_t GetCodeDataSize() const;
    KProcessAddress GetStackRegionBegin() const { return m_StackRegionBegin; }
    size_t GetStackRegionSize()           const { return m_StackRegionEnd - m_StackRegionBegin; }

    KProcessAddress GetAslrRegionBegin()     const
    {
        switch (m_AddressSpaceSize)
        {
#if defined (NN_BUILD_CONFIG_ABI_LP64)
        case 39: return NN_SVC_ADDR_MAP39_BEGIN;
        case 36: return NN_SVC_ADDR_SMALL_MAP36_BEGIN;
#endif
        case 32: return NN_SVC_ADDR_SMALL_MAP32_BEGIN;
        default:
                 NN_KERN_ABORT();
                 return Null<KProcessAddress>();
        }
    }

    size_t GetAslrRegionSize()               const
    {
        switch (m_AddressSpaceSize)
        {
#if defined (NN_BUILD_CONFIG_ABI_LP64)
        case 39: return NN_SVC_ADDR_MAP39_SIZE;
        case 36: return NN_SVC_ADDR_MEMORY_REGION_LARGE36_END - NN_SVC_ADDR_SMALL_MAP36_BEGIN;
#endif
        case 32: return NN_SVC_ADDR_MEMORY_REGION_LARGE32_END - NN_SVC_ADDR_SMALL_MAP32_BEGIN;
        default:
                 NN_KERN_ABORT();
                 return 0;
        }
    }

    KProcessAddress GetMapRegionBegin(KMemoryState state) const
    {
        switch (state)
        {
        case KMemoryState_Free:
        case KMemoryState_Kernel:
            return m_AddressBegin;
        case KMemoryState_Code:
        case KMemoryState_CodeData:
            {
                switch (m_AddressSpaceSize)
                {
#if defined (NN_BUILD_CONFIG_ABI_LP64)
                case 39: return NN_SVC_ADDR_MAP39_BEGIN;
                case 36: return NN_SVC_ADDR_SMALL_MAP36_BEGIN;
#endif
                case 32: return NN_SVC_ADDR_SMALL_MAP32_BEGIN;
                default:
                         NN_KERN_ABORT();
                         return Null<KProcessAddress>();
                }
            }
            break;
        case KMemoryState_Normal:
            return m_HeapRegionBegin;
        case KMemoryState_Static:
        case KMemoryState_Io:
        case KMemoryState_ThreadLocal:
            return m_KernelMapSmallRegionBegin;
        case KMemoryState_AliasCode:
        case KMemoryState_AliasCodeData:
        case KMemoryState_Shared:
        case KMemoryState_Transfered:
        case KMemoryState_SharedTransfered:
        case KMemoryState_SharedCode:
        case KMemoryState_GeneratedCode:
        case KMemoryState_CodeOut:
            return GetAslrRegionBegin();
        case KMemoryState_Stack:
            return GetStackRegionBegin();
        case KMemoryState_Ipc:
        case KMemoryState_NonSecureIpc:
        case KMemoryState_NonDeviceIpc:
            return GetIpcRegionBegin();
        default:
            NN_KERN_ABORT();
            return Null<KProcessAddress>();
        }
        NN_KERN_ABORT();
        return Null<KProcessAddress>();
    }

    size_t GetMapRegionSize(KMemoryState state) const
    {
        switch (state)
        {
        case KMemoryState_Free:
        case KMemoryState_Kernel:
            return m_AddressEnd - m_AddressBegin;
        case KMemoryState_Code:
        case KMemoryState_CodeData:
            {
                switch (m_AddressSpaceSize)
                {
#if defined (NN_BUILD_CONFIG_ABI_LP64)
                case 39: return NN_SVC_ADDR_MAP39_SIZE;
                case 36: return NN_SVC_ADDR_SMALL_MAP36_SIZE;
#endif
                case 32: return NN_SVC_ADDR_SMALL_MAP32_SIZE;
                default:
                         NN_KERN_ABORT();
                         return 0;
                }
            }
            break;
        case KMemoryState_Io:
        case KMemoryState_Static:
        case KMemoryState_ThreadLocal:
            return m_KernelMapSmallRegionEnd - m_KernelMapSmallRegionBegin;
        case KMemoryState_Normal:
            return m_HeapRegionEnd - m_HeapRegionBegin;
        case KMemoryState_AliasCode:
        case KMemoryState_AliasCodeData:
        case KMemoryState_Shared:
        case KMemoryState_Transfered:
        case KMemoryState_SharedTransfered:
        case KMemoryState_SharedCode:
        case KMemoryState_GeneratedCode:
        case KMemoryState_CodeOut:
            return GetAslrRegionSize();
        case KMemoryState_Stack:
            return GetStackRegionSize();
        case KMemoryState_Ipc:
        case KMemoryState_NonSecureIpc:
        case KMemoryState_NonDeviceIpc:
            return GetIpcRegionSize();
        default:
            NN_KERN_ABORT();
            return 0;
        }
        NN_KERN_ABORT();
        return 0;
    }

    Result StackFillPattern(KProcessAddress stackEnd) const;
    Result GetStackUsage(size_t* pUsageSize, size_t* pStackSize, KProcessAddress stackAddr) const;

private:
    KProcessAddress GetIpcRegionBegin() const
    {
        return m_ReservedRegionBegin;
    }
    size_t GetIpcRegionSize()           const
    {
        return m_ReservedRegionEnd - m_ReservedRegionBegin;
    }

private:
    Result MapPages(KProcessAddress* pAddr, size_t numPages, size_t align, KPhysicalAddress physAddr, bool paIsValid, KProcessAddress regionBegin, size_t regionNumPages, KMemoryState state, KMemoryPermission permission);
    Result MakePageGroupImpl(KPageGroup* pOut, KProcessAddress addr, size_t numPages) const;

    Result MapPageGroupImpl(void** ppUpdateData, KProcessAddress vaddr, const KPageGroup& pg, const KPageProperty property);
    size_t GetGuardSize() const { return (IsKernel()? PageSize: 4 * PageSize); }

    // 属性チェックが正しい限り、連続して処理する
    Result CheckMemoryStateContiguous(
            KProcessAddress addr, size_t size,
            Bit32 stateMask, Bit32 state,
            Bit32 permissionMask, Bit32 permission,
            Bit32 attributeMask, Bit32 attribute) const;

    // メモリ状態、パーミッションが不連続なら中断する
    // メモリ属性はignoreAttribute に設定されたもの以外が不連続なら中断する
    Result CheckMemoryState(KMemoryState* pState, KMemoryPermission* pPermission, KMemoryAttribute* pAttribute,
            KProcessAddress addr, size_t size,
            Bit32 stateMask, Bit32 state,
            Bit32 permissionMask, Bit32 permission,
            Bit32 attributeMask, Bit32 attribute, Bit32 ignoreAttribute = KMemoryAttribute_IpcLocked | KMemoryAttribute_DeviceShared) const;

    Result CheckMemoryState(
            KProcessAddress addr, size_t size,
            Bit32 stateMask, Bit32 state,
            Bit32 permissionMask, Bit32 permission,
            Bit32 attributeMask, Bit32 attribute, Bit32 ignoreAttribute = KMemoryAttribute_IpcLocked | KMemoryAttribute_DeviceShared) const
    {
        return CheckMemoryState(nullptr, nullptr, nullptr,
                addr, size,
                stateMask, state,
                permissionMask, permission,
                attributeMask, attribute, ignoreAttribute);
    }

    Result CheckMemoryState(const KMemoryInfo& mi, Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute) const;
    Result LockMemoryAndOpen(KPageGroup* pOut,
            KProcessAddress addr, size_t size,
            Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute,
            KMemoryPermission newPermission, Bit32 lockAttribute);
    Result UnlockMemory(KProcessAddress addr, size_t size,
            Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute,
            KMemoryPermission newPermission, Bit32 lockAttribute, const KPageGroup* pPageGroup);

    class KScopedPageTableUpdater
    {
        void* m_pData;
        KPageTableBase* m_pTable;
    public:
        explicit KScopedPageTableUpdater(KPageTableBase* pTable) : m_pData(nullptr), m_pTable(pTable) {}
        ~KScopedPageTableUpdater() { m_pTable->PageTableUpdaterFinalize(GetDataPtr()); }
        void** GetDataPtr() { return &m_pData; }
    };

    NN_FORCEINLINE KProcessAddress FindFreeAreaInRegion(KProcessAddress regionBegin, size_t regionNumPages, size_t numPages, size_t align, size_t alignOffset, size_t guardPages) const;

    Result MakePageGroupForIpcClient(KPageGroup* pOut, KProcessAddress addr, size_t size, bool isSend, Bit32 allocateOption) const;
    Result SetupForIpcClient(KPageGroup* pOut, KProcessAddress addr, size_t size, KMemoryPermission testPermission, KMemoryState toState, bool isSend, Bit32 allocateOption);

    Result NotCmpMemoryBlock(size_t* pFillSize, size_t* pSize, KProcessAddress addr, KMemoryState state, FillMemoryPattern pattern) const;
    Result FillMemoryBlock(KProcessAddress addr, KMemoryState state, FillMemoryPattern pattern) const;

    Result SetupForIpcServer(KProcessAddress* pToAddr, size_t size,
        KProcessAddress fromAddr, KMemoryPermission testPermission, KMemoryState toState, const KPageGroup& pg);

protected:
    static const size_t MapPhysicalMaxAlign = NN_KERN_FINEST_PAGE_SIZE * 0x40000;
    static const size_t RegionAlign = 0x00200000;

private:
    KProcessAddress             m_AddressBegin;
    KProcessAddress             m_AddressEnd;
    KProcessAddress             m_HeapRegionBegin;
    KProcessAddress             m_HeapRegionEnd;
    KProcessAddress             m_HeapEnd;
    KProcessAddress             m_ReservedRegionBegin;
    KProcessAddress             m_ReservedRegionEnd;
    KProcessAddress             m_StackRegionBegin;
    KProcessAddress             m_StackRegionEnd;
    KProcessAddress             m_KernelMapSmallRegionBegin;
    KProcessAddress             m_KernelMapSmallRegionEnd;
    size_t                      m_MaxHeapSize;
    size_t                      m_MapPhysicalMemorySize;
    mutable KLightMutex         m_Mutex;
    mutable KLightMutex         m_MapPhysMutex;
    KPageTableBody              m_Table;
    KMemoryBlockManager         m_MemoryBlocks;
    Bit32                       m_AllocateOption;
    int                         m_AddressSpaceSize;
    bool                        m_IsKernel;
    bool                        m_EnableAslr;
    KMemoryBlockResourceManager*    m_pMemoryBlockManager;
    KBlockInfoManager*              m_pBlockInfoManager;
protected:
    FillMemoryPattern           m_AllocateFillMemoryPattern;
    FillMemoryPattern           m_IpcFillMemoryPattern;
    FillMemoryPattern           m_StackFillMemoryPattern;
};

}}

