﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "kern_KSynchronizationObject.h"
#include "kern_KThread.h"
#include "kern_KLightClientSession.h"
#include "kern_KLightServerSession.h"

namespace nn { namespace kern {

class KClientPort;

class KLightSession
    : public KObjectAdaptor<KLightSession, KAutoObjectWithList>
{
private:
    enum State: uint8_t
    {
        STATE_INVALID,                              //!< 無効な状態(未初期化)
        STATE_ACTIVE,                               //!< セッションが有効
        STATE_CLIENT_CLOSED,                        //!< クライアントセッションが閉じられました
        STATE_SERVER_CLOSED                         //!< サーバーセッションが閉じられました
    };

private:
    KLightServerSession         m_ServerSession;    //!< サーバーセッション
    KLightClientSession         m_ClientSession;    //!< クライアントセッション
    State                       m_State;            //!< セッションの状態
    KClientPort*                m_pPort;
    uintptr_t                   m_Name;
    KProcess*                   m_pOwner;
    bool                        m_IsInitialized;

public:
    static const size_t DataSize = sizeof(Bit32) * 7;
    static const Bit32  ReplyFlag = (1u << 31);

    explicit        KLightSession() : m_State(STATE_INVALID), m_pOwner(nullptr), m_IsInitialized(false) {}
    virtual         ~KLightSession() {}

    void            Initialize(KClientPort* pPort, uintptr_t name);
    virtual void    Finalize();
    uintptr_t       GetName() const { return m_Name; }
    virtual bool IsInitialized() const { return m_IsInitialized; }
    virtual uintptr_t GetPostFinalizeArgument() const { return reinterpret_cast<uintptr_t>(m_pOwner); }
    static void PostFinalize(uintptr_t arg);

    //! サーバーセッションが閉じたことの通知 (サーバーセッションから呼ばれます)
    void        OnServerClosed();
    //! クライアントセッションが閉じたことの通知 (クライアントセッションから呼ばれます)
    void        OnClientClosed();
    //! サーバーセッションが閉じられたことを示します
    bool        IsServerClosed() const              { return m_State != STATE_ACTIVE; }
    //! クライアントセッションが閉じられたことを示します
    bool        IsClientClosed() const              { return m_State != STATE_ACTIVE; }
    //! クライアントからリクエストがあったこの通知
    Result      OnRequest(KThread* pRequestSource) { return m_ServerSession.OnRequest(pRequestSource); }
    //! サーバーセッションを取得します
    KLightServerSession&  GetServerSession()             { return m_ServerSession; }
    //! クライアントセッションを取得します
    KLightClientSession&  GetClientSession()             { return m_ClientSession; }

    //! KAutoObjectのプリセット関数セット定義です。クラスの末尾に記述する必要があります
    NN_AUTOOBJECT_DERIVED_FUNCSET(KLightSession, KAutoObject)
};

}}

