﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_KKipReader.h"
#include "kern_KMemoryManager.h"
#include "kern_Uncompress.h"
#include "kern_Kernel.h"
#include "kern_SystemControl.h"
#include <cstring>

namespace nn { namespace kern {
bool KKipReader::Bind(uint8_t* pData)
{
    KipFormat* pKipHeader = reinterpret_cast<KipFormat*>(pData);
    if (pKipHeader->IsValid())
    {
        m_pKipHeader = pKipHeader;
        return true;
    }

    return false;
}

Result KKipReader::MakeProcessParameter(nn::svc::CreateProcessParameter* pHeader, bool enableAslr) const
{
    if ((m_pKipHeader->GetExAddress() % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((m_pKipHeader->GetRoAddress() % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((m_pKipHeader->GetRwAddress() % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if (!(m_pKipHeader->GetExAddress() + KMemoryManager::SizeToPage(m_pKipHeader->GetExMemorySize()) <= m_pKipHeader->GetRoAddress() &&
            m_pKipHeader->GetExAddress() <= m_pKipHeader->GetExAddress() + KMemoryManager::SizeToPage(m_pKipHeader->GetExMemorySize())))
    {
        return nn::svc::ResultInvalidAddress();
    }
    if (!(m_pKipHeader->GetRoAddress() + KMemoryManager::SizeToPage(m_pKipHeader->GetRoMemorySize()) <= m_pKipHeader->GetRwAddress() &&
            m_pKipHeader->GetRoAddress() <= m_pKipHeader->GetRoAddress() + KMemoryManager::SizeToPage(m_pKipHeader->GetRoMemorySize())))
    {
        return nn::svc::ResultInvalidAddress();
    }
    if (!(m_pKipHeader->GetRwAddress() + m_pKipHeader->GetRwMemorySize() <= m_pKipHeader->GetBssAddress() &&
            m_pKipHeader->GetRwAddress() <= m_pKipHeader->GetRwAddress() + m_pKipHeader->GetRwMemorySize()))
    {
        return nn::svc::ResultInvalidAddress();
    }
    if (!(m_pKipHeader->GetBssAddress() <= m_pKipHeader->GetBssAddress() + m_pKipHeader->GetBssMemorySize()))
    {
        return nn::svc::ResultInvalidAddress();
    }
    if (m_pKipHeader->Is64BitSpace() && !m_pKipHeader->Is64Bit())
    {
        return nn::svc::ResultInvalidCombination();
    }

    uintptr_t startAddr = m_pKipHeader->GetExAddress();
    uintptr_t endAddr   = m_pKipHeader->GetBssMemorySize()? m_pKipHeader->GetBssAddress() + m_pKipHeader->GetBssMemorySize(): m_pKipHeader->GetRwAddress() + m_pKipHeader->GetRwMemorySize();
#if defined (NN_BUILD_CONFIG_ABI_LP64)
    uintptr_t mapBegin  = (m_pKipHeader->Is64BitSpace()? NN_SVC_ADDR_MAP39_BEGIN: NN_SVC_ADDR_SMALL_MAP32_BEGIN);
    uintptr_t mapEnd    = (m_pKipHeader->Is64BitSpace()? NN_SVC_ADDR_MAP39_END  : NN_SVC_ADDR_SMALL_MAP32_END);
#else
    uintptr_t mapBegin  = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
    uintptr_t mapEnd    = NN_SVC_ADDR_SMALL_MAP32_END;
    NN_KERN_ABORT_UNLESS(!m_pKipHeader->Is64BitSpace());
#endif

    NN_KERN_ABORT_UNLESS(startAddr == 0);
    pHeader->memoryAddress          = mapBegin + startAddr;
    pHeader->memoryNumPages         = KMemoryManager::SizeToPage(endAddr - startAddr);
    pHeader->programId              = m_pKipHeader->GetProgramId();
    pHeader->version                = m_pKipHeader->GetVersion();
    m_pKipHeader->GetName(pHeader->name, sizeof(pHeader->name));
    pHeader->flags                  = 0;

    NN_KERN_ABORT_UNLESS(pHeader->memoryAddress / NN_KERN_FINEST_PAGE_SIZE + pHeader->memoryNumPages <= mapEnd / NN_KERN_FINEST_PAGE_SIZE);

    if (enableAslr)
    {
        size_t freePages = mapEnd / KProcess::ProcessAlign - (pHeader->memoryAddress + NN_KERN_FINEST_PAGE_SIZE * pHeader->memoryNumPages + (KProcess::ProcessAlign - 1)) / KProcess::ProcessAlign;
        pHeader->memoryAddress += KSystemControl::GetRandomValue(0, freePages) * KProcess::ProcessAlign;
        pHeader->flags |= nn::svc::CreateProcessParameterFlag_EnableAslr;
    }
    NN_LOG("load %s at %p\n", pHeader->name, pHeader->memoryAddress);

    if (m_pKipHeader->Is64Bit())
    {
        pHeader->flags |= nn::svc::CreateProcessParameterFlag_64Bit;
    }
    if (m_pKipHeader->Is64BitSpace())
    {
        pHeader->flags |= nn::svc::CreateProcessParameterFlag_AddressSpace64Bit;
    }
    else
    {
        pHeader->flags |= nn::svc::CreateProcessParameterFlag_AddressSpace32Bit;
    }

    return ResultSuccess();
}

Result KKipReader::Load(KProcessAddress addr, const nn::svc::CreateProcessParameter& params) const
{
    KProcessAddress exAddr = addr + m_pKipHeader->GetExAddress();
    KProcessAddress roAddr = addr + m_pKipHeader->GetRoAddress();
    KProcessAddress rwAddr = addr + m_pKipHeader->GetRwAddress();
    const uint8_t* pEx = reinterpret_cast<const uint8_t *>(m_pKipHeader + 1);
    const uint8_t* pRo = pEx + m_pKipHeader->GetExBinarySize();
    const uint8_t* pRw = pRo + m_pKipHeader->GetRoBinarySize();

    ::std::memset(GetUntypedPointer(addr), 0, params.memoryNumPages * NN_KERN_FINEST_PAGE_SIZE);

    size_t textNumPages = KMemoryManager::SizeToPage(m_pKipHeader->GetExMemorySize());
    if (textNumPages)
    {
        ::std::memcpy(GetUntypedPointer(exAddr), pEx, m_pKipHeader->GetExBinarySize());
        if (m_pKipHeader->IsExCompressed())
        {
            UncompressBackward(GetUntypedPointer(exAddr + m_pKipHeader->GetExBinarySize()));
        }
    }

    size_t rodataNumPages = KMemoryManager::SizeToPage(m_pKipHeader->GetRoMemorySize());
    if (rodataNumPages)
    {
        ::std::memcpy(GetUntypedPointer(roAddr), pRo, m_pKipHeader->GetRoBinarySize());
        if (m_pKipHeader->IsRoCompressed())
        {
            UncompressBackward(GetUntypedPointer(roAddr + m_pKipHeader->GetRoBinarySize()));
        }
    }

    size_t rwNumPages = KMemoryManager::SizeToPage(m_pKipHeader->GetRwMemorySize());
    if (rwNumPages)
    {
        ::std::memcpy(GetUntypedPointer(rwAddr), pRw, m_pKipHeader->GetRwBinarySize());
        if (m_pKipHeader->IsRwCompressed())
        {
            UncompressBackward(GetUntypedPointer(rwAddr + m_pKipHeader->GetRwBinarySize()));
        }
    }
    KCPU::FlushEntireDataCache();
    KCPU::InvalidateEntireInstructionCache();

    return ResultSuccess();
}

Result KKipReader::SetPermission(KProcessPageTable* pPageTable, const nn::svc::CreateProcessParameter& params) const
{
    Result result = ResultSuccess();

    if (m_pKipHeader->GetExMemorySize())
    {
        uintptr_t addr = m_pKipHeader->GetExAddress();
        addr += params.memoryAddress;
        result = pPageTable->SetProcessMemoryPermission(addr, KMemoryManager::SizeToPage(m_pKipHeader->GetExMemorySize()) * NN_KERN_FINEST_PAGE_SIZE, nn::svc::MemoryPermission_ReadExecute);
        if (result.IsFailure())
        {
            return result;
        }
    }
    if (m_pKipHeader->GetRoMemorySize())
    {
        uintptr_t addr = m_pKipHeader->GetRoAddress();
        addr += params.memoryAddress;
        result = pPageTable->SetProcessMemoryPermission(addr, KMemoryManager::SizeToPage(m_pKipHeader->GetRoMemorySize()) * NN_KERN_FINEST_PAGE_SIZE, nn::svc::MemoryPermission_Read);
        if (result.IsFailure())
        {
            return result;
        }
    }
    if (m_pKipHeader->GetRwMemorySize() || m_pKipHeader->GetBssMemorySize())
    {
        uintptr_t startAddr = m_pKipHeader->GetRwMemorySize()? m_pKipHeader->GetRwAddress(): m_pKipHeader->GetBssAddress();
        uintptr_t endAddr   = m_pKipHeader->GetBssMemorySize()? m_pKipHeader->GetBssAddress() + m_pKipHeader->GetBssMemorySize(): m_pKipHeader->GetRwAddress() + m_pKipHeader->GetRwMemorySize();
        startAddr += params.memoryAddress;
        endAddr += params.memoryAddress;
        result = pPageTable->SetProcessMemoryPermission(startAddr, KMemoryManager::SizeToPage(endAddr - startAddr) * NN_KERN_FINEST_PAGE_SIZE, nn::svc::MemoryPermission_ReadWrite);
        if (result.IsFailure())
        {
            return result;
        }
    }
    return result;
}

size_t KKipReader::GetSize() const
{
    return sizeof(*m_pKipHeader) +
        m_pKipHeader->GetExBinarySize() +
        m_pKipHeader->GetRoBinarySize() +
        m_pKipHeader->GetRwBinarySize();
}
size_t KKipReader::GetMemorySize() const
{
    uintptr_t startAddr = m_pKipHeader->GetExAddress();
    uintptr_t endAddr   = m_pKipHeader->GetBssMemorySize()? m_pKipHeader->GetBssAddress() + m_pKipHeader->GetBssMemorySize(): m_pKipHeader->GetRwAddress() + m_pKipHeader->GetRwMemorySize();
    return KMemoryManager::SizeToPage(endAddr - startAddr) * NN_KERN_FINEST_PAGE_SIZE;
}
}}


