﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_KInterruptTask.h"
#include "kern_KTimerTask.h"
#include "kern_KSimpleLock.h"
#include "kern_KScopedDisableDispatch.h"
#include "kern_InterruptManagerSelect.h"

namespace nn { namespace kern {
    struct KTimerTaskCompare
    {
        int operator() (const KTimerTask& a, const KTimerTask& b)
        {
            int64_t tick = a.GetTime();

            if (tick < b.GetTime())
            {
                return -1;
            }
            return 1;
        }
    };

    class KHardwareTimerBase : public KInterruptTask
    {
    private:
        typedef IntrusiveRbTree<KTimerTask, IntrusiveRbTreeBaseNodeTraits<KTimerTask>, KTimerTaskCompare> TimerTaskList;
    public:
        KHardwareTimerBase() : m_pTopTask(nullptr) {}
        virtual KInterruptTask* OnInterrupt(int32_t interruptNo)
        {
            NN_UNUSED(interruptNo);
            return this;
        }

        void CancelTask(KTimerTask* pTask)
        {
            KScopedDisableDispatch dd;
            KScopedSimpleLock lock(&m_Mutex);

            int64_t tick = pTask->GetTime();
            if (tick > 0)
            {
                TimerTaskList::iterator it = m_List.iterator_to(*pTask);
                it = m_List.erase(it);
                pTask->SetTime(0);
                if (m_pTopTask == pTask)
                {
                    m_pTopTask = ((it != m_List.end())? &*it: nullptr);
                }
            }
        }

    protected:
        int64_t DoInterruptTaskImpl(int64_t now)
        {
            for (;;)
            {
                KTimerTask* pTask = m_pTopTask;
                if (pTask == nullptr)
                {
                    return 0;
                }

                int64_t tick = pTask->GetTime();
                if (tick > now)
                {
                    return tick;
                }

                TimerTaskList::iterator it = m_List.iterator_to(*pTask);
                it = m_List.erase(it);
                pTask->SetTime(0);

                m_pTopTask = ((it != m_List.end())? &*it: nullptr);

                pTask->OnTimer();
            }
        }

        KSimpleLock& GetLockMutex() { return m_Mutex; }

        bool RegisterTaskAbsoluteImpl(KTimerTask* pTask, int64_t tick)
        {
            NN_KERN_ASSERT(tick > 0);

            pTask->SetTime(tick);
            m_List.insert(*pTask);

            if (m_pTopTask != nullptr && m_pTopTask->GetTime() <= tick)
            {
                return false;
            }
            m_pTopTask = pTask;
            return true;
        }

    private:
        KSimpleLock     m_Mutex;
        TimerTaskList   m_List;
        KTimerTask*     m_pTopTask;
    };



}}

