﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>

#include "kern_Platform.h"
#include "kern_KDumpObject.h"
#include "kern_KThread.h"
#include "kern_KPort.h"
#include "kern_KEvent.h"
#include "kern_KInterruptEvent.h"
#include "kern_KPageGroup.h"
#include "kern_KMemoryBlock.h"
#include "kern_KSharedMemory.h"
#include "kern_KTransferMemory.h"
#include "kern_KCodeMemory.h"
#include "kern_KDeviceAddressSpace.h"
#include "kern_KSession.h"
#include "kern_KLightSession.h"
#include "kern_KObjectName.h"
#include "kern_Kernel.h"
#include "kern_DebugString.h"
#include "kern_KSessionRequest.h"
#include "kern_KServerSession.h"
#include "kern_InterruptManagerSelect.h"
#include "kern_KWritableEvent.h"
#include "kern_KReadableEvent.h"
#include "kern_KTrace.h"
#include "kern_KPageBuffer.h"
#include "kern_SystemControl.h"
#include "kern_DebugSelect.h"
#include "kern_Utility.h"
#include "kern_KScopedSchedulingLockAndSleep.h"
#include "kern_DpcManager.h"

namespace nn { namespace kern { namespace KDumpObject {
namespace {
void DumpThread(KThread* pThread)
{
    const char* threadState[] = {
        "INITIALIZED",
        "WAIT",
        "RUNNABLE",
        "TERMINATED"
    };
    NN_UNUSED(threadState);
    KProcess* pProcess = pThread->GetParentPointer();
    if (pProcess)
    {
        NN_KERN_RELEASE_LOG("Thread ID=%5lld pid=%3lld %-11s Pri=%2d %-11s KernelStack=%4ld/%4ld Run=%d Ideal=%d Affinity=%01lx\n",
                pThread->GetId(), pProcess->GetId(), pProcess->GetName(),
                pThread->GetPriority(), threadState[pThread->GetState()],
                pThread->GetKernelStackUsage(), NN_KERN_THREAD_SVC_STACK_SIZE,
                pThread->GetRunningProcessor(),
                pThread->GetIdealProcessor(),
                pThread->GetAffinityMask().GetAffinityMask());
        NN_KERN_RELEASE_LOG("    State: 0x%x Suspend: 0x%x Dpc: 0x%x\n",
                pThread->GetStateRaw(),
                pThread->GetSuspendedRaw(),
                pThread->GetDpc());
        NN_KERN_RELEASE_LOG("    TLS: %p (%p)\n",
                pThread->GetThreadLocalRegionAddr(),
                pThread->GetThreadLocalRegionHeapAddr());
#ifdef NN_KERN_STACK_FILL_PATTERN
        {
            size_t usageSize;
            size_t stackSize;
            Result result = pThread->GetUserStackInfo(&usageSize, &stackSize);
            if (result.IsSuccess())
            {
                NN_KERN_RELEASE_LOG("    UserStack: %ld / %ld\n", usageSize, stackSize);
            }
            else
            {
                NN_KERN_RELEASE_LOG("    UserStack: Failed\n");
            }
        }
#endif
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
        NN_KERN_RELEASE_LOG("    Cycle Count: %16lu (%16lu)\n",
                pThread->GetPerformanceCounter(0),
                pThread->GetPerformanceCounter(1));
        NN_KERN_RELEASE_LOG("    Instruction: %16lu (%16lu)\n",
                pThread->GetPerformanceCounter(2),
                pThread->GetPerformanceCounter(3));
        NN_KERN_RELEASE_LOG("    Cache Miss : %16lu (%16lu)\n",
                pThread->GetPerformanceCounter(4),
                pThread->GetPerformanceCounter(5));
#endif
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
        NN_KERN_RELEASE_LOG("    Switch     :%16lu\n", pThread->GetNumSwitch());
        NN_KERN_RELEASE_LOG("    FPU EXCE   :%16lu\n", pThread->GetNumFpuSwitch());
#endif
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
        NN_KERN_RELEASE_LOG("    IPC        :%16lu\n", pThread->GetNumIpc());
        NN_KERN_RELEASE_LOG("    IPC REPLY  :%16lu\n", pThread->GetNumIpcReply());
        NN_KERN_RELEASE_LOG("    IPC RECV   :%16lu\n", pThread->GetNumIpcRecv());
#endif
    }
    else
    {
        NN_KERN_RELEASE_LOG("Thread ID=%5lld (kernel) Pri=%2d %-11s KernelStack=%4ld/%4ld Run=%d Ideal=%d Affinity=%01lx\n",
                pThread->GetId(), pThread->GetPriority(),
                threadState[pThread->GetState()],
                pThread->GetKernelStackUsage(), NN_KERN_THREAD_SVC_STACK_SIZE,
                pThread->GetRunningProcessor(),
                pThread->GetIdealProcessor(),
                pThread->GetAffinityMask().GetAffinityMask());
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
        NN_KERN_RELEASE_LOG("    Cycle Count: %16lu (%16lu)\n",
                pThread->GetPerformanceCounter(0),
                pThread->GetPerformanceCounter(1));
        NN_KERN_RELEASE_LOG("    Instruction: %16lu (%16lu)\n",
                pThread->GetPerformanceCounter(2),
                pThread->GetPerformanceCounter(3));
        NN_KERN_RELEASE_LOG("    Cache Miss : %16lu (%16lu)\n",
                pThread->GetPerformanceCounter(4),
                pThread->GetPerformanceCounter(5));
#endif
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
        NN_KERN_RELEASE_LOG("    Switch     :%16lu\n", pThread->GetNumSwitch());
#endif
    }
}

void DumpThreadCallstack(KThread* pThread)
{
    const char* threadState[] = {
        "INITIALIZED",
        "WAIT",
        "RUNNABLE",
        "TERMINATED"
    };
    NN_UNUSED(threadState);

    KProcess* pProcess = pThread->GetParentPointer();
    if (pProcess)
    {
        NN_KERN_RELEASE_LOG("Thread ID=%lld pid=%lld name=%s Pri=%d %s KernelStack=%ld/%ld\n",
                pThread->GetId(), pProcess->GetId(), pProcess->GetName(),
                pThread->GetPriority(), threadState[pThread->GetState()],
                pThread->GetKernelStackUsage(), NN_KERN_THREAD_SVC_STACK_SIZE);
        KDebug::PrintRegister(pThread);
        KDebug::PrintBacktrace(pThread);
    }
    else
    {
        NN_KERN_RELEASE_LOG("Thread ID=%lld (kernel) Pri=%d %s KernelStack=%ld/%ld\n",
                pThread->GetId(), pThread->GetPriority(),
                threadState[pThread->GetState()],
                pThread->GetKernelStackUsage(), NN_KERN_THREAD_SVC_STACK_SIZE);
    }
}

void DumpProcess(KProcess* pProcess)
{
    NN_UNUSED(pProcess);
    NN_KERN_RELEASE_LOG("Process ID=%3lld index=%3d State=%d (%s)\n",
            pProcess->GetId(), pProcess->GetIndex(), pProcess->GetStateNoCheck(),
            pProcess->GetName());
}

void DumpHandle(KProcess* pProcess, const KProcess::ListAccessor& a)
{
    NN_KERN_RELEASE_LOG("PID=%d (%s)\n", static_cast<Bit32>(pProcess->GetId()), pProcess->GetName());
    const KObjectContainer::ObjectListIterator end = a.GetEnd();
    KHandleTable& handleTable = pProcess->GetHandleTable();
    int maxHandle = handleTable.GetMaxCount();
    for (int i = 0; i < maxHandle; i++)
    {
        nn::svc::Handle handle;
        KAutoObject* pObj =  handleTable.GetObjectByIndex(&handle, i);
        if (pObj)
        {
            if (pObj->DynamicCast<KServerSession*>())
            {
                KServerSession* pSession = pObj->DynamicCast<KServerSession*>();
                NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s Client=%p\n",
                        handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(),
                        &pSession->GetParent()->GetClientSession());
                pSession->Dump();
            }
            else if (pObj->DynamicCast<KClientSession*>())
            {
                KClientSession* pSession = pObj->DynamicCast<KClientSession*>();
                NN_UNUSED(pSession);
                NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s Server=%p\n",
                        handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(),
                        &pSession->GetParent()->GetServerSession());
            }
            else if (pObj->DynamicCast<KThread*>())
            {
                KThread* pThread = pObj->DynamicCast<KThread*>();
                NN_UNUSED(pThread);
                NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s ID=%d PID=%d\n",
                        handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(),
                        static_cast<Bit32>(pThread->GetId()),
                        static_cast<Bit32>(pThread->GetParentPointer()? pThread->GetParentPointer()->GetId(): 0xffffffff));
            }
            else if (pObj->DynamicCast<KProcess*>())
            {
                KProcess* pObjProcess = pObj->DynamicCast<KProcess*>();
                NN_UNUSED(pObjProcess);
                NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s ID=%d\n",
                        handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(),
                        static_cast<Bit32>(pObjProcess->GetId()));
            }
            else if (pObj->DynamicCast<KSharedMemory*>())
            {
                KSharedMemory* pSharedMemory = pObj->DynamicCast<KSharedMemory*>();
                KProcess* pOwner = nullptr;
                NN_UNUSED(pOwner);
                for (KObjectContainer::ObjectListIterator it0 = a.GetBegin(); it0 != end; it0++)
                {
                    if (static_cast<KProcess*>(&*it0)->GetId() == pSharedMemory->GetOwnerId())
                    {
                        pOwner = static_cast<KProcess*>(&*it0);
                        break;
                    }
                }
                NN_KERN_NULL_ASSERT(pOwner);
                NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s Size=%ld KB OwnerPID=%d (%s)\n",
                        handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(),
                        pSharedMemory->GetSize() / 1024,
                        static_cast<Bit32>(pOwner->GetId()),
                        pOwner->GetName());
            }
            else if (pObj->DynamicCast<KTransferMemory*>())
            {
                KTransferMemory* pTransferMemory = pObj->DynamicCast<KTransferMemory*>();
                KProcess* pOwner = pTransferMemory->GetOwner();
                NN_UNUSED(pOwner);
                NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s OwnerPID=%d (%s) OwnerAddress=%lx Size=%ld KB \n",
                        handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(),
                        static_cast<Bit32>(pOwner->GetId()),
                        pOwner->GetName(),
                        GetAsInteger(pTransferMemory->GetFromAddress()),
                        pTransferMemory->GetSize() / 1024
                        );
            }
            else if (pObj->DynamicCast<KCodeMemory*>())
            {
                KCodeMemory* pCodeMemory = pObj->DynamicCast<KCodeMemory*>();
                KProcess* pOwner = pCodeMemory->GetOwner();
                NN_UNUSED(pOwner);
                NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s OwnerPID=%d (%s) OwnerAddress=%lx Size=%ld KB \n",
                        handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(),
                        static_cast<Bit32>(pOwner->GetId()),
                        pOwner->GetName(),
                        GetAsInteger(pCodeMemory->GetFromAddress()),
                        pCodeMemory->GetSize() / 1024
                        );
            }
            else if (pObj->DynamicCast<KInterruptEvent*>())
            {
                KInterruptEvent* pInterEvent = pObj->DynamicCast<KInterruptEvent*>();
                NN_UNUSED(pInterEvent);
                NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s irq=%d\n",
                        handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(), pInterEvent->GetIrqNo());
            }
            else if (pObj->DynamicCast<KWritableEvent*>())
            {
                KWritableEvent* pWriteEvent = pObj->DynamicCast<KWritableEvent*>();
                KEvent* pEvent = pWriteEvent->GetParent();
                if (pEvent)
                {
                    NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s Pair=%p\n",
                            handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(), &pEvent->GetReadableEvent());
                }
                else
                {
                    NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s\n",
                            handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName());
                }
            }
            else if (pObj->DynamicCast<KReadableEvent*>())
            {
                KReadableEvent* pReadableEvent = pObj->DynamicCast<KReadableEvent*>();
                KEvent* pEvent = pReadableEvent->GetParent();
                if (pEvent)
                {
                    NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s Pair=%p\n",
                            handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName(), &pEvent->GetWritableEvent());
                }
                else
                {
                    NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s\n",
                            handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName());
                }
            }
            else
            {
                NN_KERN_RELEASE_LOG("Handle %08x pObj=%p Ref=%d Type=%s\n",
                        handle, pObj, pObj->GetReferenceCount() - 1, pObj->GetTypeName());
            }
            if (pObj->DynamicCast<KSynchronizationObject*>())
            {
                KSynchronizationObject* pSync = pObj->DynamicCast<KSynchronizationObject*>();
                pSync->DumpWaiter();
            }
            pObj->Close();
        }
    }
    NN_KERN_RELEASE_LOG("%d(max %d)/%d used.\n", handleTable.GetCount(), maxHandle, handleTable.GetTableSize());
    NN_KERN_RELEASE_LOG("\n\n");
}

void DumpMemory(KProcess* pProcess)
{
    NN_KERN_RELEASE_LOG("Process ID=%lld (%s)\n", pProcess->GetId(), pProcess->GetName());
    pProcess->GetPageTable().Dump();
    size_t codeTotal = pProcess->GetPageTable().GetCodeSize();
    size_t codeDataTotal = pProcess->GetPageTable().GetCodeDataSize();
    size_t heapTotal = pProcess->GetPageTable().GetNormalMemorySize();
    size_t initialStack = pProcess->GetMainStackSize();
    size_t sharedMemTotal = 0;
    NN_UNUSED(codeTotal);
    NN_UNUSED(codeDataTotal);
    NN_UNUSED(heapTotal);
    NN_UNUSED(initialStack);
    pProcess->GetPageTable().GetCodeSize();
    {
        KSharedMemory::ListAccessor s;
        const KObjectContainer::ObjectListIterator ends = s.GetEnd();
        for (KObjectContainer::ObjectListIterator its = s.GetBegin(); its != ends; its++)
        {
            KSharedMemory* pShared = static_cast<KSharedMemory*>(&*its);
            if (pShared->GetOwnerId() == pProcess->GetId())
            {
                sharedMemTotal += pShared->GetSize();
            }
        }
    }

    NN_KERN_RELEASE_LOG("---\n");
    NN_KERN_RELEASE_LOG("Code         %8ld KB\n", codeTotal / 1024);
    NN_KERN_RELEASE_LOG("CodeData     %8ld KB\n", codeDataTotal / 1024);
    NN_KERN_RELEASE_LOG("Heap         %8ld KB\n", heapTotal / 1024);
    NN_KERN_RELEASE_LOG("SharedMemory %8ld KB\n", sharedMemTotal / 1024);
    NN_KERN_RELEASE_LOG("InitialStack %8ld KB\n", initialStack / 1024);
    NN_KERN_RELEASE_LOG("---\n");
    NN_KERN_RELEASE_LOG("TOTAL        %8ld KB\n", (codeTotal + codeDataTotal + heapTotal + sharedMemTotal + initialStack) / 1024);
    NN_KERN_RELEASE_LOG("\n\n");
}

void DumpPageTable(KProcess* pProcess)
{
    NN_KERN_RELEASE_LOG("Process ID=%lld (%s)\n", pProcess->GetId(), pProcess->GetName());
    pProcess->GetPageTable().DumpPageTable();
    NN_KERN_RELEASE_LOG("\n\n");
}

}


void DumpThread()
{
    NN_KERN_RELEASE_LOG("Dump Thread\n");
    KThread::ListAccessor a;
    const KObjectContainer::ObjectListIterator end = a.GetEnd();
    for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
    {
        KThread* pThread = static_cast<KThread*>(&*it);
        DumpThread(pThread);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpThread(Bit64 tid)
{
    NN_KERN_RELEASE_LOG("Dump Thread\n");
    KThread* pThread = KThread::GetThreadById(tid);
    if (pThread)
    {
        KScopedAutoObject<KThread> autoCloser(pThread);
        DumpThread(pThread);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpThreadCallstack()
{
    NN_KERN_RELEASE_LOG("Dump Thread Callstack\n");
    KThread::ListAccessor a;
    const KObjectContainer::ObjectListIterator end = a.GetEnd();
    for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
    {
        KThread* pThread = static_cast<KThread*>(&*it);
        DumpThreadCallstack(pThread);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpThreadCallstack(Bit64 tid)
{
    NN_KERN_RELEASE_LOG("Dump Thread Callstack\n");
    KThread* pThread = KThread::GetThreadById(tid);
    if (pThread)
    {
        KScopedAutoObject<KThread> autoCloser(pThread);
        DumpThreadCallstack(pThread);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpProcess()
{
    NN_KERN_RELEASE_LOG("Dump Process\n");
    KProcess::ListAccessor a;
    const KObjectContainer::ObjectListIterator end = a.GetEnd();
    for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
    {
        KProcess* pProcess = static_cast<KProcess*>(&*it);
        DumpProcess(pProcess);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpProcess(Bit64 pid)
{
    NN_KERN_RELEASE_LOG("Dump Process\n");
    KProcess* pProcess = KProcess::GetProcessFromId(pid);
    if (pProcess)
    {
        KScopedAutoObject<KProcess> autoCloser(pProcess);
        DumpProcess(pProcess);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpHandle()
{
    NN_KERN_RELEASE_LOG("Dump Handle\n");
    KProcess::ListAccessor a;
    const KObjectContainer::ObjectListIterator end = a.GetEnd();
    for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
    {
        KProcess* pProcess = static_cast<KProcess*>(&*it);
        DumpHandle(pProcess, a);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpHandle(Bit64 pid)
{
    NN_KERN_RELEASE_LOG("Dump Handle\n");
    KProcess* pProcess = KProcess::GetProcessFromId(pid);
    if (pProcess)
    {
        KScopedAutoObject<KProcess> autoCloser(pProcess);
        KProcess::ListAccessor a;
        DumpHandle(pProcess, a);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpKernelObject()
{
    NN_KERN_RELEASE_LOG("Kernel Object\n");
    NN_KERN_RELEASE_LOG("    used / peak / num\n");
#define DUMP_INFO(kobj) \
    NN_KERN_RELEASE_LOG(#kobj "\n"); \
    NN_KERN_RELEASE_LOG("    %d / %d / %d\n", kobj::GetSlabSize() - kobj::GetNumRemain(), kobj::GetPeakNum(), kobj::GetSlabSize())

    DUMP_INFO(KPageBuffer);
    DUMP_INFO(KEvent);
    DUMP_INFO(KInterruptEvent);
    DUMP_INFO(KProcess);
    DUMP_INFO(KThread);
    DUMP_INFO(KPort);
    DUMP_INFO(KSharedMemory);
    DUMP_INFO(KTransferMemory);
    DUMP_INFO(KCodeMemory);
    DUMP_INFO(KDeviceAddressSpace);
    DUMP_INFO(KDebug);
    DUMP_INFO(KSession);
    DUMP_INFO(KLightSession);
    DUMP_INFO(KLinkedListNode);
    DUMP_INFO(KThreadLocalPage);
    DUMP_INFO(KObjectName);
    DUMP_INFO(KEventInfo);
    DUMP_INFO(KSessionRequest);
    DUMP_INFO(KResourceLimit);
#undef DUMP_INFO
    NN_KERN_RELEASE_LOG("\n");

    {
        NN_KERN_RELEASE_LOG("Memory Block usage\n");
        NN_KERN_RELEASE_LOG("    used  /  peak  /  num\n");
        NN_KERN_RELEASE_LOG("App %6ld / %6ld / %6ld\n",
                Kernel::GetAppMemoryBlockManager().GetUsed(),
                Kernel::GetAppMemoryBlockManager().GetPeak(),
                Kernel::GetAppMemoryBlockManager().GetNum());
        NN_KERN_RELEASE_LOG("Sys %6ld / %6ld / %6ld\n",
                Kernel::GetSysMemoryBlockManager().GetUsed(),
                Kernel::GetSysMemoryBlockManager().GetPeak(),
                Kernel::GetSysMemoryBlockManager().GetNum());
    }
    {
        NN_KERN_RELEASE_LOG("KBlockInfo usage\n");
        NN_KERN_RELEASE_LOG("    used  /  peak  /  num\n");
        NN_KERN_RELEASE_LOG("  %6ld / %6ld / %6ld\n",
                Kernel::GetBlockInfoManager().GetUsed(),
                Kernel::GetBlockInfoManager().GetPeak(),
                Kernel::GetBlockInfoManager().GetNum());
    }
    {
        NN_KERN_RELEASE_LOG("Page Table usage\n");
        NN_KERN_RELEASE_LOG("    used  /  peak  /  num\n");
        NN_KERN_RELEASE_LOG("  %6ld / %6ld / %6ld\n",
                Kernel::GetPageTableManager().GetUsed(),
                Kernel::GetPageTableManager().GetPeak(),
                Kernel::GetPageTableManager().GetNum());
    }
    NN_KERN_RELEASE_LOG("\n");

    size_t processTables = 0;
    {
        KProcess::ListAccessor a;
        const KObjectContainer::ObjectListIterator end = a.GetEnd();
        for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
        {
            KProcess* pProcess = static_cast<KProcess*>(&*it);

            int thread = 0;
            {
                KThread::ListAccessor a0;
                const KObjectContainer::ObjectListIterator end0 = a0.GetEnd();
                for (KObjectContainer::ObjectListIterator it0 = a0.GetBegin(); it0 != end0; it0++)
                {
                    KThread* pThread = static_cast<KThread*>(&*it0);
                    if (pThread->GetParentPointer() == pProcess)
                    {
                        thread++;
                    }
                }
            }

            int event = 0;
            {
                KEvent::ListAccessor a0;
                const KObjectContainer::ObjectListIterator end0 = a0.GetEnd();
                for (KObjectContainer::ObjectListIterator it0 = a0.GetBegin(); it0 != end0; it0++)
                {
                    KEvent* pEvent = static_cast<KEvent*>(&*it0);
                    if (pEvent->GetOwner() == pProcess)
                    {
                        event++;
                    }
                }
            }
            size_t tables = pProcess->GetPageTable().CountPageTables();
            NN_UNUSED(tables);
            processTables += tables;

            size_t extraResSize = pProcess->GetExtraResourceSize();
            size_t extraResUsedSize = pProcess->GetExtraResourceUsage();
            NN_UNUSED(extraResSize);
            NN_UNUSED(extraResUsedSize);

            NN_KERN_RELEASE_LOG("%-12s: PID=%3lld Thread %4d / Event %4d / PageTable %5ld\n",
                    pProcess->GetName(), pProcess->GetId(), thread, event, tables);
            if (extraResSize != 0)
            {
                NN_KERN_RELEASE_LOG("                    used  /  peak  /  num\n");
                NN_KERN_RELEASE_LOG("    ExtraResource %6ld / %6ld / %6ld\n",
                        pProcess->GetUnusedPageManager().GetUsed(), pProcess->GetUnusedPageManager().GetPeak(), pProcess->GetUnusedPageManager().GetNum());
                NN_KERN_RELEASE_LOG("    MemoryBlock   %6ld / %6ld / %6ld\n",
                        pProcess->GetMemoryBlockResourceManager().GetUsed(), pProcess->GetMemoryBlockResourceManager().GetPeak(), pProcess->GetMemoryBlockResourceManager().GetNum());
                NN_KERN_RELEASE_LOG("    PageTable     %6ld / %6ld / %6ld\n",
                        pProcess->GetPageTableManager().GetUsed(), pProcess->GetPageTableManager().GetPeak(), pProcess->GetPageTableManager().GetNum());
                NN_KERN_RELEASE_LOG("    BlockInfo     %6ld / %6ld / %6ld\n",
                        pProcess->GetBlockInfoManager().GetUsed(), pProcess->GetBlockInfoManager().GetPeak(), pProcess->GetBlockInfoManager().GetNum());
            }
        }
        NN_KERN_RELEASE_LOG("Process PageTable %ld / Kernel PageTable %ld\n", processTables, Kernel::GetKernelPageTable().CountPageTables());
        NN_KERN_RELEASE_LOG("\n");
    }

    {
        uint64_t cur;
        uint64_t lim;
        NN_UNUSED(cur);
        NN_UNUSED(lim);

        cur = Kernel::Kernel::GetSystemResourceLimit().GetCurrentValue(nn::svc::LimitableResource_PhysicalMemoryMax);
        lim = Kernel::Kernel::GetSystemResourceLimit().GetLimitValue(nn::svc::LimitableResource_PhysicalMemoryMax);
        NN_KERN_RELEASE_LOG("System ResourceLimit PhysicalMemory 0x%01x_%08x / 0x%01x_%08x\n",
                static_cast<uint32_t>(cur >> 32), static_cast<uint32_t>(cur),
                static_cast<uint32_t>(lim >> 32), static_cast<uint32_t>(lim));
        NN_KERN_RELEASE_LOG("System ResourceLimit Thread         %4lld / %4lld\n",
                Kernel::Kernel::GetSystemResourceLimit().GetCurrentValue(nn::svc::LimitableResource_ThreadCountMax),
                Kernel::Kernel::GetSystemResourceLimit().GetLimitValue(nn::svc::LimitableResource_ThreadCountMax));
        NN_KERN_RELEASE_LOG("System ResourceLimit Event          %4lld / %4lld\n",
                Kernel::Kernel::GetSystemResourceLimit().GetCurrentValue(nn::svc::LimitableResource_EventCountMax),
                Kernel::Kernel::GetSystemResourceLimit().GetLimitValue(nn::svc::LimitableResource_EventCountMax));
        NN_KERN_RELEASE_LOG("System ResourceLimit TransferMemory %4lld / %4lld\n",
                Kernel::Kernel::GetSystemResourceLimit().GetCurrentValue(nn::svc::LimitableResource_TransferMemoryCountMax),
                Kernel::Kernel::GetSystemResourceLimit().GetLimitValue(nn::svc::LimitableResource_TransferMemoryCountMax));
        NN_KERN_RELEASE_LOG("System ResourceLimit Session        %4lld / %4lld\n",
                Kernel::Kernel::GetSystemResourceLimit().GetCurrentValue(nn::svc::LimitableResource_SessionCountMax),
                Kernel::Kernel::GetSystemResourceLimit().GetLimitValue(nn::svc::LimitableResource_SessionCountMax));

        {
            KResourceLimit::ListAccessor a;
            const KObjectContainer::ObjectListIterator end = a.GetEnd();
            for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
            {
                KResourceLimit* pResourceLimit = static_cast<KResourceLimit*>(&*it);
                cur = pResourceLimit->GetCurrentValue(nn::svc::LimitableResource_PhysicalMemoryMax);
                lim = pResourceLimit->GetLimitValue(nn::svc::LimitableResource_PhysicalMemoryMax);
                NN_KERN_RELEASE_LOG("ResourceLimit %d PhysicalMemory 0x%01x_%08x / 0x%01x_%08x\n",
                        pResourceLimit->GetIndex(),
                        static_cast<uint32_t>(cur >> 32), static_cast<uint32_t>(cur),
                        static_cast<uint32_t>(lim >> 32), static_cast<uint32_t>(lim));

            }
            NN_KERN_RELEASE_LOG("\n");
        }
    }
    {
        uint64_t cur;
        uint64_t lim;
        NN_UNUSED(cur);
        NN_UNUSED(lim);

        KMemoryManager& mm = Kernel::GetKernelHeapManager();
        cur = mm.GetSize() - mm.GetFreeSize();
        lim = mm.GetSize();
        NN_KERN_RELEASE_LOG("Kernel Heap Size 0x%01x_%08x / 0x%01x_%08x\n",
                static_cast<uint32_t>(cur >> 32), static_cast<uint32_t>(cur),
                static_cast<uint32_t>(lim >> 32), static_cast<uint32_t>(lim));

        cur = mm.GetSize(KMemoryManager::Region_Application) - mm.GetFreeSize(KMemoryManager::Region_Application);
        lim = mm.GetSize(KMemoryManager::Region_Application);
        NN_KERN_RELEASE_LOG("    Application     0x%01x_%08x / 0x%01x_%08x\n",
                static_cast<uint32_t>(cur >> 32), static_cast<uint32_t>(cur),
                static_cast<uint32_t>(lim >> 32), static_cast<uint32_t>(lim));
        mm.DumpFreeList(KMemoryManager::Region_Application);
        NN_KERN_RELEASE_LOG("\n");

        cur = mm.GetSize(KMemoryManager::Region_Applet) - mm.GetFreeSize(KMemoryManager::Region_Applet);
        lim = mm.GetSize(KMemoryManager::Region_Applet);
        NN_KERN_RELEASE_LOG("    Applet          0x%01x_%08x / 0x%01x_%08x\n",
                static_cast<uint32_t>(cur >> 32), static_cast<uint32_t>(cur),
                static_cast<uint32_t>(lim >> 32), static_cast<uint32_t>(lim));
        mm.DumpFreeList(KMemoryManager::Region_Applet);
        NN_KERN_RELEASE_LOG("\n");

        cur = mm.GetSize(KMemoryManager::Region_SecureSystem) - mm.GetFreeSize(KMemoryManager::Region_SecureSystem);
        lim = mm.GetSize(KMemoryManager::Region_SecureSystem);
        NN_KERN_RELEASE_LOG("    SecureSystem    0x%01x_%08x / 0x%01x_%08x\n",
                static_cast<uint32_t>(cur >> 32), static_cast<uint32_t>(cur),
                static_cast<uint32_t>(lim >> 32), static_cast<uint32_t>(lim));
        mm.DumpFreeList(KMemoryManager::Region_SecureSystem);
        NN_KERN_RELEASE_LOG("\n");

        cur = mm.GetSize(KMemoryManager::Region_NonSecureSystem) - mm.GetFreeSize(KMemoryManager::Region_NonSecureSystem);
        lim = mm.GetSize(KMemoryManager::Region_NonSecureSystem);
        NN_KERN_RELEASE_LOG("    NonSecureSystem 0x%01x_%08x / 0x%01x_%08x\n",
                static_cast<uint32_t>(cur >> 32), static_cast<uint32_t>(cur),
                static_cast<uint32_t>(lim >> 32), static_cast<uint32_t>(lim));
        mm.DumpFreeList(KMemoryManager::Region_NonSecureSystem);
        NN_KERN_RELEASE_LOG("\n");
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpKernelMemory()
{
    NN_KERN_RELEASE_LOG("Dump Kernel Memory Info\n");
    Kernel::GetKernelPageTable().Dump();
    NN_KERN_RELEASE_LOG("\n\n");
}

void DumpMemory()
{
    NN_KERN_RELEASE_LOG("Dump Memory Info\n");
    KProcess::ListAccessor a;
    const KObjectContainer::ObjectListIterator end = a.GetEnd();
    for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
    {
        KProcess* pProcess = static_cast<KProcess*>(&*it);
        DumpMemory(pProcess);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpMemory(Bit64 pid)
{
    NN_KERN_RELEASE_LOG("Dump Memory Info\n");
    KProcess* pProcess = KProcess::GetProcessFromId(pid);
    if (pProcess)
    {
        KScopedAutoObject<KProcess> autoCloser(pProcess);
        DumpMemory(pProcess);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpKernelPageTable()
{
    NN_KERN_RELEASE_LOG("Dump Kernel PageTable\n");
    Kernel::GetKernelPageTable().DumpPageTable();
    NN_KERN_RELEASE_LOG("\n\n");
}

void DumpPageTable()
{
    NN_KERN_RELEASE_LOG("Dump PageTable\n");
    KProcess::ListAccessor a;
    const KObjectContainer::ObjectListIterator end = a.GetEnd();
    for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
    {
        KProcess* pProcess = static_cast<KProcess*>(&*it);
        DumpPageTable(pProcess);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpPageTable(Bit64 pid)
{
    NN_KERN_RELEASE_LOG("Dump PageTable\n");
    KProcess* pProcess = KProcess::GetProcessFromId(pid);
    if (pProcess)
    {
        KScopedAutoObject<KProcess> autoCloser(pProcess);
        DumpPageTable(pProcess);
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpKernelCpuUtilization()
{
    const size_t MaxLog = 64;
    uint32_t cpuUtilization[MaxLog];
    KAutoObject* obj[MaxLog];
    int64_t ctxSwitch[KCPU::NUM_CORE];
    int64_t numUpdate[KCPU::NUM_CORE];
    int64_t numCallUpdate[KCPU::NUM_CORE];
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
    uint32_t threadRunnable[MaxLog];
    uint32_t threadSwitch[MaxLog];
#endif
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
    Bit64 prevPmcc[KCPU::NUM_CORE];
    Bit64 newPmcc[KCPU::NUM_CORE];
    Bit32 prevInst[KCPU::NUM_CORE];
    Bit32 newInst[KCPU::NUM_CORE];
    Bit32 prevL2fill[KCPU::NUM_CORE];
    Bit32 newL2fill[KCPU::NUM_CORE];
#endif
    int64_t startTick;
    size_t index;
    size_t num;
    DpcManager::Sync();
    {
        KThread::ListAccessor a;
        startTick = KHardwareTimer::GetTick();
        const KObjectContainer::ObjectListIterator end = a.GetEnd();
        index = 0;
        for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
        {
            KThread* pThread = static_cast<KThread*>(&*it);
            KProcess* pProcess = pThread->GetParentPointer();
            if (!pProcess)
            {
                if (index < MaxLog)
                {
                    if (pThread->Open())
                    {
                        cpuUtilization[index] = pThread->GetCpuTime();
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
                        threadRunnable[index] = pThread->GetRunnableTime();
                        threadSwitch[index] = pThread->GetNumSwitch();
#endif
                        obj[index] = pThread;
                        index++;
                    }
                }
            }
        }
        num = index;
    }
    for (int i = 0; i < KCPU::NUM_CORE; i++)
    {
        ctxSwitch[i] = Kernel::GetNumThreadSwitch(i);
        numUpdate[i] = Kernel::GetNumSchedulerUpdate(i);
        numCallUpdate[i] = Kernel::GetNumCallSchedulerUpdate(i);
    }
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
    KCPU::GetCycleCounter(prevPmcc);
    KCPU::GetPerformanceCounter(prevInst, 2);
    KCPU::GetPerformanceCounter(prevL2fill, 4);
#endif
    int64_t timeout = KHardwareTimer::GetTick() + nn::svc::Tick(TimeSpan::FromSeconds(1));
    GetCurrentThread().Sleep(timeout);
    DpcManager::Sync();
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
    KCPU::GetCycleCounter(newPmcc);
    KCPU::GetPerformanceCounter(newInst, 2);
    KCPU::GetPerformanceCounter(newL2fill, 4);
#endif

    for (index = 0; index < num; index++)
    {
        KThread* pThread = static_cast<KThread*>(obj[index]);
        cpuUtilization[index] = pThread->GetCpuTime() - cpuUtilization[index];
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
        threadRunnable[index] = pThread->GetRunnableTime() - threadRunnable[index];
        threadSwitch[index] = pThread->GetNumSwitch() - threadSwitch[index];
#endif
    }
    for (int i = 0; i < KCPU::NUM_CORE; i++)
    {
        ctxSwitch[i] = Kernel::GetNumThreadSwitch(i) - ctxSwitch[i];
        numUpdate[i] = Kernel::GetNumSchedulerUpdate(i) - numUpdate[i];
        numCallUpdate[i] = Kernel::GetNumCallSchedulerUpdate(i) - numCallUpdate[i];
    }
    int64_t endTick = KHardwareTimer::GetTick();

    for (index = 0; index < num; index++)
    {
        uint64_t x = static_cast<uint64_t>(cpuUtilization[index]) * 1000 / (endTick - startTick);
        NN_UNUSED(x);
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
        uint64_t y = static_cast<uint64_t>(threadRunnable[index]) * 1000 / (endTick - startTick);
        NN_UNUSED(y);
#endif
        KThread* pThread = static_cast<KThread*>(obj[index]);
        NN_KERN_RELEASE_LOG("tid=%3lld (kernel) %3lld.%lld%% pri=%2d af=%llx"
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
                " Run %3lld.%lld%% swt %6u"
#endif
                "\n",
                pThread->GetId(), x / 10, x % 10, pThread->GetPriority(), pThread->GetAffinityMask().GetAffinityMask()
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
                , y / 10, y % 10, threadSwitch[index]
#endif
                );
        pThread->Close();
    }
    for (int i = 0; i < KCPU::NUM_CORE; i++)
    {
        NN_KERN_RELEASE_LOG("Core%d swt %6lld resch %6lld update %6lld"
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
                " CC %lld Inst %d L2 Fill %d"
#endif
                "\n"
                , i, ctxSwitch[i], numUpdate[i], numCallUpdate[i]
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
                , newPmcc[i] - prevPmcc[i], newInst[i] - prevInst[i], newL2fill[i] - prevL2fill[i]
#endif
                );
    }

    NN_KERN_RELEASE_LOG("\n");
}

void DumpCpuUtilization()
{
    const size_t MaxLog = 64;
    uint32_t cpuUtilization[MaxLog];
    KAutoObject* obj[MaxLog];
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
    uint32_t ipc[MaxLog];
    uint32_t ipcRecv[MaxLog];
#endif
    size_t index;
    size_t num;

    int64_t startTick;
    DpcManager::Sync();
    {
        KProcess::ListAccessor a;
        startTick = KHardwareTimer::GetTick();
        const KObjectContainer::ObjectListIterator end = a.GetEnd();
        index = 0;
        for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
        {
            KProcess* pProcess = static_cast<KProcess*>(&*it);
            if (index < MaxLog)
            {
                if (pProcess->Open())
                {
                    cpuUtilization[index] = pProcess->GetCpuTime();
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
                    ipc[index] = pProcess->GetNumIpc();
                    ipcRecv[index] = pProcess->GetNumIpcRecv();
#endif
                    obj[index] = pProcess;
                    index++;
                }
            }
        }
        num = index;
    }

    int64_t timeout = KHardwareTimer::GetTick() + nn::svc::Tick(TimeSpan::FromSeconds(1));
    GetCurrentThread().Sleep(timeout);
    DpcManager::Sync();

    for (index = 0; index < num; index++)
    {
        KProcess* pProcess = static_cast<KProcess*>(obj[index]);
        cpuUtilization[index] = pProcess->GetCpuTime() - cpuUtilization[index];
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
        ipc[index] = pProcess->GetNumIpc() - ipc[index];
        ipcRecv[index] = pProcess->GetNumIpcRecv() - ipcRecv[index];
#endif
    }
    int64_t endTick = KHardwareTimer::GetTick();

    for (index = 0; index < num; index++)
    {
        uint64_t x = static_cast<uint64_t>(cpuUtilization[index]) * 1000 / (endTick - startTick);
        NN_UNUSED(x);
        KProcess* pProcess = static_cast<KProcess*>(obj[index]);
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
        NN_KERN_RELEASE_LOG("pid=%3lld %-11s %3lld.%lld%% Ipc %5u IpcRecv %5u\n", pProcess->GetId(), pProcess->GetName(), x / 10, x % 10, ipc[index], ipcRecv[index]);
#else
        NN_KERN_RELEASE_LOG("pid=%3lld %-11s %3lld.%lld%%\n", pProcess->GetId(), pProcess->GetName(), x / 10, x % 10);
#endif
#if defined(NN_KERN_ENABLE_SVC_PROFILE)
        for (int svcNo = 0; svcNo < 128; svcNo++)
        {
            size_t n = pProcess->GetNumSystemCall(svcNo);
            if (n != 0)
            {
                NN_KERN_RELEASE_LOG("    svc%d %ld\n", svcNo, n);
            }
        }
#endif
        pProcess->Close();
    }

    NN_KERN_RELEASE_LOG("\n");
}


void DumpCpuUtilization(Bit64 pid)
{
    const size_t MaxLog = 64;
    uint32_t cpuUtilization[MaxLog];
    KAutoObject* obj[MaxLog];
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
    uint32_t threadRunnable[MaxLog];
    uint32_t threadSwitch[MaxLog];
    uint32_t threadCoreSwitch[MaxLog];
#endif
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
    uint32_t ipc[MaxLog];
    uint32_t ipcRecv[MaxLog];
#endif
    size_t index;
    size_t num;

    int64_t startTick;
    DpcManager::Sync();
    {
        KThread::ListAccessor a;
        startTick = KHardwareTimer::GetTick();
        const KObjectContainer::ObjectListIterator end = a.GetEnd();
        index = 0;
        for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
        {
            KThread* pThread = static_cast<KThread*>(&*it);
            KProcess* pProcess = pThread->GetParentPointer();
            if (pProcess && pProcess->GetId() == pid)
            {
                if (index < MaxLog)
                {
                    if (pThread->Open())
                    {
                        cpuUtilization[index] = pThread->GetCpuTime();
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
                        threadRunnable[index] = pThread->GetRunnableTime();
                        threadSwitch[index] = pThread->GetNumSwitch();
                        threadCoreSwitch[index] = pThread->GetNumCoreSwitch();
#endif
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
                        ipc[index] = pThread->GetNumIpc();
                        ipcRecv[index] = pThread->GetNumIpcRecv();
#endif
                        obj[index] = pThread;
                        index++;
                    }
                }
            }
        }
        num = index;
    }

    int64_t timeout = KHardwareTimer::GetTick() + nn::svc::Tick(TimeSpan::FromSeconds(1));
    GetCurrentThread().Sleep(timeout);
    DpcManager::Sync();

    for (index = 0; index < num; index++)
    {
        KThread* pThread = static_cast<KThread*>(obj[index]);
        cpuUtilization[index] = pThread->GetCpuTime() - cpuUtilization[index];
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
        threadRunnable[index] = pThread->GetRunnableTime() - threadRunnable[index];
        threadSwitch[index] = pThread->GetNumSwitch() - threadSwitch[index];
        threadCoreSwitch[index] = pThread->GetNumCoreSwitch() - threadCoreSwitch[index];
#endif
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
        ipc[index] = pThread->GetNumIpc() - ipc[index];
        ipcRecv[index] = pThread->GetNumIpcRecv() - ipcRecv[index];
#endif
    }
    int64_t endTick = KHardwareTimer::GetTick();

    for (index = 0; index < num; index++)
    {
        uint64_t x = static_cast<uint64_t>(cpuUtilization[index]) * 1000 / (endTick - startTick);
        NN_UNUSED(x);
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
        uint64_t y = static_cast<uint64_t>(threadRunnable[index]) * 1000 / (endTick - startTick);
        NN_UNUSED(y);
#endif
        KThread* pThread = static_cast<KThread*>(obj[index]);
        KProcess* pProcess = pThread->GetParentPointer();
        NN_UNUSED(pProcess);
        NN_KERN_RELEASE_LOG("tid=%3lld pid=%3lld %-11s %3lld.%lld%% pri=%2d af=%llx"
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
                " Run %3lld.%lld%% swt %6u mig %6u"
#endif
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
                " Ipc %5u IpcRecv %5u"
#endif
                "\n",
                pThread->GetId(), pProcess->GetId(), pProcess->GetName(), x / 10, x % 10, pThread->GetPriority(), pThread->GetAffinityMask().GetAffinityMask()
#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
                , y / 10, y % 10, threadSwitch[index], threadCoreSwitch[index]
#endif
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
                , ipc[index], ipcRecv[index]
#endif
                );
#if defined(NN_KERN_ENABLE_SVC_PROFILE)
        for (int svcNo = 0; svcNo < 128; svcNo++)
        {
            size_t n = pThread->GetNumSystemCall(svcNo);
            if (n != 0)
            {
                NN_KERN_RELEASE_LOG("    svc%d %ld\n", svcNo, n);
            }
        }
#endif
        pThread->Close();
    }

    NN_KERN_RELEASE_LOG("\n");
}

void DumpPort(KProcess* pProcess, const KProcess::ListAccessor& a)
{
    NN_KERN_RELEASE_LOG("Dump Port PID=%lld (%s)\n", pProcess->GetId(), pProcess->GetName());

    KHandleTable& handleTable = pProcess->GetHandleTable();
    int maxHandle = handleTable.GetMaxCount();
    for (int i = 0; i < maxHandle; i++)
    {
        nn::svc::Handle handle;
        KAutoObject* pObj =  handleTable.GetObjectByIndex(&handle, i);
        if (pObj)
        {
            if (pObj->DynamicCast<KServerPort*>())
            {
                KServerPort* pServerPort = pObj->DynamicCast<KServerPort*>();
                uintptr_t portName = pServerPort->GetParent()->GetName();
                const KClientPort* pClientPort = &pServerPort->GetParent()->GetClientPort();
                KProcess* pClientPortProcess = nullptr;

                {
                    const KObjectContainer::ObjectListIterator end = a.GetEnd();
                    for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
                    {
                        KProcess* p = static_cast<KProcess*>(&*it);
                        for (int j = 0; j < p->GetHandleTable().GetMaxCount(); j++)
                        {
                            nn::svc::Handle h;
                            KAutoObject* pObj2 = p->GetHandleTable().GetObjectByIndex(&h, j);
                            if (pObj2 != nullptr)
                            {
                                pObj2->Close();
                                if (pObj2 == pClientPort)
                                {
                                    pClientPortProcess = p;
                                    pClientPortProcess->Open();
                                    break;
                                }
                            }
                        }
                        if (pClientPortProcess != nullptr)
                        {
                            break;
                        }
                    }
                }

                char name[9] = {};
                if (pClientPortProcess != nullptr)
                {
                    if (pClientPortProcess->GetPageTable().CopyMemoryToKernel(KProcessAddress(name), 8, portName, 0, 0, KMemoryPermission_UserRead, 0, 0).IsFailure())
                    {
                        std::memset(name, 0, sizeof(name));
                    }
                    pClientPortProcess->Close();
                }
                NN_KERN_RELEASE_LOG("%-9s: Handle %08x pObj=%p %3d / %3d / %3d\n",
                        name,
                        handle,
                        pObj,
                        pClientPort->GetNumSessions(),
                        pClientPort->GetPeakSessions(),
                        pClientPort->GetMaxSessions());

                {
                    const KObjectContainer::ObjectListIterator end = a.GetEnd();
                    for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
                    {
                        KProcess* p = static_cast<KProcess*>(&*it);
                        for (int j = 0; j < p->GetHandleTable().GetMaxCount(); j++)
                        {
                            nn::svc::Handle h;
                            KAutoObject* pObj2 = p->GetHandleTable().GetObjectByIndex(&h, j);
                            if (pObj2 != nullptr)
                            {
                                if (pObj2->DynamicCast<KClientSession*>())
                                {
                                    KClientSession* pSession = pObj2->DynamicCast<KClientSession*>();
                                    if (pSession->GetParent()->GetParent() == pClientPort)
                                    {
                                        NN_KERN_RELEASE_LOG("    Client %p Server %p %-12s: PID=%3lld\n",
                                                pSession, &pSession->GetParent()->GetServerSession(), p->GetName(), p->GetId());
                                    }
                                }
                                pObj2->Close();
                            }
                        }
                    }
                }
            }
            pObj->Close();
        }
    }
    NN_KERN_RELEASE_LOG("\n");
}

void DumpPort()
{
    KProcess::ListAccessor a;
    const KObjectContainer::ObjectListIterator end = a.GetEnd();
    for (KObjectContainer::ObjectListIterator it = a.GetBegin(); it != end; it++)
    {
        KProcess* pProcess = static_cast<KProcess*>(&*it);
        DumpPort(pProcess, a);
    }
}

void DumpPort(Bit64 pid)
{
    KProcess* pProcess = KProcess::GetProcessFromId(pid);
    if (pProcess)
    {
        KScopedAutoObject<KProcess> autoCloser(pProcess);
        KProcess::ListAccessor a;
        DumpPort(pProcess, a);
    }
}


}}}

