﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>

#include "kern_KSynchronizationObject.h"
#include "kern_KSpinLockMutex.h"
#include "kern_KEventInfo.h"
#include "kern_KObjectAdaptor.h"
#include "kern_KWorkerTask.h"
#include "kern_PageTableSelect.h"
#include "kern_KLightMutex.h"
#include "kern_KProcess.h"

namespace nn { namespace kern {
    class KProcess;
    class KThread;

class KDebugBase: public KSynchronizationObject
{
private:
    typedef nn::util::IntrusiveList<KEventInfo, nn::util::IntrusiveListBaseNodeTraits<KEventInfo>> DebugEventList;
    typedef DebugEventList::iterator            DebugEventIterator;

private:
    DebugEventList      m_DebugEventList;
    Bit32               m_ContinueFlag;
    KProcess*           m_Process;
    KLightMutex         m_Mutex;
    KProcess::State     m_OldState;

    KThread*            GetThreadFromId(const Bit32 threadId);

    void                PushEventQueue(KEventInfo* pInfo);
    void                DestroyEventQueue();
    Bit32               GetContinueFlag() const { return m_ContinueFlag; }

    static Result       ProcessDebugEvent(const nn::svc::DebugEvent eventtype, const uintptr_t param1, const uintptr_t param2, const uintptr_t param3, const uintptr_t param4, const uintptr_t param5);
    static bool         IsAsynchronousDebugEvent(const nn::svc::DebugEvent eventtype);

    virtual void        OnFinalizeSynchronization();

    void                PushDebugEvent(const nn::svc::DebugEvent eventtype, const uintptr_t param1 = 0, const uintptr_t param2 = 0, const uintptr_t param3 = 0, const uintptr_t param4 = 0, const uintptr_t param5 = 0);

    Result              SetWriteContext(const KThread* pThread);
    bool                IsEmptyEventQueue() const;

public:
    virtual Result      GetThreadContext_core(nn::svc::ThreadContext* pContext, KThread* pThread, const Bit32 controlFlags) = 0;
    virtual Result      SetThreadContext_core(const nn::svc::ThreadContext& context, KThread* pThread, const Bit32 controlFlags) = 0;

public:
    explicit            KDebugBase() {}
    virtual             ~KDebugBase(){}

    void                Initialize();
    bool                Is64Bit() const;

    Result              Attach(KProcess* pProcess);
    Result              BreakProcess();

    Result              ContinueDebug(const Bit32 flags, const Bit64* threadIds, size_t numThreads);
    Result              TerminateProcess();
    Result              PopEventQueue();
    Result              ReadMemory(void* buf, KProcessAddress addr, size_t size);
    Result              WriteMemory(const void* buf, KProcessAddress addr, size_t size);
    Result              QueryInfo(nn::svc::MemoryInfo* pBlock, nn::svc::PageInfo* pPage, KProcessAddress addr);
    Result              GetThreadContext(nn::svc::ThreadContext* pContext, const Bit32 threadId, const Bit32 controlFlags);
    Result              SetThreadContext(const nn::svc::ThreadContext& context, Bit32 threadId, const Bit32 controlFlags);

    Result GetDebugEventInfo(nn::svc::DebugEventInfo* pDebugEventInfo);
#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
    Result GetDebugEventInfo(nn::svc::ilp32::DebugEventInfo* pDebugEventInfo);
#endif

    KProcess* GetProcess();

    // kern_SvcDebugEvent.cpp以外から呼び出すメソッド
    static Result       OnInitializeKernel();
    static Result       OnKernelInitialized();
    static Result       OnExitProcess(KProcess* pProcess);
    static Result       OnTerminateProcess(KProcess* pProcess);
    static Result       OnExitThread(KThread* pThread);
    static Result       OnTerminateThread(KThread* pThread);
    static Result       OnDebugEvent(const nn::svc::DebugEvent eventtype, const uintptr_t param1 = 0, const uintptr_t param2 = 0, const uintptr_t param3 = 0, const uintptr_t param4 = 0, const uintptr_t param5 = 0);
    static Result       OnDebugEvent(const nn::svc::DebugEvent eventtype, const nn::svc::DebugException exception_type, const uintptr_t param2 = 0, const uintptr_t param3 = 0, const uintptr_t param4 = 0, const uintptr_t param5 = 0);
    static KEventInfo*  CreateDebugEvent(const nn::svc::DebugEvent eventtype, uintptr_t param1, uintptr_t param2, uintptr_t param3, uintptr_t param4, uintptr_t param5, Bit64 threadid);

    // override KSynchronizationObject
    virtual bool IsSignaled() const;
};

}}

