﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_BitTypes.h>
#include "kern_CPUSelect.h"

namespace nn { namespace kern {

    class KThread;
    class KProcess;
    class KScheduler;
    class KInterruptTaskManager;

    /*!
        @brief: コア固有領域の情報

    */
    struct CurrentSet
    {
        KThread*                pCurrentThread;        //!< カレントスレッド
        KProcess*               pCurrentProcess;       //!< カレントプロセス
        KScheduler*             pScheduler;            //!< スケジューラ
        KInterruptTaskManager*  pInterruptTaskManager; //!< 割り込みタスクマネージャ
        int32_t                 coreNo;
        void*                   exceptionStackEnd;
    };

    namespace detail
    {
        inline CurrentSet& GetCurrentSet()
        {
            return *reinterpret_cast<CurrentSet*>(KCPU::GetCurrentValue());
        }
    }

    /*!
        @brief     カレント スレッドの参照を取得します。
    */
    inline KThread& GetCurrentThread()
    {
        return *detail::GetCurrentSet().pCurrentThread;
    }

    inline void SetCurrentThread(KThread* pThread)
    {
        detail::GetCurrentSet().pCurrentThread = pThread;
    }

    /*!
        @brief     カレント プロセスの参照を取得します。
    */
    inline KProcess& GetCurrentProcess()
    {
        return *detail::GetCurrentSet().pCurrentProcess;
    }

    /*!
        @brief     カレント プロセスへのポインタを取得します。
    */
    inline KProcess* GetCurrentProcessPointer()
    {
        return detail::GetCurrentSet().pCurrentProcess;
    }

    inline void SetCurrentProcess(KProcess* pProcess)
    {
        detail::GetCurrentSet().pCurrentProcess = pProcess;
    }

    /*!
        @brief     スケジューラの参照を取得します。
    */
    inline KScheduler& GetCoreScheduler()
    {
        return *detail::GetCurrentSet().pScheduler;
    }

    /*!
        @brief     割り込みタスクマネージャの参照を取得します。
    */
    inline KInterruptTaskManager& GetCoreInterruptTaskManager()
    {
        return *detail::GetCurrentSet().pInterruptTaskManager;
    }

    inline int GetCurrentCpuNo()
    {
        return detail::GetCurrentSet().coreNo;
    }
}}

