﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_MemoryMap.h"
#include "kern_KPageTable.h"

namespace nn { namespace kern { namespace ARMv8A {

class KProcessPageTable
{
public:
    Bit32 GetAllocateOption()  const { return m_Table.GetAllocateOption(); }
    Result Initialize(Bit32 spaceId, nn::svc::CreateProcessParameterFlag addressSapceType, bool enableAslr, bool fromBack, KMemoryManager::Region region, KProcessAddress codeRegionAddr, size_t codeRegionSize, KMemoryBlockResourceManager* pMemoryBlockManager, KBlockInfoManager* pBlockInfoManager, KPageTableManager* pPageTableManager) { return m_Table.Initialize(spaceId, addressSapceType, enableAslr, fromBack, region, codeRegionAddr, codeRegionSize, pMemoryBlockManager, pBlockInfoManager, pPageTableManager); }
    void Finalize()  { m_Table.Finalize(); }

    KBlockInfoManager* GetBlockInfoManager() const
    {
        return m_Table.GetBlockInfoManager();
    }
    bool GetPhysicalAddress(KPhysicalAddress* pOut, KProcessAddress addr) const
        { return m_Table.GetPhysicalAddress(pOut, addr); }

    Result MapPageGroup(KProcessAddress addr, const KPageGroup& pg, KMemoryState state, KMemoryPermission permission)
    {
        return m_Table.MapPageGroup(addr, pg, state, permission);
    }

    Result UnmapPageGroup(KProcessAddress addr, const KPageGroup& pg, KMemoryState state)
    {
        return m_Table.UnmapPageGroup(addr, pg, state);
    }
    Result CopyMemoryFromUser(KProcessAddress toAddr, size_t size,
            Bit32 toStateMask, Bit32 toState, KMemoryPermission toTestPermission,
            Bit32 toAttributeMask, Bit32 toAttribute,
            KProcessAddress fromAddr)
    {
        return m_Table.CopyMemoryFromUser(toAddr, size,
                toStateMask, toState, toTestPermission,
                toAttributeMask, toAttribute,
                fromAddr);
    }
    Result CopyMemoryToUser(KProcessAddress toAddr, size_t size,
            KProcessAddress fromAddr,
            Bit32 fromStateMask, Bit32 fromState, KMemoryPermission fromTestPermission,
            Bit32 fromAttributeMask, Bit32 fromAttribute)
    {
        return m_Table.CopyMemoryToUser(toAddr, size,
                fromAddr,
                fromStateMask, fromState, fromTestPermission,
                fromAttributeMask, fromAttribute);
    }
    Result CopyMemoryFromKernel(KProcessAddress toAddr, size_t size,
            Bit32 toStateMask, Bit32 toState, KMemoryPermission toTestPermission,
            Bit32 toAttributeMask, Bit32 toAttribute,
            KProcessAddress fromAddr)
    {
        return m_Table.CopyMemoryFromKernel(toAddr, size,
                toStateMask, toState, toTestPermission,
                toAttributeMask, toAttribute,
                fromAddr);
    }
    Result CopyMemoryToKernel(KProcessAddress toAddr, size_t size,
            KProcessAddress fromAddr,
            Bit32 fromStateMask, Bit32 fromState, KMemoryPermission fromTestPermission,
            Bit32 fromAttributeMask, Bit32 fromAttribute)
    {
        return m_Table.CopyMemoryToKernel(toAddr, size,
                fromAddr,
                fromStateMask, fromState, fromTestPermission,
                fromAttributeMask, fromAttribute);
    }

    Result SetupForIpc(KProcessAddress* pToAddr, size_t size,
            KProcessAddress fromAddr, KProcessPageTable* pFromTable,
            KMemoryPermission testPermission, KMemoryState toState,
            bool isSend)
    {
        return m_Table.SetupForIpc(pToAddr, size, fromAddr, &pFromTable->GetPageTable(), testPermission, toState, isSend);
    }
    Result CleanupForIpcServer(KProcessAddress addr, size_t size, KMemoryState toState)
    {
        return m_Table.CleanupForIpcServer(addr, size, toState);
    }
    Result CleanupForIpcClient(KProcessAddress addr, size_t size, KMemoryState toState)
    {
        return m_Table.CleanupForIpcClient(addr, size, toState);
    }

    void Activate(Bit64 id)
        { return KPageTable::Activate(&m_Table, id); }
    Result QueryInfo(KMemoryInfo* pBlock, nn::svc::PageInfo* pPage, KProcessAddress addr) const
        { return m_Table.QueryInfo(pBlock, pPage, addr); }
    Result MakePageGroupAndOpen(KPageGroup* pOut, KProcessAddress addr, size_t numPages, Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute) const
        { return m_Table.MakePageGroupAndOpen(pOut, addr, numPages, stateMask, state, permissionMask, permission, attributeMask, attribute); }
    Result MakePageGroupContiguousAndOpen(KPageGroup* pOut, KProcessAddress addr, size_t numPages, Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute) const
        { return m_Table.MakePageGroupContiguousAndOpen(pOut, addr, numPages, stateMask, state, permissionMask, permission, attributeMask, attribute); }

    // for HardwareBreak
    uint8_t GetAsid() { return m_Table.GetAsid(); }

    KPageTable& GetPageTable() { return m_Table; }

    Result SetHeapSize(KProcessAddress* pOut, size_t size)
    {
        return m_Table.SetHeapSize(pOut, size);
    }
    Result SetMaxHeapSize(size_t size)
    {
        return m_Table.SetMaxHeapSize(size);
    }
    Result SetMemoryPermission(KProcessAddress addr, size_t size, nn::svc::MemoryPermission permission)
    {
        return m_Table.SetMemoryPermission(addr, size, permission);
    }
    Result SetProcessMemoryPermission(KProcessAddress addr, size_t size, nn::svc::MemoryPermission permission)
    {
        return m_Table.SetProcessMemoryPermission(addr, size, permission);
    }
    Result MapMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size)
    {
        return m_Table.MapMemory(toAddr, fromAddr, size);
    }
    Result UnmapMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size)
    {
        return m_Table.UnmapMemory(toAddr, fromAddr, size);
    }
    Result MapCodeMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size)
    {
        return m_Table.MapCodeMemory(toAddr, fromAddr, size);
    }
    Result UnmapCodeMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size)
    {
        return m_Table.UnmapCodeMemory(toAddr, fromAddr, size);
    }
    Result QueryPhysicalAddress(nn::svc::PhysicalMemoryInfo* pBlockInfo, KProcessAddress addr)
    {
        return m_Table.QueryPhysicalAddress(pBlockInfo, addr);
    }
    Result QueryIoMapping(KProcessAddress* pAddr, KPhysicalAddress ioAddr, size_t ioSize)
    {
        return m_Table.QueryIoMapping(pAddr, ioAddr, ioSize);
    }
    Result MapIo(KPhysicalAddress physAddr, size_t size, KMemoryPermission userPermission)
    {
        return m_Table.MapIo(physAddr, size, userPermission);
    }
    Result MapStatic(KPhysicalAddress physAddr, size_t size, KMemoryPermission userPermission)
    {
        return m_Table.MapStatic(physAddr, size, userPermission);
    }
    Result SetMemoryAttribute(KProcessAddress addr, size_t size, Bit32 mask, Bit32 attribute)
    {
        return m_Table.SetMemoryAttribute(addr, size, mask, attribute);
    }
    Result InvalidateProcessDataCache(KProcessAddress addr, size_t size)
    {
        return m_Table.InvalidateProcessDataCache(addr, size);
    }
    Result ReadDebugMemory(void* buf, KProcessAddress addr, size_t size)
    {
        return m_Table.ReadDebugMemory(buf, addr, size);
    }
    Result WriteDebugMemory(KProcessAddress addr, const void* buf, size_t size)
    {
        return m_Table.WriteDebugMemory(addr, buf, size);
    }

    Result LockForDeviceAddressSpace(KPageGroup* pOut, KProcessAddress addr, size_t size, KMemoryPermission permission, bool aligned)
    {
        return m_Table.LockForDeviceAddressSpace(pOut, addr, size, permission, aligned);
    }
    Result UnlockForDeviceAddressSpace(KProcessAddress addr, size_t size)
    {
        return m_Table.UnlockForDeviceAddressSpace(addr, size);
    }
    Result LockForTransferMemory(KPageGroup* pOut, KProcessAddress addr, size_t size, KMemoryPermission permission)
    {
        return m_Table.LockForTransferMemory(pOut, addr, size, permission);
    }
    Result UnlockForTransferMemory(KProcessAddress addr, size_t size, const KPageGroup& pg)
    {
        return m_Table.UnlockForTransferMemory(addr, size, pg);
    }
    Result LockForIpcUserBuffer(KProcessAddress addr, size_t size)
    {
        return m_Table.LockForIpcUserBuffer(addr, size);
    }
    Result UnlockForIpcUserBuffer(KProcessAddress addr, size_t size)
    {
        return m_Table.UnlockForIpcUserBuffer(addr, size);
    }
    Result LockForCodeMemory(KPageGroup* pOut, KProcessAddress addr, size_t size)
    {
        return m_Table.LockForCodeMemory(pOut, addr, size);
    }
    Result UnlockForCodeMemory(KProcessAddress addr, size_t size, const KPageGroup& pg)
    {
        return m_Table.UnlockForCodeMemory(addr, size, pg);
    }

    Result  MapPages(
                KProcessAddress*    pAddr,
                size_t              numPages,
                KMemoryState        state,
                KMemoryPermission   permission)
    {
        return m_Table.MapPages(pAddr, numPages, state, permission);
    }

    Result  MapPages(
                KProcessAddress*    pAddr,
                size_t              numPages,
                size_t              align,
                KPhysicalAddress    physAddr,
                KMemoryState        state,
                KMemoryPermission   permission)
    {
        return m_Table.MapPages(pAddr, numPages, align, physAddr, state, permission);
    }

    Result  MapPages(
                KProcessAddress     addr,
                size_t              numPages,
                KMemoryState        state,
                KMemoryPermission   permission)
    {
        return m_Table.MapPages(addr, numPages, state, permission);
    }

    Result  MapPhysicalMemory(
                KProcessAddress     addr,
                size_t              size)
    {
        return m_Table.MapPhysicalMemory(addr, size);
    }

    Result  UnmapPhysicalMemory(
                KProcessAddress     addr,
                size_t              size)
    {
        return m_Table.UnmapPhysicalMemory(addr, size);
    }

    Result  MapPhysicalMemoryUnsafe(
                KProcessAddress     addr,
                size_t              size)
    {
        return m_Table.MapPhysicalMemoryUnsafe(addr, size);
    }

    Result  UnmapPhysicalMemoryUnsafe(
                KProcessAddress     addr,
                size_t              size)
    {
        return m_Table.UnmapPhysicalMemoryUnsafe(addr, size);
    }

    Result  UnmapPages(
            KProcessAddress     addr,
            size_t              numPages,
            KMemoryState        state)
    {
        return m_Table.UnmapPages(addr, numPages, state);
    }

    Result StackFillPattern(KProcessAddress stackEnd) const
    {
        return m_Table.StackFillPattern(stackEnd);
    }

    Result GetStackUsage(size_t* pUsageSize, size_t* pStackSize, KProcessAddress stackAddr) const
    {
        return m_Table.GetStackUsage(pUsageSize, pStackSize, stackAddr);
    }

    void Dump() const { return m_Table.DumpMemoryBlocks(); }
    void DumpPageTable() const { return m_Table.DumpPageTable(); }
    size_t CountPageTables() const { return m_Table.CountPageTables(); }


    KProcessAddress GetReservedRegionBegin() const { return m_Table.GetReservedRegionBegin(); }
    size_t GetReservedRegionSize()  const { return m_Table.GetReservedRegionSize(); }

    KProcessAddress GetHeapRegionBegin() const { return m_Table.GetHeapRegionBegin(); }
    size_t GetHeapRegionSize()  const { return m_Table.GetHeapRegionSize(); }

    KProcessAddress GetAslrRegionBegin() const { return m_Table.GetAslrRegionBegin(); }
    size_t GetAslrRegionSize()  const { return m_Table.GetAslrRegionSize(); }

    KProcessAddress GetStackRegionBegin() const { return m_Table.GetStackRegionBegin(); }
    size_t GetStackRegionSize()  const { return m_Table.GetStackRegionSize(); }

    KProcessAddress GetMapRegionBegin(KMemoryState state) const { return m_Table.GetMapRegionBegin(state); }
    size_t GetMapRegionSize(KMemoryState state) const { return m_Table.GetMapRegionSize(state); }

    size_t GetNormalMemorySize() const { return m_Table.GetNormalMemorySize(); }
    size_t GetCodeSize()       const { return m_Table.GetCodeSize(); }
    size_t GetCodeDataSize()   const { return m_Table.GetCodeDataSize(); }


    bool IsInRange(KProcessAddress addr, size_t size, KMemoryState state) const { return m_Table.IsInRange(addr, size, state); }
    bool IsInRange(KProcessAddress addr, size_t size) const { return m_Table.IsInRange(addr, size); }
    bool IsInReservedRegion(KProcessAddress addr, size_t size) const { return m_Table.IsInReservedRegion(addr, size); }
    bool IsInAslrRegion(KProcessAddress addr, size_t size) const { return m_Table.IsInAslrRegion(addr, size); }

    void Info() { return m_Table.Info(); }
private:
    KPageTable  m_Table;
};

}}} // namespace

