﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>
#include "../../kern_Platform.h"
#include "kern_KPageTable.h"
#include "kern_MemoryMap.h"
#include "../../kern_InterruptManagerSelect.h"
#include "../../kern_InterruptNameSelect.h"
#include "../../kern_Kernel.h"
#include "../../kern_KProcess.h"
#include "../../kern_KPageTableManager.h"
#include "../../kern_KPageBuffer.h"
#include "../../kern_Utility.h"
#include <cstring>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace kern {

namespace ARMv8A{

namespace
{
    const Bit64 UNMAPPED_ENTRY = 0;  // ALL 0

    Bit64 MakeTtbrValue(KPhysicalAddress tableAddresss, uint16_t asid)
    {
        const uintptr_t tableAddr = GetAsInteger(tableAddresss);
        return (static_cast<Bit64>(asid) << 48) | tableAddr;
    }

    inline Bit32 Rightmost0(Bit32 x) { return ~x & (x + 1); }

    class AsidManager
    {
    public:
        static const uint8_t NULL_ASID = 0;

    private:
        static const int NUM_IDS    = 256;
        static const int NUM_WORDS  = NUM_IDS / 32;

    private:
        Bit32       m_AsidMap[NUM_WORDS];
        KLightMutex m_Lock;
        uint8_t          m_Index;

    public:
        AsidManager()
        {
            std::memset(m_AsidMap, 0, sizeof(m_AsidMap));
            m_Index = 0;

            // NULL_ASID を予約しておく
            SetBit(m_AsidMap, NULL_ASID);
        }

        uint8_t Acquire()
        {
            Bit32 id;

            KScopedLightLock lock(&m_Lock);

            if( GetBit(m_AsidMap, m_Index) )
            {
                id = FindClear(m_AsidMap);
                m_Index = id;
            }
            else
            {
                id = m_Index;
            }

            SetBit(m_AsidMap, id);
            m_Index++;              // 8 bit で回り込む

            return static_cast<uint8_t>(id);
        }

        void Release(uint8_t id)
        {
            NN_KERN_ASSERT( GetBit(m_AsidMap, id) );
            KScopedLightLock lock(&m_Lock);
            ClearBit(m_AsidMap, id);
        }

    private:
        static void SetBit(Bit32* pBits, uint32_t pos)
        {
            const int wordIndex = pos / 32;
            const int bitIndex  = pos % 32;
            pBits[wordIndex] |= (1u << bitIndex);
        }
        static void ClearBit(Bit32* pBits, uint32_t pos)
        {
            const int wordIndex = pos / 32;
            const int bitIndex  = pos % 32;
            pBits[wordIndex] &= ~(1u << bitIndex);
        }
        static bool GetBit(const Bit32* pBits, uint32_t pos)
        {
            const int wordIndex = pos / 32;
            const int bitIndex  = pos % 32;
            return (pBits[wordIndex] >> bitIndex) & 0x1;
        }
        static uint8_t FindClear(const Bit32* pBits)
        {
            for( int i = 0; i < NUM_WORDS; ++i )
            {
                uint8_t x = __builtin_clz(Rightmost0(pBits[i]));
                if( x < 32 )
                {
                    return 32 * i + (31 - x);
                }
            }

            NN_KERNEL_PANIC("too many asid");
            return 0;
        }
    };

    AsidManager s_AsidManager;

    class AlignedMemoryBlock
    {
    public:
        static size_t NextAlign(size_t align)
        {
            return KPageTable::GetPageSizeFromSzc(KPageTable::GetSzcFromPageSize(align) - 1);
        }
        static size_t NextLargeAlign(size_t align)
        {
            return KPageTable::GetPageSizeFromSzc(KPageTable::GetSzcFromPageSize(align) + 1);
        }

        void Initialize(uintptr_t start, size_t numPages, size_t align)
        {
            NN_KERN_ASSERT((start & (KPageTable::PageSize - 1)) == 0);
            NN_KERN_ASSERT(numPages > 0);

            uintptr_t startPfn = start / KPageTable::PageSize;
            align /= KPageTable::PageSize;
            while (RoundUp(startPfn, align) >= RoundDown(startPfn + numPages, align))
            {
                align = NextAlign(align * KPageTable::PageSize) / KPageTable::PageSize;
            }
            m_LoAreaStart = startPfn;
            m_LoAreaEnd = m_HiAreaStart = RoundUp(startPfn, align);
            m_HiAreaEnd = startPfn + numPages;
            m_CurrentAlign = align;
            NN_KERN_ASSERT(m_CurrentAlign >= 1);
        }

        // GetAlign()でアラインされたメモリを獲得する。
        //   numPages が 0で呼び出されたとき、最大サイズのメモリを獲得する。
        //   0以外のときは、サイズがnumPages以下のメモリを獲得する。
        //
        void FindBlock(uintptr_t* pStart, size_t* pNumPages)
        {
            if ((m_HiAreaEnd - m_HiAreaStart) >= m_CurrentAlign)
            {
                *pStart = m_HiAreaStart * KPageTable::PageSize;
                if ((RoundDown(m_HiAreaEnd, m_CurrentAlign) - m_HiAreaStart) < *pNumPages || *pNumPages == 0)
                {
                    *pNumPages = RoundDown(m_HiAreaEnd, m_CurrentAlign) - m_HiAreaStart;
                }
                m_HiAreaStart += *pNumPages;
            }
            else if ((m_LoAreaEnd - m_LoAreaStart) >= m_CurrentAlign)
            {
                if ((m_LoAreaEnd - RoundUp(m_LoAreaStart, m_CurrentAlign)) < *pNumPages || *pNumPages == 0)
                {
                    *pNumPages = m_LoAreaEnd - RoundUp(m_LoAreaStart, m_CurrentAlign);
                }
                m_LoAreaEnd -= *pNumPages;
                *pStart = m_LoAreaEnd * KPageTable::PageSize;
            }
            else
            {
                *pStart = 0;
                *pNumPages = 0;
            }
        }
        void SetAlign(size_t align)
        {
            NN_KERN_ASSERT(m_CurrentAlign >= align / KPageTable::PageSize);
            m_CurrentAlign = align / KPageTable::PageSize;
        }
        size_t GetAlign()
        {
            return m_CurrentAlign * KPageTable::PageSize;
        }
    private:
        uintptr_t m_LoAreaStart;
        uintptr_t m_LoAreaEnd;
        uintptr_t m_HiAreaStart;
        uintptr_t m_HiAreaEnd;
        size_t m_CurrentAlign;
    };


    void ClearPageTable(KVirtualAddress addr)
    {
        for (size_t offset = 0; offset < KPageTable::PageSize; offset += KCPU::DATA_CACHE_LINE_SIZE)
        {
            uint64_t vaddr = (GetAsInteger(addr) + offset);
            asm volatile ("DC ZVA, %0"::"r"(vaddr):"memory");
        }
    }
}


/* =======================================================================
        初期化・終了処理
   ======================================================================== */

void
KPageTable::Initialize(int32_t coreNo)
{
    NN_UNUSED(coreNo);
}

Result
KPageTable::Initialize(Bit32 spaceId, nn::svc::CreateProcessParameterFlag addressSapceType, bool enableAslr, bool fromBack, KMemoryManager::Region region, KProcessAddress codeRegionAddr, size_t codeRegionSize, KMemoryBlockResourceManager* pMemoryBlockManager, KBlockInfoManager* pBlockInfoManager, KPageTableManager* pPageTableManager)
{
    NN_UNUSED(spaceId);
    int addressSpaceSize;
    int t0shift;
    switch (addressSapceType)
    {
    case nn::svc::CreateProcessParameterFlag_AddressSpace64Bit:
        addressSpaceSize = 39;
        t0shift = NN_KERN_T0_39BIT_SPACE_SHIFT;
        break;
    case nn::svc::CreateProcessParameterFlag_AddressSpace64BitOld:
        addressSpaceSize = 36;
        t0shift = NN_KERN_T0_36BIT_SPACE_SHIFT;
        break;
    case nn::svc::CreateProcessParameterFlag_AddressSpace32Bit:
    case nn::svc::CreateProcessParameterFlag_AddressSpace32BitNoReserved:
        addressSpaceSize = 32;
        t0shift = NN_KERN_T0_32BIT_SPACE_SHIFT;
        break;
    default:
        NN_KERN_ABORT();
        break;
    }

    m_Asid              = s_AsidManager.Acquire();
    const KProcessAddress addressBegin = 0;
    const KProcessAddress addressEnd   = (1ull << addressSpaceSize);
    m_pPageTableManager = pPageTableManager;

    // ページテーブル用メモリの確保、ポインタの設定
    const KVirtualAddress tableVAddress = GetPageTableManager().Allocate();
    Bit64* pTable = GetTypedPointer<Bit64>(tableVAddress);

    if(pTable == 0)
    {
        NN_WARNING(false, "Failed to allocate a process page table.");
        s_AsidManager.Release(m_Asid);
        return nn::svc::ResultOutOfMemory();
    }

    m_TtbrValue = MakeTtbrValue(GetPageTablePhysicalAddress(tableVAddress), m_Asid);
    Bit64 tcr = KCPU::GetCurrentTcr();
    tcr &= ~(HW_TCR_EPD0 | HW_TCR_T0SZ_MASK);
    tcr |= HW_TCR_T0SZ(64 - t0shift);
    m_TcrValue  = tcr;


    Result result = KPageTableBase::InitializeForProcess(addressSapceType, enableAslr, fromBack, pTable, addressBegin, addressEnd, region, codeRegionAddr, codeRegionSize, pMemoryBlockManager, pBlockInfoManager);
    if (result.IsFailure())
    {
        GetPageTableManager().Free(reinterpret_cast<uintptr_t>(pTable));
        s_AsidManager.Release(m_Asid);
        return result;
    }

    MarkUpdated();

    return ResultSuccess();
}

Result KPageTable::InitializeForKernel(void* pTable, KProcessAddress begin, KProcessAddress end)
{
    m_Asid              = 0;
    m_TtbrValue         = 0;
    m_TcrValue          = KCPU::GetCurrentTcr() | HW_TCR_EPD0;
    m_pPageTableManager = &Kernel::GetPageTableManager();

    Result result = KPageTableBase::InitializeForKernel(true, pTable, begin, end);
    if (result.IsFailure())
    {
        NN_KERNEL_PANIC("failed to initialize page table");
    }

    return ResultSuccess();
}

Result KPageTable::Finalize()
{
    NN_KERN_ASSERT( ! IsKernel() );
    KPageTableBody& table = GetPageTable();

    // 全コアで ASID を無効化
    MarkUpdated();

    {
        KMemoryManager& mm = Kernel::GetKernelHeapManager();

        // 全てのページを解放
        {
            TraverseContext     context;
            TraverseData        current;
            bool                currentIsValid;
            TraverseData        next;
            bool                nextIsValid;
            const size_t        spaceSize = GetAddressEnd() - GetAddressBegin();
            size_t              sizeSum  = 0;

            current.address = 0;
            current.size    = 0;
            currentIsValid  = false;

            nextIsValid = table.TraverseBegin(&next, &context, GetAddressBegin());

            // ページを解放
            for(;;)
            {
                if( (nextIsValid && currentIsValid && next.address == current.address + current.size)
                 || ( !currentIsValid && !nextIsValid ) )
                {
                    current.size += next.size;
                }
                else
                {
                    if (currentIsValid)
                    {
                        if (IsHeapPhysicalAddress(current.address))
                        {
                            const KVirtualAddress  va = GetHeapVirtualAddress(current.address);
                            mm.Close( va, current.size / PageSize);
                        }
                        else if (KMemoryLayout::InSlabRegionPhysical(current.address))
                        {
                            // KPageBuffer が プロセスにマップされるとき、エイリアス不可な領域(現在、ThreadLocalのみ)
                            // MemoryBlock枯渇を救済する
                            KProcessAddress slabBase = KPageBuffer::GetSlabAddr();
                            KProcessAddress slabEnd = slabBase + KPageBuffer::GetSlabSize() * sizeof(KPageBuffer);
                            KProcessAddress va = KMemoryLayout::GetSlabRegionBegin() + (current.address - KMemoryLayout::GetSlabRegionPhysicalBegin());
                            if ((slabBase <= va) && ((va + current.size - 1) <= (slabEnd - 1)))
                            {
                                for (size_t i = 0; i < current.size / sizeof(KPageBuffer); i++)
                                {
                                    KPageBuffer::Free(GetTypedPointer<KPageBuffer>(va + i * sizeof(KPageBuffer)));
                                }
                            }
                        }
                    }

                    sizeSum += current.size;
                    current = next;
                    currentIsValid = nextIsValid;
                }

                if( sizeSum + current.size >= spaceSize )
                {
                    break;
                }

                nextIsValid = table.TraverseNext(&next, &context);
            }

            if (currentIsValid)
            {
                if (IsHeapPhysicalAddress(current.address))
                {
                    const KVirtualAddress  va = GetHeapVirtualAddress(current.address);
                    mm.Close( va, current.size / PageSize);
                }
                else if (KMemoryLayout::InSlabRegionPhysical(current.address))
                {
                    // KPageBuffer が プロセスにマップされるとき、エイリアス不可な領域(現在、ThreadLocalのみ)
                    // MemoryBlock枯渇を救済する
                    KProcessAddress slabBase = KPageBuffer::GetSlabAddr();
                    KProcessAddress slabEnd = slabBase + KPageBuffer::GetSlabSize() * sizeof(KPageBuffer);
                    KProcessAddress va = KMemoryLayout::GetSlabRegionBegin() + (current.address - KMemoryLayout::GetSlabRegionPhysicalBegin());
                    if ((slabBase <= va) && ((va + current.size - 1) <= (slabEnd - 1)))
                    {
                        for (size_t i = 0; i < current.size / sizeof(KPageBuffer); i++)
                        {
                            KPageBuffer::Free(GetTypedPointer<KPageBuffer>(va + i * sizeof(KPageBuffer)));
                        }
                    }
                }
            }
        }

        // 全ての L3 ページテーブルを解放
        // TODO: ループを最適化できる
        {
            for (KProcessAddress vaddr = GetAddressBegin(); vaddr <= GetAddressEnd() - 1; vaddr += L2BlockSize)
            {
                Bit64 l1Entry = table.GetL1Entry(vaddr);
                KPhysicalAddress l2TablePAddress;
                if (table.GetTableAddressFromEntry(&l2TablePAddress, l1Entry))
                {
                    KVirtualAddress l2TableVAddress = GetPageTableVirtualAddress(l2TablePAddress);
                    Bit64* pL2Table = GetTypedPointer<Bit64>(l2TableVAddress);
                    Bit64 l2Entry = table.GetL2Entry(vaddr, pL2Table);
                    KPhysicalAddress l3TablePAddress;
                    if (table.GetTableAddressFromEntry(&l3TablePAddress, l2Entry))
                    {
                        KVirtualAddress l3TableVAddress = GetPageTableVirtualAddress(l3TablePAddress);
                        if (GetPageTableManager().InPageTableHeap(l3TableVAddress))
                        {
                            while (!GetPageTableManager().Close(l3TableVAddress, 1)) {}
                            GetPageTableManager().Free(l3TableVAddress);
                        }
                    }
                }
            }
        }

        // 全ての L2 ページテーブルを解放
        // TODO: ループを最適化できる(が、そんなに効果ない)
        {
            for (KProcessAddress vaddr = GetAddressBegin(); vaddr <= GetAddressEnd() - 1; vaddr += L1BlockSize)
            {
                Bit64 l1Entry = table.GetL1Entry(vaddr);
                KPhysicalAddress l2TablePAddress;
                if (table.GetTableAddressFromEntry(&l2TablePAddress, l1Entry))
                {
                    KVirtualAddress l2TableVAddress = GetPageTableVirtualAddress(l2TablePAddress);
                    if (GetPageTableManager().InPageTableHeap(l2TableVAddress))
                    {
                        while (!GetPageTableManager().Close(l2TableVAddress, 1)) {}
                        GetPageTableManager().Free(l2TableVAddress);
                    }
                }
            }
        }

        // L1 ページテーブルに割り当てたメモリを解放
        {
            Bit64* pTable = table.Finalize();
            GetPageTableManager().Free(reinterpret_cast<uintptr_t>(pTable));
        }

        KPageTableBase::Finalize();
    }

    // ASID を返却
    s_AsidManager.Release(m_Asid);

    return ResultSuccess();
}   // NOLINT(readability/fn_size)

bool KPageTable::GetStaticMap(KProcessAddress* pOut, KPhysicalAddress paddr, size_t size)
{
    NN_UNUSED(pOut);
    NN_UNUSED(paddr);
    NN_UNUSED(size);
    return false;
}

KVirtualAddress KPageTable::AllocatePageTable(void** ppData)
{
    KVirtualAddress pTable = GetPageTableManager().Allocate();

    if (pTable == Null<KVirtualAddress>())
    {
        if (*ppData)
        {
            void* pNext = *static_cast<void**>(*ppData);
            pTable = KVirtualAddress(*ppData);
            *ppData = pNext;
        }
        else
        {
            return Null<KVirtualAddress>();
        }
    }

    ClearPageTable(pTable);

    NN_KERN_ASSERT(GetPageTableManager().GetRefCount(pTable) == 0);

    return pTable;
}

void KPageTable::FreePageTable(void** ppData, KVirtualAddress pTable)
{
    NN_KERN_ASSERT(GetPageTableManager().InPageTableHeap(pTable));
    NN_KERN_ASSERT(GetPageTableManager().GetRefCount(pTable) == 0);
    void* pNext = *ppData;
    *GetTypedPointer<void*>(pTable) = pNext;
    *ppData = GetUntypedPointer(pTable);
}

void KPageTable::PageTableUpdaterFinalize(void** ppData)
{
    while (*ppData)
    {
        KVirtualAddress pTable = KVirtualAddress(*ppData);
        *ppData = *static_cast<void**>(*ppData);
        NN_KERN_ASSERT(GetPageTableManager().InPageTableHeap(pTable));
        NN_KERN_ASSERT(GetPageTableManager().GetRefCount(pTable) == 0);
        GetPageTableManager().Free(pTable);
    }
}

Result
KPageTable::OperateImpl(void** ppData,
        KProcessAddress vaddr, size_t numPages, const KPageGroup& pg,
        const KPageProperty property, PageTableOperation operation)
{
    Result result;
    NN_KERN_ASSERT(IsLockedByMe());
    NN_KERNEL_PAGE_ALIGN_ASSERT(GetAsInteger(vaddr));
    NN_KERN_ASSERT(numPages != 0);
    NN_KERN_ASSERT(numPages == pg.GetTotalNumPages());

    Bit64 entryTemplate = GetEntryTemplate(property);
    switch (operation)
    {
    case PageTableOperation_MapRelocate:
        {
            result = MapAdaptiveWithTemplate(vaddr, pg, numPages, entryTemplate, ppData);
            if (result.IsFailure())
            {
                NN_WARNING(false, "failed to map pages.");
                return result;
            }
        }
        break;

    default:
        {
            NN_KERN_ABORT();
        }
        break;
    }

    return ResultSuccess();
}

Result
KPageTable::OperateImpl(
            void**              ppData,
            KProcessAddress     vaddr,
            size_t              numPages,
            KPhysicalAddress    paddr,
            bool                paIsValid,
            const KPageProperty property,
            PageTableOperation  operation)
{
    Result result;
    NN_KERN_ASSERT(IsLockedByMe());
    NN_KERNEL_PAGE_ALIGN_ASSERT(GetAsInteger(vaddr));
    NN_KERN_ASSERT(numPages != 0);
    NN_KERN_ASSERT(IsPagesInRange(vaddr, numPages));

    if (operation == PageTableOperation_Map)
    {
        NN_KERN_ABORT_UNLESS(paIsValid);
        NN_KERNEL_PAGE_ALIGN_ASSERT(GetAsInteger(paddr));
    }
    else
    {
        NN_KERN_ABORT_UNLESS(!paIsValid);
    }

    if (operation != PageTableOperation_Unmap)
    {
        Bit64 entryTemplate = GetEntryTemplate(property);

        switch (operation)
        {
        case PageTableOperation_AllocateAndMap:
            {
                KScopedPageGroup pg(GetBlockInfoManager());
                KMemoryManager& mm = Kernel::GetKernelHeapManager();
                result = mm.Allocate(&pg.GetPageGroup(), numPages, GetAllocateOption());
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to allocate memory, numPages = %d", numPages);
                    return result;
                }

                for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter(); it != pg.GetPageGroup().GetBlockEndIter(); it++)
                {
                    ::std::memset(GetUntypedPointer(it->GetBlockAddr()), m_AllocateFillMemoryPattern, it->GetNumPages() * PageSize);
                }

                result = MapAdaptiveWithTemplate(vaddr, pg.GetPageGroup(), numPages, entryTemplate, ppData);
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to map pages.");
                    return result;
                }
            }
            break;
        case PageTableOperation_Map:
            {
                result = MapAdaptiveContinuousWithTemplate(vaddr, paddr, numPages, entryTemplate, ppData);
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to map pages.");
                    return result;
                }
            }
            break;
        case PageTableOperation_ChangeProperties:
            {
                result = UpdateProperties(vaddr, numPages, entryTemplate, false, ppData, false);
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to change permission");
                    return result;
                }
            }
            break;
        case PageTableOperation_ChangePropertiesWithBreakBeforeMake:
            {
                result = UpdateProperties(vaddr, numPages, entryTemplate, true, ppData, false);
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to change page attribute");
                    return result;
                }
            }
            break;
        default:
            {
                NN_KERN_ABORT();
            }
            break;
        }
    }
    else
    {
        // Unmap
        result = FreePages(vaddr, numPages, ppData);
        if (result.IsFailure())
        {
            NN_WARNING(false, "failed to free pages");
            return result;
        }
    }

    return ResultSuccess();
}

/* =======================================================================
        インスタンスメンバ / private
   ======================================================================== */
// pageSize以下のページ、セクションにする。
Result KPageTable::SeparatePage(KProcessAddress vAddr, size_t pageSize, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    KPageTableBody& table = GetPageTable();
    Bit64 masterEntry = table.GetL1Entry(vAddr);
    KProcessAddress vAddrTop;
    KPhysicalAddress pAddrTop = 0;
    const Bit64 entryAttrMask = (0xFFF0000000000FFFull & ~(PteTypeMask | HW_MMU_CONTIGUOUS));

    switch (masterEntry & PteTypeMask)
    {
    case PteTypeBlock:
        {
            if (pageSize >= L1BlockSize)
            {
                return ResultSuccess();
            }
            // L1 Block -> Contiguous L2 Block
            vAddrTop = RoundDown(vAddr, L1BlockSize);
            table.GetBlockAddressFromEntry(&pAddrTop, masterEntry);

            KVirtualAddress pL2Table = AllocatePageTable(ppData);
            if (pL2Table == Null<KVirtualAddress>())
            {
                goto error_exit;
            }
            const Bit64 entryTemplate = (masterEntry & entryAttrMask);
            for (size_t i = 0; i < L1BlockSize / L2BlockSize; i++)
            {
                table.SetL2Entry(vAddrTop + L2BlockSize * i, HW_MMU_CONTIGUOUS | GetAsInteger(pAddrTop + L2BlockSize * i) | entryTemplate | PteTypeBlock, GetTypedPointer<Bit64>(pL2Table));
            }

            GetPageTableManager().Open(pL2Table, L1BlockSize / L2BlockSize);

            // L1ページテーブルの設定
            masterEntry = (IsKernel()? HW_MMU_APTABLE_U_NA | HW_MMU_UXNTABLE: 0) | HW_MMU_PXNTABLE | GetAsInteger(GetPageTablePhysicalAddress(pL2Table)) | PteTypeTable;
            PteSynchronizationBarrier();
            table.SetL1Entry(vAddrTop, masterEntry);
            MarkUpdated();
        }
        NN_FALL_THROUGH;
    case PteTypeTable:
        {
            if (pageSize >= ContiguousL2BlockSize)
            {
                return ResultSuccess();
            }

            KPhysicalAddress l2TablePAddress = 0;
            table.GetTableAddressFromEntry(&l2TablePAddress, masterEntry);
            KVirtualAddress pL2Table = GetPageTableVirtualAddress(l2TablePAddress);
            Bit64 l2Entry = table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(pL2Table));

            switch (l2Entry & PteTypeMask)
            {
            case PteTypeBlock:
                {
                    if (l2Entry & HW_MMU_CONTIGUOUS)
                    {
                        // Contiguous L2 Block -> L2 Block
                        vAddrTop = RoundDown(vAddr, ContiguousL2BlockSize);
                        table.GetBlockAddressFromEntry(&pAddrTop, l2Entry);
                        pAddrTop = RoundDown(pAddrTop, ContiguousL2BlockSize);

                        const Bit64 entryTemplate = (l2Entry & entryAttrMask);
                        for (size_t i = 0; i < ContiguousL2BlockSize / L2BlockSize; i++)
                        {
                            table.SetL2Entry(vAddrTop + L2BlockSize * i, GetAsInteger(pAddrTop + L2BlockSize * i) | entryTemplate | PteTypeBlock, GetTypedPointer<Bit64>(pL2Table));
                        }
                        MarkUpdated();
                    }
                    if (pageSize >= L2BlockSize)
                    {
                        return ResultSuccess();
                    }

                    // L2 Block -> ContiguousPage
                    vAddrTop = RoundDown(vAddr, L2BlockSize);
                    table.GetBlockAddressFromEntry(&pAddrTop, l2Entry);
                    KVirtualAddress pL3Table = AllocatePageTable(ppData);
                    if (pL3Table == Null<KVirtualAddress>())
                    {
                        goto error_exit;
                    }

                    const Bit64 entryTemplate = (l2Entry & entryAttrMask);

                    for (size_t i = 0; i < L2BlockSize / PageSize; i++)
                    {
                        table.SetL3Entry(vAddrTop + PageSize * i, HW_MMU_CONTIGUOUS | GetAsInteger(pAddrTop + PageSize * i) | entryTemplate | PteTypePage, GetTypedPointer<Bit64>(pL3Table));
                    }

                    GetPageTableManager().Open(pL3Table, L2BlockSize / PageSize);

                    // L2ページテーブルの設定
                    l2Entry = (IsKernel()? HW_MMU_APTABLE_U_NA | HW_MMU_UXNTABLE: 0) | HW_MMU_PXNTABLE | GetAsInteger(GetPageTablePhysicalAddress(pL3Table)) | PteTypeTable;
                    PteSynchronizationBarrier();
                    table.SetL2Entry(vAddrTop, l2Entry, GetTypedPointer<Bit64>(pL2Table));
                    MarkUpdated();
                }
                NN_FALL_THROUGH;
            case PteTypeTable:
                {
                    if (pageSize >= ContiguousPageSize)
                    {
                        return ResultSuccess();
                    }

                    KPhysicalAddress l3TablePAddress = 0;
                    table.GetTableAddressFromEntry(&l3TablePAddress, l2Entry);
                    KVirtualAddress pL3Table = GetPageTableVirtualAddress(l3TablePAddress);
                    Bit64 l3Entry = table.GetL3Entry(vAddr, GetTypedPointer<Bit64>(pL3Table));

                    if (l3Entry & HW_MMU_CONTIGUOUS)
                    {
                        // Contiguous Page -> Page
                        vAddrTop = RoundDown(vAddr, ContiguousPageSize);
                        table.GetPageAddressFromEntry(&pAddrTop, l3Entry);
                        pAddrTop = RoundDown(pAddrTop, ContiguousPageSize);

                        const Bit64 entryTemplate = (l3Entry & entryAttrMask);
                        for (size_t i = 0; i < ContiguousPageSize / PageSize; i++)
                        {
                            table.SetL3Entry(vAddrTop + PageSize * i, GetAsInteger(pAddrTop + PageSize * i) | entryTemplate | PteTypePage, GetTypedPointer<Bit64>(pL3Table));
                        }
                        MarkUpdated();
                    }
                }
                break;
            default:
                break;
            }
        }
        break;
    default:
        break;
    }
    return ResultSuccess();
error_exit:

    MergePage(vAddr, ppData);

    return  nn::svc::ResultOutOfResource();
}


// 可能なら、より大きいページ、セクションでマッピングする。
bool KPageTable::MergePage(KProcessAddress vAddr, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    bool merged = false;
    const Bit64 entryAttrMask = (0xFFF0000000000FFFull & ~(PteTypeMask | HW_MMU_CONTIGUOUS));
    KPageTableBody& table = GetPageTable();

    Bit64 masterEntry   = table.GetL1Entry(vAddr);

    switch (masterEntry & PteTypeMask)
    {
    case PteTypeTable:
        {
            KPhysicalAddress l2TablePAddress;
            table.GetTableAddressFromEntry(&l2TablePAddress, masterEntry);
            KVirtualAddress pL2Table = GetPageTableVirtualAddress(l2TablePAddress);
            Bit64 l2Entry = table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(pL2Table));

            switch (l2Entry & PteTypeMask)
            {
            case PteTypeTable:
                {
                    KPhysicalAddress l3TablePAddress;
                    table.GetTableAddressFromEntry(&l3TablePAddress, l2Entry);
                    KVirtualAddress pL3Table = GetPageTableVirtualAddress(l3TablePAddress);
                    Bit64 l3Entry = table.GetL3Entry(vAddr, GetTypedPointer<Bit64>(pL3Table));

                    switch (l3Entry & PteTypeMask)
                    {
                    case PteTypePage:
                        {
                            if ((l3Entry & HW_MMU_CONTIGUOUS) == 0)
                            {
                                vAddr = RoundDown(vAddr, ContiguousPageSize);
                                KPhysicalAddress pAddr;
                                table.GetTableAddressFromEntry(&pAddr, l3Entry);
                                const Bit64 entryAttr = (l3Entry & entryAttrMask);
                                pAddr = RoundDown(pAddr, ContiguousPageSize);

                                // Check
                                for (size_t i = 0; i < ContiguousPageSize / PageSize; i++)
                                {
                                    const Bit64 entry = table.GetL3Entry(vAddr + PageSize * i, GetTypedPointer<Bit64>(pL3Table));
                                    if (entry != (GetAsInteger(pAddr + PageSize * i) | entryAttr | PteTypePage))
                                    {
                                        return merged;
                                    }
                                }

                                // Upgrade
                                for (size_t i = 0; i < ContiguousPageSize / PageSize; i++)
                                {
                                    const Bit64 entry = (HW_MMU_CONTIGUOUS | GetAsInteger(pAddr + PageSize * i) | entryAttr | PteTypePage);
                                    table.SetL3Entry(vAddr + PageSize * i, entry, GetTypedPointer<Bit64>(pL3Table));
                                }
                                MarkUpdated();
                                merged = true;
                            }

                            vAddr = RoundDown(vAddr, L2BlockSize);
                            KPhysicalAddress pAddr;
                            table.GetTableAddressFromEntry(&pAddr, l3Entry);
                            const Bit64 entryAttr = (l3Entry & entryAttrMask);
                            pAddr = RoundDown(pAddr, L2BlockSize);

                            // Check
                            for (size_t i = 0; i < L2BlockSize / ContiguousPageSize; i++)
                            {
                                const Bit64 entry = table.GetL3Entry(vAddr + ContiguousPageSize * i, GetTypedPointer<Bit64>(pL3Table));
                                if (entry != (HW_MMU_CONTIGUOUS | GetAsInteger(pAddr + ContiguousPageSize * i) | entryAttr))
                                {
                                    return merged;
                                }
                            }

                            // Upgrade
                            const Bit64 entryTemplate = (entryAttr & entryAttrMask);
                            l2Entry = GetAsInteger(pAddr) | entryTemplate | PteTypeBlock;
                            PteSynchronizationBarrier();
                            table.SetL2Entry(vAddr, l2Entry, GetTypedPointer<Bit64>(pL2Table));
                            MarkUpdated();
                            if (GetPageTableManager().InPageTableHeap(pL3Table))
                            {
                                GetPageTableManager().Close(pL3Table, L2BlockSize / PageSize);
                                FreePageTable(ppData, pL3Table);
                            }
                            merged = true;
                        }
                        break;
                    default:
                        return merged;
                    }
                }
                NN_FALL_THROUGH;
            case PteTypeBlock:
                {
                    if ((l2Entry & HW_MMU_CONTIGUOUS) == 0)
                    {
                        vAddr = RoundDown(vAddr, ContiguousL2BlockSize);
                        KPhysicalAddress pAddr = 0;
                        table.GetTableAddressFromEntry(&pAddr, l2Entry);
                        const Bit64 entryAttr = (l2Entry & entryAttrMask);
                        pAddr = RoundDown(pAddr, ContiguousL2BlockSize);

                        // Check
                        for (size_t i = 0; i < ContiguousL2BlockSize / L2BlockSize; i++)
                        {
                            const Bit64 entry = table.GetL2Entry(vAddr + L2BlockSize * i, GetTypedPointer<Bit64>(pL2Table));
                            if (entry != (GetAsInteger(pAddr + L2BlockSize * i) | entryAttr | PteTypeBlock))
                            {
                                return merged;
                            }
                        }

                        // Upgrade
                        for (size_t i = 0; i < ContiguousL2BlockSize / L2BlockSize; i++)
                        {
                            const Bit64 entry = (HW_MMU_CONTIGUOUS | GetAsInteger(pAddr + L2BlockSize * i) | entryAttr | PteTypeBlock);
                            table.SetL2Entry(vAddr + L2BlockSize * i, entry, GetTypedPointer<Bit64>(pL2Table));
                        }
                        MarkUpdated();
                        merged = true;
                    }

                    vAddr = RoundDown(vAddr, L1BlockSize);
                    KPhysicalAddress pAddr = 0;
                    table.GetBlockAddressFromEntry(&pAddr, l2Entry);
                    const Bit64 entryAttr = (l2Entry & entryAttrMask);
                    pAddr = RoundDown(pAddr, L1BlockSize);

                    // Check
                    for (size_t i = 0; i < L1BlockSize / ContiguousL2BlockSize; i++)
                    {
                        const Bit64 entry = table.GetL2Entry(vAddr + ContiguousL2BlockSize * i, GetTypedPointer<Bit64>(pL2Table));
                        if (entry != (HW_MMU_CONTIGUOUS | GetAsInteger(pAddr + ContiguousL2BlockSize * i) | entryAttr))
                        {
                            return merged;
                        }
                    }

                    // Upgrade
                    const Bit64 entryTemplate = (entryAttr & entryAttrMask);
                    masterEntry = (IsKernel()? HW_MMU_APTABLE_U_NA | HW_MMU_UXNTABLE: 0) | HW_MMU_PXNTABLE | GetAsInteger(pAddr) | entryTemplate | PteTypeBlock;
                    table.SetL1Entry(vAddr, masterEntry);
                    MarkUpdated();
                    if (GetPageTableManager().InPageTableHeap(pL2Table))
                    {
                        GetPageTableManager().Close(pL2Table, L1BlockSize / L2BlockSize);
                        FreePageTable(ppData, pL2Table);
                    }

                    merged = true;
                }
                break;
            default:
                return merged;
            }
        }
        break;
    default:
        return merged;
    }

    return merged;
}

Result KPageTable::MapL1BlocksWithTemplate(
        KProcessAddress vAddr, KPhysicalAddress pAddr,
        size_t numPages, Bit64 entryTemplate, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    NN_KERN_ASSERT((vAddr & (L1BlockSize - 1)) == 0);
    NN_KERN_ASSERT((pAddr & (L1BlockSize - 1)) == 0);
    NN_KERN_ASSERT((numPages & (L1BlockSize / PageSize - 1)) == 0);
    NN_UNUSED(ppData);
    KPageTableBody& table = GetPageTable();

    entryTemplate |= ((IsKernel()? HW_MMU_APTABLE_U_NA | HW_MMU_UXNTABLE: 0) | HW_MMU_PXNTABLE | PteTypeBlock);

    for (size_t i = 0; i < numPages; i+= L1BlockSize / PageSize)
    {
        table.SetL1Entry(vAddr, GetAsInteger(pAddr) | entryTemplate);
        pAddr += L1BlockSize;
        vAddr += L1BlockSize;
    }

    return ResultSuccess();
}

Result KPageTable::MapL2BlocksWithTemplate(
        KProcessAddress vAddr, KPhysicalAddress pAddr,
        size_t numPages, Bit64 entryTemplate, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    NN_KERN_ASSERT((vAddr & (L2BlockSize - 1)) == 0);
    NN_KERN_ASSERT((pAddr & (L2BlockSize - 1)) == 0);
    NN_KERN_ASSERT(((numPages * PageSize) & (L2BlockSize - 1)) == 0);
    KPageTableBody& table = GetPageTable();

    KVirtualAddress  l2TableVAddress = Null<KVirtualAddress>();
    KPhysicalAddress l2TablePAddress;
    int openL2Count = 0;

    for (size_t i = 0; i < numPages; i+= L2BlockSize / PageSize)
    {
        if (l2TableVAddress == Null<KVirtualAddress>())
        {
            if (!table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)))
            {
                l2TableVAddress = AllocatePageTable(ppData);
                if (l2TableVAddress == Null<KVirtualAddress>())
                {
                    return nn::svc::ResultOutOfResource();
                }
                l2TablePAddress = GetPageTablePhysicalAddress(l2TableVAddress);
                PteSynchronizationBarrier();
                table.SetL1Entry(vAddr, (IsKernel()? HW_MMU_APTABLE_U_NA | HW_MMU_UXNTABLE: 0) | HW_MMU_PXNTABLE | GetAsInteger(l2TablePAddress) | PteTypeTable);
                PteSynchronizationBarrier();
            }
            else
            {
                l2TableVAddress = GetPageTableVirtualAddress(l2TablePAddress);
            }
        }

        table.SetL2Entry(vAddr, GetAsInteger(pAddr) | entryTemplate | PteTypeBlock, GetTypedPointer<Bit64>(l2TableVAddress));
        openL2Count++;

        pAddr += L2BlockSize;
        vAddr += L2BlockSize;

        if ((vAddr & (L1BlockSize - 1)) == 0)
        {
            if (GetPageTableManager().InPageTableHeap(l2TableVAddress))
            {
                GetPageTableManager().Open(l2TableVAddress, openL2Count);
            }
            openL2Count = 0;
            l2TableVAddress = Null<KVirtualAddress>();
        }
    }

    if ((openL2Count > 0) && GetPageTableManager().InPageTableHeap(l2TableVAddress))
    {
        GetPageTableManager().Open(l2TableVAddress, openL2Count);
    }

    return ResultSuccess();
}


Result KPageTable::MapPagesWithTemplate(
        KProcessAddress vAddr, KPhysicalAddress pAddr,
        size_t numPages, Bit64 entryTemplate, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    NN_KERN_ASSERT((vAddr & (PageSize - 1)) == 0);
    NN_KERN_ASSERT((pAddr & (PageSize - 1)) == 0);
    KPageTableBody& table = GetPageTable();

    KVirtualAddress  l2TableVAddress = Null<KVirtualAddress>();
    KVirtualAddress  l3TableVAddress = Null<KVirtualAddress>();

    int openL2Count = 0;
    int openL3Count = 0;
    for (size_t i = 0; i < numPages; i++)
    {
        KPhysicalAddress l3TablePAddress;
        bool l2Allocated = false;
        if (l3TableVAddress == Null<KVirtualAddress>())
        {
            KPhysicalAddress l2TablePAddress = 0;
            if (l2TableVAddress == Null<KVirtualAddress>())
            {
                if (!table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)))
                {
                    l2TableVAddress = AllocatePageTable(ppData);
                    if (l2TableVAddress == Null<KVirtualAddress>())
                    {
                        return nn::svc::ResultOutOfResource();
                    }
                    l2TablePAddress = GetPageTablePhysicalAddress(l2TableVAddress);
                    PteSynchronizationBarrier();
                    table.SetL1Entry(vAddr, (IsKernel()? HW_MMU_APTABLE_U_NA | HW_MMU_UXNTABLE: 0) | HW_MMU_PXNTABLE | GetAsInteger(l2TablePAddress) | PteTypeTable);
                    PteSynchronizationBarrier();
                    l2Allocated = true;
                }
                else
                {
                    l2TableVAddress = GetPageTableVirtualAddress(l2TablePAddress);
                }
            }

            if (!table.GetTableAddressFromEntry(&l3TablePAddress, table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(l2TableVAddress))))
            {
                l3TableVAddress = AllocatePageTable(ppData);
                if (l3TableVAddress == Null<KVirtualAddress>())
                {
                    if (l2Allocated)
                    {
                        table.SetL1Entry(vAddr, UNMAPPED_ENTRY);
                        MarkUpdated();
                        FreePageTable(ppData, l2TableVAddress);
                    }
                    else if (GetPageTableManager().InPageTableHeap(l2TableVAddress) && (openL2Count > 0))
                    {
                        GetPageTableManager().Open(l2TableVAddress, openL2Count);
                    }
                    return nn::svc::ResultOutOfResource();
                }
                l3TablePAddress = GetPageTablePhysicalAddress(l3TableVAddress);
                PteSynchronizationBarrier();
                table.SetL2Entry(vAddr, (IsKernel()? HW_MMU_APTABLE_U_NA | HW_MMU_UXNTABLE: 0) | HW_MMU_PXNTABLE | GetAsInteger(l3TablePAddress) | PteTypeTable, GetTypedPointer<Bit64>(l2TableVAddress));
                openL2Count++;
                PteSynchronizationBarrier();
            }
            else
            {
                l3TableVAddress = GetPageTableVirtualAddress(l3TablePAddress);
            }
        }

        table.SetL3Entry(vAddr, GetAsInteger(pAddr) | entryTemplate | PteTypePage, GetTypedPointer<Bit64>(l3TableVAddress));
        openL3Count++;

        pAddr += PageSize;
        vAddr += PageSize;

        if ((vAddr & (L2BlockSize - 1)) == 0)
        {
            if (GetPageTableManager().InPageTableHeap(l3TableVAddress))
            {
                GetPageTableManager().Open(l3TableVAddress, openL3Count);
            }
            openL3Count = 0;
            l3TableVAddress = Null<KVirtualAddress>();

            if ((vAddr & (L1BlockSize - 1)) == 0)
            {
                if (GetPageTableManager().InPageTableHeap(l2TableVAddress))
                {
                    GetPageTableManager().Open(l2TableVAddress, openL2Count);
                }
                openL2Count = 0;
                l2TableVAddress = Null<KVirtualAddress>();
            }
        }
    }

    if ((openL2Count > 0) && GetPageTableManager().InPageTableHeap(l2TableVAddress))
    {
        GetPageTableManager().Open(l2TableVAddress, openL2Count);
    }
    if ((openL3Count > 0) && GetPageTableManager().InPageTableHeap(l3TableVAddress))
    {
        GetPageTableManager().Open(l3TableVAddress, openL3Count);
    }
    return ResultSuccess();
}

Result KPageTable::MapAdaptiveWithTemplate(
        KProcessAddress vAddr, const KPageGroup& pg,
        size_t numPages, Bit64 entryTemplate, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    const KProcessAddress cVirtualAddress = vAddr;
    Result result;
    size_t totalPages = 0;

    pg.Open();

    if (numPages < ContiguousPageSize / PageSize)
    {
        for (KPageGroup::BlockInfoList::const_iterator it = pg.GetBlockBeginIter(); it != pg.GetBlockEndIter(); it++)
        {
            size_t mapNumPages = it->GetNumPages();
            const KPhysicalAddress pMapAddr = GetHeapPhysicalAddress(it->GetBlockAddr());

            result = MapWithTemplate(vAddr, pMapAddr, mapNumPages, entryTemplate, PageSize, ppData);
            if (result.IsFailure())
            {
                NN_KERN_ABORT_IF_FAILED(Unmap(cVirtualAddress, numPages, nullptr, ppData, true));
                pg.Close();
                return result;
            }

            vAddr += mapNumPages * PageSize;
            totalPages += mapNumPages;
        }
    }
    else
    {
        AlignedMemoryBlock virtBlock;
        virtBlock.Initialize(GetAsInteger(vAddr), numPages, L1BlockSize);
        for (KPageGroup::BlockInfoList::const_iterator it = pg.GetBlockBeginIter(); it != pg.GetBlockEndIter(); it++)
        {
            size_t mapNumPages = it->GetNumPages();
            KPhysicalAddress pMapAddr = GetHeapPhysicalAddress(it->GetBlockAddr());

            AlignedMemoryBlock physBlock;
            physBlock.Initialize(GetAsInteger(pMapAddr), mapNumPages, virtBlock.GetAlign());
            virtBlock.SetAlign(physBlock.GetAlign());

            while (mapNumPages > 0)
            {
                uintptr_t physStart;
                size_t physNumPages = 0;

                physBlock.FindBlock(&physStart, &physNumPages);

                if (physNumPages == 0)
                {
                    const size_t nextAlign = AlignedMemoryBlock::NextAlign(physBlock.GetAlign());
                    NN_KERN_ASSERT(nextAlign >= PageSize);
                    physBlock.SetAlign(nextAlign);
                    virtBlock.SetAlign(nextAlign);
                    continue;
                }

                while (physNumPages > 0)
                {
                    uintptr_t virtStart;
                    size_t virtNumPages = physNumPages;

                    virtBlock.FindBlock(&virtStart, &virtNumPages);

                    if (virtNumPages == 0)
                    {
                        const size_t nextAlign = AlignedMemoryBlock::NextAlign(virtBlock.GetAlign());
                        NN_KERN_ASSERT(nextAlign >= PageSize);
                        physBlock.SetAlign(nextAlign);
                        virtBlock.SetAlign(nextAlign);
                        continue;
                    }

                    result = MapWithTemplate(virtStart, physStart, virtNumPages, entryTemplate, virtBlock.GetAlign(), ppData);
                    if (result.IsFailure())
                    {
                        NN_KERN_ABORT_IF_FAILED(Unmap(cVirtualAddress, numPages, nullptr, ppData, true));
                        pg.Close();
                        return result;
                    }
                    physStart += virtNumPages * PageSize;
                    physNumPages -= virtNumPages;
                    mapNumPages -= virtNumPages;
                    totalPages += virtNumPages;
                }
            }
        }
    }
    NN_KERN_ASSERT(totalPages == numPages);

    MergePage(cVirtualAddress, ppData);
    if (numPages > 1)
    {
        MergePage(cVirtualAddress + (numPages - 1) * PageSize, ppData);
    }

    return ResultSuccess();
}

// 連続した物理メモリをできるだけ大きいサイズのページ、セクションでマッピングする。
Result KPageTable::MapAdaptiveContinuousWithTemplate(
        KProcessAddress vAddr, KPhysicalAddress pAddr,
        size_t numPages, Bit64 entryTemplate, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    const KProcessAddress cVirtualAddress = vAddr;
    const KPhysicalAddress cPhysicalAddr  = pAddr;
    size_t leftPages = numPages;
    Result result;
    size_t align;

    if (numPages <  ContiguousPageSize / PageSize)
    {
        result = MapWithTemplate(vAddr, pAddr, numPages, entryTemplate, PageSize, ppData);
        if (result.IsFailure())
        {
            NN_KERN_ABORT_IF_FAILED(Unmap(cVirtualAddress, numPages, nullptr, ppData, true));
            return result;
        }
    }
    else
    {
        // 先頭からラージページ、セクションのアラインの端数部分をマッピングする。
        for (align = ContiguousPageSize; (vAddr & (align - 1)) == (pAddr & (align - 1));
                align = AlignedMemoryBlock::NextLargeAlign(align))
        {
            const size_t mapPages =
                (((align - GetAsInteger(vAddr)) & (align - 1)) & (align - 1)) / PageSize;

            if (mapPages + align / PageSize > leftPages)
            {
                break;
            }
            if (mapPages > 0)
            {
                result = MapWithTemplate(vAddr, pAddr, mapPages, entryTemplate, AlignedMemoryBlock::NextAlign(align), ppData);
                if (result.IsFailure())
                {
                    NN_KERN_ABORT_IF_FAILED(Unmap(cVirtualAddress, numPages, nullptr, ppData, true));
                    return result;
                }
                leftPages -= mapPages;
                vAddr += mapPages * PageSize;
                pAddr += mapPages * PageSize;
            }
            if (align == L1BlockSize)
            {
                break;
            }
        }

        // できるだけ大きいサイズのページ、セクションでマッピングする。
        while (leftPages > 0)
        {
            NN_KERN_ASSERT(align > PageSize);
            align = AlignedMemoryBlock::NextAlign(align);
            NN_KERN_ASSERT((GetAsInteger(vAddr) & (align - 1)) == 0);
            NN_KERN_ASSERT((GetAsInteger(pAddr) & (align - 1)) == 0);
            const size_t mapPages = leftPages & ~(align / PageSize - 1);
            if (mapPages > 0)
            {
                result = MapWithTemplate(vAddr, pAddr, mapPages, entryTemplate, align, ppData);
                if (result.IsFailure())
                {
                    NN_KERN_ABORT_IF_FAILED(Unmap(cVirtualAddress, numPages, nullptr, ppData, true));
                    return result;
                }
                leftPages -= mapPages;
                vAddr += mapPages * PageSize;
                pAddr += mapPages * PageSize;
            }
        }
    }

    MergePage(cVirtualAddress, ppData);
    if (numPages > 1)
    {
        MergePage(cVirtualAddress + (numPages - 1) * PageSize, ppData);
    }

    if (IsHeapPhysicalAddress(cPhysicalAddr))
    {
        Kernel::GetKernelHeapManager().Open(GetHeapVirtualAddress(cPhysicalAddr), numPages);
    }

    return ResultSuccess();
}

Result KPageTable::UpdateProperties(KProcessAddress vAddr, size_t numPages,
        Bit64 entryTemplate, bool breakBeforeMake, void** ppData, bool recursive)
{
    NN_KERN_ASSERT(IsLockedByMe());
    size_t                 leftPages     = numPages;
    TraverseContext        context;
    TraverseData           next;
    bool                   nextIsValid;
    KPageTableBody& table = GetPageTable();
    Result result = ResultSuccess();
    const KProcessAddress cVirtualAddress = vAddr;
    Bit64 prevEntryTemplate = 0;

    nextIsValid = table.TraverseBegin(&next, &context, vAddr);
    if (nextIsValid && (next.address & (next.size - 1)) != 0)
    {
        result = SeparatePage(vAddr, next.address & -GetAsInteger(next.address), ppData);
        if (result.IsFailure())
        {
            return result;
        }
        nextIsValid = table.TraverseBegin(&next, &context, vAddr);
    }

    while (leftPages > 0)
    {
        if (!nextIsValid)
        {
            NN_KERNEL_PANIC("failed to update properties.");
        }
        if (next.size > leftPages * PageSize)
        {
            result = SeparatePage(vAddr, leftPages * PageSize, ppData);
            if (result.IsFailure())
            {
                switch (next.size)
                {
                case L1BlockSize:
                    {
                        prevEntryTemplate = table.GetL1Entry(vAddr);
                        prevEntryTemplate &= (0xFFF0000000000FFFull & ~(PteTypeMask | HW_MMU_CONTIGUOUS));
                    }
                    break;
                case ContiguousL2BlockSize:
                case L2BlockSize:
                    {
                        KPhysicalAddress l2TablePAddress = 0;
                        NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));

                        prevEntryTemplate = table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
                        prevEntryTemplate &= (0xFFF0000000000FFFull & ~(PteTypeMask | HW_MMU_CONTIGUOUS));
                    }
                    break;
                case ContiguousPageSize:
                case PageSize:
                    {
                        KPhysicalAddress l2TablePAddress = 0;
                        KPhysicalAddress l3TablePAddress = 0;
                        NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));
                        NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l3TablePAddress, table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)))));

                        prevEntryTemplate = table.GetL3Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l3TablePAddress)));
                        prevEntryTemplate &= (0xFFF0000000000FFFull & ~(PteTypeMask | HW_MMU_CONTIGUOUS));
                    }
                    break;
                default:
                    {
                        NN_KERN_ABORT();
                    }
                    break;
                }

                goto error_exit;
            }

            nextIsValid = table.TraverseBegin(&next, &context, vAddr);
            NN_KERN_ASSERT(nextIsValid);
        }
        NN_KERN_ASSERT((next.address & (next.size - 1)) == 0);
        NN_KERN_ASSERT(next.size / PageSize <= leftPages);

        switch (next.size)
        {
        case L1BlockSize:
            {
                if (breakBeforeMake)
                {
                    table.SetL1Entry(vAddr, UNMAPPED_ENTRY);
                    MarkUpdated();
                    if (IsHeapPhysicalAddress(next.address))
                    {
                        KCPU::FlushDataCache(GetUntypedPointer(GetHeapVirtualAddress(next.address)), L1BlockSize);
                    }
                }
                table.SetL1Entry(vAddr, (IsKernel()? HW_MMU_APTABLE_U_NA | HW_MMU_UXNTABLE: 0) | HW_MMU_PXNTABLE | GetAsInteger(next.address) | entryTemplate | PteTypeBlock);
            }
            break;

        case ContiguousL2BlockSize:
            {
                KPhysicalAddress l2TablePAddress = 0;
                NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));
                if (breakBeforeMake)
                {
                    for (size_t i = 0; i < ContiguousL2BlockSize / L2BlockSize; i++)
                    {
                        table.SetL2Entry(vAddr + L2BlockSize * i, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
                    }
                    MarkUpdated();
                    if (IsHeapPhysicalAddress(next.address))
                    {
                        KCPU::FlushDataCache(GetUntypedPointer(GetHeapVirtualAddress(next.address)), ContiguousL2BlockSize);
                    }
                }
                for (size_t i = 0; i < ContiguousL2BlockSize / L2BlockSize; i++)
                {
                    table.SetL2Entry(vAddr + L2BlockSize * i, HW_MMU_CONTIGUOUS | GetAsInteger(next.address + L2BlockSize * i) | entryTemplate | PteTypeBlock, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
                }
            }
            break;


        case L2BlockSize:
            {
                KPhysicalAddress l2TablePAddress = 0;
                NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));
                if (breakBeforeMake)
                {
                    table.SetL2Entry(vAddr, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
                    MarkUpdated();
                    if (IsHeapPhysicalAddress(next.address))
                    {
                        KCPU::FlushDataCache(GetUntypedPointer(GetHeapVirtualAddress(next.address)), L2BlockSize);
                    }
                }
                table.SetL2Entry(vAddr, GetAsInteger(next.address) | entryTemplate | PteTypeBlock, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
            }
            break;

        case ContiguousPageSize:
            {
                KPhysicalAddress l2TablePAddress = 0;
                KPhysicalAddress l3TablePAddress = 0;
                NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));
                NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l3TablePAddress, table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)))));

                if (breakBeforeMake)
                {
                    for (size_t i = 0; i < ContiguousPageSize / PageSize; i++)
                    {
                        table.SetL3Entry(vAddr + PageSize * i, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l3TablePAddress)));
                    }
                    MarkUpdated();
                    if (IsHeapPhysicalAddress(next.address))
                    {
                        KCPU::FlushDataCache(GetUntypedPointer(GetHeapVirtualAddress(next.address)), ContiguousPageSize);
                    }
                }
                for (size_t i = 0; i < ContiguousPageSize / PageSize; i++)
                {
                    table.SetL3Entry(vAddr + PageSize * i, HW_MMU_CONTIGUOUS | GetAsInteger(next.address + PageSize * i) | entryTemplate | PteTypePage, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l3TablePAddress)));
                }
            }
            break;

        case PageSize:
            {
                KPhysicalAddress l2TablePAddress = 0;
                KPhysicalAddress l3TablePAddress = 0;
                NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));
                NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l3TablePAddress, table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)))));

                if (breakBeforeMake)
                {
                    table.SetL3Entry(vAddr, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l3TablePAddress)));
                    MarkUpdated(vAddr);
                    if (IsHeapPhysicalAddress(next.address))
                    {
                        KCPU::FlushDataCache(GetUntypedPointer(GetHeapVirtualAddress(next.address)), PageSize);
                    }
                }
                table.SetL3Entry(vAddr, GetAsInteger(next.address) | entryTemplate | PteTypePage, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l3TablePAddress)));
            }
            break;

        default:
            {
                NN_KERN_ABORT();
            }
            break;
        }

        if (MergePage(vAddr, ppData))
        {
            nextIsValid = table.TraverseBegin(&next, &context, vAddr);
            NN_KERN_ASSERT(nextIsValid);
            const size_t dataSize = next.size - (GetAsInteger(next.address) & (next.size - 1));
            vAddr += dataSize;
            leftPages = (dataSize / PageSize >= leftPages)? 0: leftPages - dataSize / PageSize;
        }
        else
        {
            vAddr += next.size;
            leftPages -= next.size / PageSize;
        }

        nextIsValid = table.TraverseNext(&next, &context);
    }

    MarkUpdated();

    return ResultSuccess();

error_exit:
    NN_KERN_ABORT_UNLESS(!recursive);
    NN_KERN_ABORT_UNLESS(prevEntryTemplate);
    NN_KERN_ABORT_IF_FAILED(UpdateProperties(cVirtualAddress, numPages - leftPages, prevEntryTemplate, breakBeforeMake, ppData, true));

    return result;
}


Result KPageTable::Unmap(KProcessAddress vAddr, size_t numPages, KPageGroup* pPgToBeClose, void** ppData, bool force)
{
    NN_KERN_ASSERT(IsLockedByMe());
    TraverseContext        context;
    TraverseData           next;
    size_t                 leftPages = numPages;
    bool                   nextIsValid;
    KPageTableBody& table = GetPageTable();
    Result result = ResultSuccess();
    const KProcessAddress cVirtualAddress = vAddr;
    Bit64 entryTemplate = 0;

    nextIsValid = table.TraverseBegin(&next, &context, vAddr);
    if (nextIsValid && (next.address & (next.size - 1)) != 0)
    {
        result = SeparatePage(vAddr, next.address & -GetAsInteger(next.address), ppData);
        if (result.IsFailure())
        {
            return result;
        }
        nextIsValid = table.TraverseBegin(&next, &context, vAddr);
    }

    while (leftPages > 0)
    {
        if (!nextIsValid)
        {
            NN_KERN_ABORT_UNLESS(force);
            size_t dataSize = next.size - (GetAsInteger(vAddr) & (next.size - 1));
            if (dataSize / PageSize > leftPages)
            {
                dataSize = leftPages * PageSize;
            }
            leftPages -= dataSize / PageSize;
            vAddr += dataSize;
        }
        else
        {
            if (next.size > leftPages * PageSize)
            {
                result = SeparatePage(vAddr, leftPages * PageSize, ppData);
                if (result.IsFailure())
                {
                    switch (next.size)
                    {
                    case L1BlockSize:
                        {
                            entryTemplate = table.GetL1Entry(vAddr);
                            entryTemplate &= (0xFFF0000000000FFFull & ~(PteTypeMask | HW_MMU_CONTIGUOUS));
                        }
                        break;
                    case ContiguousL2BlockSize:
                    case L2BlockSize:
                        {
                            KPhysicalAddress l2TablePAddress = 0;
                            if (!table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)))
                            {
                                NN_KERN_ABORT();
                            }
                            entryTemplate = table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
                            entryTemplate &= (0xFFF0000000000FFFull & ~(PteTypeMask | HW_MMU_CONTIGUOUS));
                        }
                        break;
                    case ContiguousPageSize:
                    case PageSize:
                        {
                            KPhysicalAddress l2TablePAddress = 0;
                            KPhysicalAddress l3TablePAddress = 0;
                            if (!table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)))
                            {
                                NN_KERN_ABORT();
                            }
                            if (!table.GetTableAddressFromEntry(&l3TablePAddress, table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)))))
                            {
                                NN_KERN_ABORT();
                            }
                            entryTemplate = table.GetL3Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l3TablePAddress)));
                            entryTemplate &= (0xFFF0000000000FFFull & ~(PteTypeMask | HW_MMU_CONTIGUOUS));
                        }
                        break;
                    default:
                        {
                            NN_KERN_ABORT();
                        }
                        break;
                    }

                    goto error_exit;
                }
                nextIsValid = table.TraverseBegin(&next, &context, vAddr);
                NN_KERN_ASSERT(nextIsValid);
            }
            NN_KERN_ASSERT((next.address & (next.size - 1)) == 0);
            NN_KERN_ASSERT(next.size / PageSize <= leftPages);

            switch (next.size)
            {
            case L1BlockSize:
                {
                    table.SetL1Entry(vAddr, UNMAPPED_ENTRY);
                    if (pPgToBeClose && IsHeapPhysicalAddress(next.address))
                    {
                        if (pPgToBeClose->AddBlock(GetHeapVirtualAddress(next.address), next.size / PageSize).IsFailure())
                        {
                            MarkUpdated();
                            Kernel::GetKernelHeapManager().Close(GetHeapVirtualAddress(next.address), next.size / PageSize);
                        }
                    }
                    vAddr += next.size;
                    leftPages -= next.size / PageSize;
                }
                break;

            case ContiguousL2BlockSize:
                {
                    KPhysicalAddress l2TablePAddress = 0;
                    NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));

                    for (size_t i = 0; i < ContiguousL2BlockSize / L2BlockSize; i++)
                    {
                        table.SetL2Entry(vAddr + L2BlockSize * i, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
                    }
                    PteSynchronizationBarrier();

                    KVirtualAddress l2TableVAddress = GetPageTableVirtualAddress(l2TablePAddress);
                    if (GetPageTableManager().InPageTableHeap(l2TableVAddress))
                    {
                        if (GetPageTableManager().Close(l2TableVAddress, ContiguousL2BlockSize / L2BlockSize))
                        {
                            table.SetL1Entry(vAddr, UNMAPPED_ENTRY);
                            MarkUpdated();
                            FreePageTable(ppData, l2TableVAddress);
                        }
                    }
                    if (pPgToBeClose && IsHeapPhysicalAddress(next.address))
                    {
                        if (pPgToBeClose->AddBlock(GetHeapVirtualAddress(next.address), next.size / PageSize).IsFailure())
                        {
                            MarkUpdated();
                            Kernel::GetKernelHeapManager().Close(GetHeapVirtualAddress(next.address), next.size / PageSize);
                        }
                    }
                    vAddr += next.size;
                    leftPages -= next.size / PageSize;
                }
                break;

            case L2BlockSize:
                {
                    KPhysicalAddress l2TablePAddress = 0;
                    NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));

                    table.SetL2Entry(vAddr, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
                    PteSynchronizationBarrier();

                    KVirtualAddress l2TableVAddress = GetPageTableVirtualAddress(l2TablePAddress);
                    if (GetPageTableManager().InPageTableHeap(l2TableVAddress))
                    {
                        if (GetPageTableManager().Close(l2TableVAddress, 1))
                        {
                            table.SetL1Entry(vAddr, UNMAPPED_ENTRY);
                            MarkUpdated();
                            FreePageTable(ppData, l2TableVAddress);
                        }
                    }
                    if (pPgToBeClose && IsHeapPhysicalAddress(next.address))
                    {
                        if (pPgToBeClose->AddBlock(GetHeapVirtualAddress(next.address), next.size / PageSize).IsFailure())
                        {
                            MarkUpdated();
                            Kernel::GetKernelHeapManager().Close(GetHeapVirtualAddress(next.address), next.size / PageSize);
                        }
                    }
                    vAddr += next.size;
                    leftPages -= next.size / PageSize;
                }
                break;

            case ContiguousPageSize:
                {
                    KPhysicalAddress l2TablePAddress = 0;
                    KPhysicalAddress l3TablePAddress = 0;
                    NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));
                    NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l3TablePAddress, table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)))));

                    for (size_t i = 0; i < ContiguousPageSize / PageSize; i++)
                    {
                        table.SetL3Entry(vAddr + PageSize * i, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l3TablePAddress)));
                    }
                    PteSynchronizationBarrier();

                    KVirtualAddress l3TableVAddress = GetPageTableVirtualAddress(l3TablePAddress);
                    if (GetPageTableManager().InPageTableHeap(l3TableVAddress))
                    {
                        if (GetPageTableManager().Close(l3TableVAddress, ContiguousPageSize / PageSize))
                        {
                            table.SetL2Entry(vAddr, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
                            MarkUpdated(vAddr);

                            KVirtualAddress l2TableVAddress = GetPageTableVirtualAddress(l2TablePAddress);
                            if (GetPageTableManager().InPageTableHeap(l2TableVAddress))
                            {
                                if (GetPageTableManager().Close(l2TableVAddress, 1))
                                {
                                    table.SetL1Entry(vAddr, UNMAPPED_ENTRY);
                                    MarkUpdated();
                                    FreePageTable(ppData, l2TableVAddress);
                                }
                            }
                            FreePageTable(ppData, l3TableVAddress);
                        }
                    }
                    if (pPgToBeClose && IsHeapPhysicalAddress(next.address))
                    {
                        if (pPgToBeClose->AddBlock(GetHeapVirtualAddress(next.address), next.size / PageSize).IsFailure())
                        {
                            MarkUpdated();
                            Kernel::GetKernelHeapManager().Close(GetHeapVirtualAddress(next.address), next.size / PageSize);
                        }
                    }
                    vAddr += next.size;
                    leftPages -= next.size / PageSize;
                }
                break;

            case PageSize:
                {
                    KPhysicalAddress l2TablePAddress = 0;
                    KPhysicalAddress l3TablePAddress = 0;

                    NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l2TablePAddress, table.GetL1Entry(vAddr)));
                    NN_KERN_ABORT_UNLESS(table.GetTableAddressFromEntry(&l3TablePAddress, table.GetL2Entry(vAddr, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)))));

                    table.SetL3Entry(vAddr, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l3TablePAddress)));
                    PteSynchronizationBarrier();

                    KVirtualAddress l3TableVAddress = GetPageTableVirtualAddress(l3TablePAddress);
                    if (GetPageTableManager().InPageTableHeap(l3TableVAddress))
                    {
                        if (GetPageTableManager().Close(l3TableVAddress, 1))
                        {
                            table.SetL2Entry(vAddr, UNMAPPED_ENTRY, GetTypedPointer<Bit64>(GetPageTableVirtualAddress(l2TablePAddress)));
                            MarkUpdated();

                            KVirtualAddress l2TableVAddress = GetPageTableVirtualAddress(l2TablePAddress);
                            if (GetPageTableManager().InPageTableHeap(l2TableVAddress))
                            {
                                if (GetPageTableManager().Close(l2TableVAddress, 1))
                                {
                                    table.SetL1Entry(vAddr, UNMAPPED_ENTRY);
                                    MarkUpdated();
                                    FreePageTable(ppData, l2TableVAddress);
                                }
                            }
                            FreePageTable(ppData, l3TableVAddress);
                        }
                    }
                    if (pPgToBeClose && IsHeapPhysicalAddress(next.address))
                    {
                        if (pPgToBeClose->AddBlock(GetHeapVirtualAddress(next.address), next.size / PageSize).IsFailure())
                        {
                            MarkUpdated();
                            Kernel::GetKernelHeapManager().Close(GetHeapVirtualAddress(next.address), next.size / PageSize);
                        }
                    }
                    vAddr += next.size;
                    leftPages -= next.size / PageSize;
                }
                break;
            default:
                {
                    NN_KERN_ABORT();
                }
                break;
            }
        }

        nextIsValid = table.TraverseNext(&next, &context);
    }

    if (IsKernel() && (numPages == 1))
    {
        // スーパバイザスタック解放用最適化
        MarkUpdatedForSvStack(cVirtualAddress);
    }
    else
    {
        MarkUpdated();
    }

    return ResultSuccess();

error_exit:
    NN_KERN_ABORT_UNLESS(!force);

    vAddr = cVirtualAddress;
    for (KPageGroup::BlockInfoList::const_iterator it = pPgToBeClose->GetBlockBeginIter(); it != pPgToBeClose->GetBlockEndIter(); it++)
    {
        NN_KERN_ABORT_IF_FAILED(MapAdaptiveContinuousWithTemplate(vAddr, GetHeapPhysicalAddress(it->GetBlockAddr()), it->GetNumPages(), entryTemplate, ppData));
        // MapAdaptiveContinuousWithTemplate() の Open を Close する
        Kernel::GetKernelHeapManager().Close(it->GetBlockAddr(), it->GetNumPages());
        vAddr += it->GetNumPages() * PageSize;
    }

    if (IsKernel() && (numPages == 1))
    {
        // スーパバイザスタック解放用最適化
        MarkUpdatedForSvStack(cVirtualAddress);
    }
    else
    {
        MarkUpdated();
    }

    return result;
}

Result KPageTable::FreePages(KProcessAddress vAddr, size_t numPages, void** pData)
{
    NN_KERN_ASSERT(IsLockedByMe());

    KScopedPageGroup pg(GetBlockInfoManager());

    Result result = Unmap(vAddr, numPages, &pg.GetPageGroup(), pData, false);
    if (result.IsFailure())
    {
        return result;
    }

    // 物理メモリの解放
    for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter();
         it != pg.GetPageGroup().GetBlockEndIter();
         it++)
    {
        Kernel::GetKernelHeapManager().Close(it->GetBlockAddr(), it->GetNumPages());

    }
    return ResultSuccess();
}

void KPageTable::MarkUpdatedForSvStack(KProcessAddress va)
{
    KCPU::DataSynchronizationBarrier();

    OnKernelTableUpdatedForSvStack(va);
}

void KPageTable::MarkUpdated()
{
    KCPU::DataSynchronizationBarrier();

    if (IsKernel())
    {
        OnKernelTableUpdated();
    }
    else
    {
        OnTableUpdated();
    }
}

void KPageTable::MarkUpdated(KProcessAddress va)
{
    KCPU::DataSynchronizationBarrier();

    if (IsKernel())
    {
        OnKernelTableUpdatedForSvStack(va);
    }
    else
    {
        OnTableUpdated(va);
    }
}

/* =======================================================================
        クラスメンバ / public
   ======================================================================== */

void KPageTable::Info()
{
    NN_KERN_RELEASE_LOG("Page Table usage\n");
    NN_KERN_RELEASE_LOG("    used / peak / num\n");
    NN_KERN_RELEASE_LOG("    %ld / %ld / %ld\n", GetPageTableManager().GetUsed(), GetPageTableManager().GetPeak(), GetPageTableManager().GetNum());
}

}
}}
