﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../../kern_Platform.h"
#include "kern_ExceptionHandlerPrint.h"
#include "../../kern_DebugString.h"
#include "../../kern_KProcess.h"
#include "../../kern_KThread.h"
#include "../../kern_InterruptManagerSelect.h"
#include "../ARM64/kern_RegisterAccess.h"

namespace nn { namespace kern { namespace ARMv8A {

#ifdef NN_KERN_FOR_DEVELOPMENT
    namespace
    {
        int s_ExceptionCount = 0;

        void PrintAbortStatus( Bit32 esr, uintptr_t fa )
        {
            Bit32 ec = (esr & HW_ESR_EC_MASK) >> HW_ESR_EC_SHIFT;
            Bit32 il = (esr & HW_ESR_IL) >> HW_ESR_IL_SHIFT;
            Bit32 iss = (esr & HW_ESR_ISS_MASK);
            NN_KERN_RELEASE_LOG("-- abort status ----\n");
            NN_KERN_RELEASE_LOG("Fault Address Register      = 0x%016llx\n", fa);
            NN_KERN_RELEASE_LOG("Exception Syndrome Register = 0x%08X\n", esr);
            NN_KERN_RELEASE_LOG("   EC  = 0x%02X\n", ec);
            NN_KERN_RELEASE_LOG("   IL  = 0x%01X\n", il);
            NN_KERN_RELEASE_LOG("   ISS = 0x%08X\n", iss);
        }

        void PrintGeneralRegisterInfo32( const ARM64::ExceptionContext& context )
        {
            int i = 0;
            for( ; i < 5; ++i )
            {
                NN_KERN_RELEASE_LOG("r%-2d=0x%08x ", i, static_cast<Bit32>(context.x[i]));
            }
            NN_KERN_RELEASE_LOG("\n");
            for( ; i < 10; ++i )
            {
                NN_KERN_RELEASE_LOG("r%-2d=0x%08x ", i, static_cast<Bit32>(context.x[i]));
            }
            NN_KERN_RELEASE_LOG("\n");
            for( ; i < 13; ++i )
            {
                NN_KERN_RELEASE_LOG("r%-2d=0x%08x ", i, static_cast<Bit32>(context.x[i]));
            }
            NN_KERN_RELEASE_LOG("sp=0x%08x lr0x%08x psr=0x%08x\n", context.x[13], context.x[14], context.psr);
            NN_KERN_RELEASE_LOG("pc=0x%08x\n", static_cast<Bit32>(context.pc));
            NN_KERN_RELEASE_LOG("tpidr=0x%08x\n", static_cast<Bit32>(context.tpidr));
        }

        void PrintGeneralRegisterInfo64( const ARM64::ExceptionContext& context )
        {
            for(int i = 0 ; i < 31; ++i )
            {
                NN_KERN_RELEASE_LOG("x%-2d=0x%016llx\n", i, context.x[i]);
            }
            NN_KERN_RELEASE_LOG("sp=0x%016llx psr=0x%08x\n", context.sp, static_cast<Bit32>(context.psr));
            NN_KERN_RELEASE_LOG("pc=0x%016llx\n", context.pc);
            NN_KERN_RELEASE_LOG("tpidr_el0=0x%016llx\n", context.tpidr);
        }


        void PrintAbortInfo( const ARM64::ExceptionContext& context, Bit32 esr, uintptr_t fa )
        {
            NN_KERN_RELEASE_LOG("-- registers ----\n");

            bool isAArch32 = false;

            switch (context.psr & HW_PSR_CPU_MODE_MASK)
            {
            case HW_PSR_USR_MODE:
            case HW_PSR_FIQ_MODE:
            case HW_PSR_IRQ_MODE:
            case HW_PSR_SVC_MODE:
            case HW_PSR_MON_MODE:
            case HW_PSR_ABORT_MODE:
            case HW_PSR_HYP_MODE:
            case HW_PSR_UNDEF_MODE:
            case HW_PSR_SYS_MODE:
                {
                    isAArch32 = true;
                }
                break;
            default:
                {
                    isAArch32 = false;
                }
                break;
            }

            if (isAArch32)
            {
                PrintGeneralRegisterInfo32(context);
            }
            else
            {
                PrintGeneralRegisterInfo64(context);
            }
            NN_KERN_RELEASE_LOG("\n");

            PrintAbortStatus( esr, fa );
        }

        void PrintCurrent()
        {
            KDisableInterrupt di;

            NN_KERN_RELEASE_LOG("-- current ----\n");
            KThread& currentThread       = GetCurrentThread();
            KProcess* pCurrentProcess    = GetCurrentProcessPointer();

            NN_KERN_RELEASE_LOG("cur thread               = %p (id = %lld)\n", &currentThread, currentThread.GetId());
            if( pCurrentProcess != NULL )
            {
                NN_KERN_RELEASE_LOG("cur process              = %p (id = %lld)\n", pCurrentProcess, pCurrentProcess->GetId());
            }
            else
            {
                NN_KERN_RELEASE_LOG("cur process              = NULL (id = 0)\n");
            }
            if( currentThread.GetParentPointer() != NULL )
            {
                NN_KERN_RELEASE_LOG("process of cur thread    = %p (id = %lld)\n", currentThread.GetParentPointer(), currentThread.GetParentPointer()->GetId());
            }
            else
            {
                NN_KERN_RELEASE_LOG("process of cur thread    = kernel\n");
            }
            if( pCurrentProcess != NULL )
            {
                NN_KERN_RELEASE_LOG("pagetable of cur process = %p\n", &pCurrentProcess->GetPageTable());
                pCurrentProcess->GetPageTable().GetPageTable().DumpMemoryBlocksLocked();
            }

            NN_KERN_RELEASE_LOG("\n");
        }
    }

    void CommonExceptionHandlerDebug(ARM64::ExceptionContext* pContext, Bit32 esr, uintptr_t fa)
    {
        Bit32 ec = ((esr & HW_ESR_EC_MASK) >> HW_ESR_EC_SHIFT);

        switch(ec)
        {
        case HW_ESR_EC_SERROR_INTERRUPT:
            NN_KERN_RELEASE_LOG("========== System Error Interrupt ==========\n");
            break;

        case HW_ESR_EC_DABORT_EL0:
        case HW_ESR_EC_DABORT_EL1:
            NN_KERN_RELEASE_LOG("========== Data Abort Exception ==========\n");
            break;

        case HW_ESR_EC_IABORT_EL0:
        case HW_ESR_EC_IABORT_EL1:
            NN_KERN_RELEASE_LOG("========== Instruction Abort Exception ==========\n");
            break;

        case HW_ESR_EC_SVC32:
        case HW_ESR_EC_SVC64:
            NN_KERN_RELEASE_LOG("========== Invalid SVC Call Exception ==========\n");
            break;

        case HW_ESR_EC_UNKNOWN:
        case HW_ESR_EC_ILLEGAL_EXECUTION:
        case HW_ESR_EC_BKPT:
        case HW_ESR_EC_BRK:
            NN_KERN_RELEASE_LOG("========== Undefined Instruction Exception ==========\n");
            break;

        default:
            NN_KERN_RELEASE_LOG("========== Synchronous Exception ==========\n");
            break;
        }

        switch (pContext->psr & HW_PSR_CPU_MODE_MASK)
        {
        case HW_PSR_EL0T_MODE:
        case HW_PSR_USR_MODE:
            break;

        default:
            {
                s_ExceptionCount++;
                if( s_ExceptionCount >= 2 )
                {
                    NN_KERNEL_PANIC("exception handler not implemented yet.");
                }
            }
            break;
        }

        PrintAbortInfo(*pContext, esr, fa);
        PrintCurrent();

        //NN_KERNEL_PANIC("stop.");
    }
#endif

}}}


