﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "../../kern_InterlockedSelect.h"
#include <nn/nn_BitTypes.h>

#include "../../kern_Result.h"
#include "../../kern_Assert.h"
#include "../../kern_KLinkedList.h"
#include "../../kern_KTaggedAddress.h"
#include "../../kern_CPUSelect.h"

#include "../../kern_KPageTableBase.h"
#include "../../kern_KPageGroup.h"
#include "../../kern_KMemoryBlock.h"
#include "../../kern_KMemoryManager.h"
#include "../../kern_KInterruptTask.h"
#include "../../kern_InterruptManagerSelect.h"

#include "kern_KPageTableBody.h"
#include "kern_KMemoryBlockManager.h"

namespace nn { namespace kern {

namespace ARMv7A
{

class KPageTable
    : public KPageTableBase
{
public:
    typedef KPageTableBody::TraverseContext TraverseContext;
    typedef KPageTableBody::TraverseData TraverseData;

private:

    // readonly
    Bit32                       m_TtbrValue;
    Bit32                       m_TtbcrValue;
    uint8_t                     m_Asid;

public:
    KPageTable() {}
    ~KPageTable(){}
    KPageTable& operator=(const KPageTable&) = delete;
    KPageTable(const KPageTable&) = delete;

    Result  Initialize(Bit32 spaceId, nn::svc::CreateProcessParameterFlag addressSapceType, bool enableAslr, bool fromBack, KMemoryManager::Region region, KProcessAddress codeRegionAddr, size_t codeRegionSize, KMemoryBlockResourceManager* pMemoryBlockManager, KBlockInfoManager* pBlockInfoManager, KPageTableManager* pPageTableManager);
    Result  InitializeForKernel(void* pTable, KProcessAddress begin, KProcessAddress end);
    Result  Finalize();
    bool GetStaticMap(KProcessAddress* pOut, KPhysicalAddress paddr, size_t numPages);

    // static
    static void Initialize(int32_t coreNo);
    static void Activate(KPageTable* pTable);

    // for HardwareBreak
    uint8_t GetAsid() { return m_Asid; }

    static void Info();
protected:
    Result OperateImpl(void** ppUpdateData,
            KProcessAddress vaddr, size_t numPages, KPhysicalAddress paddr, bool paIsValid,
            const KPageProperty property, PageTableOperation operation);
    Result  OperateImpl(void** ppUpdateData,
            KProcessAddress vaddr, size_t numPages, const KPageGroup& pg,
            const KPageProperty property, PageTableOperation operation);
    void PageTableUpdaterFinalize(void** ppData);

private:
    Bit32 GetEntryTemplate(const KPageProperty property) const
    {
        Bit32 entryTemplate = 0;
        Bit32 apx = 0;
        Bit32 global = (IsKernel() ? HW_MMU7_T2_GLOBAL: HW_MMU7_T2_ASID_ENABLE);
        const bool xn = ((property.permission & KMemoryPermission_All) != KMemoryPermission_UserReadExecute);

        switch ((property.permission & KMemoryPermission_UserReadWrite))
        {
        case KMemoryPermission_UserReadWrite:
            {
                apx = HW_MMU7_APX_ALL;
            }
            break;
        case KMemoryPermission_KernelReadWrite:
            {
                apx = HW_MMU7_APX_S_RW_U_NA;
            }
            break;
        case KMemoryPermission_UserRead:
            {
                apx = HW_MMU7_APX_S_RO_U_RO;
            }
            break;
        case KMemoryPermission_KernelRead:
            {
                apx = HW_MMU7_APX_S_RO_U_NA;
            }
            break;
        default:
            {
                NN_KERNEL_PANIC("invalid property.permission %x", property.permission);
            }
            break;
        }

        if (property.isIo)
        {
            NN_KERN_ABORT_UNLESS(!property.isUncached);
            NN_KERN_ABORT_UNLESS(xn);
            NN_KERN_ABORT_UNLESS(!(property.permission & (KMemoryPermission_KernelExecute | KMemoryPermission_UserExecute)));

            entryTemplate = HW_MMU7_T2_SP_PACK(0, apx, HW_MMU7_T2_SP_RGT_SHARED_DEV, global, true, xn);
        }
        else if (property.isUncached)
        {
            NN_KERN_ABORT_UNLESS(xn);
            NN_KERN_ABORT_UNLESS(!(property.permission & (KMemoryPermission_KernelExecute | KMemoryPermission_UserExecute)));

            entryTemplate = HW_MMU7_T2_SP_PACK(0, apx, HW_MMU7_T2_SP_RGT_L1L2C_NC, global, true, xn);
        }
        else
        {
            entryTemplate = HW_MMU7_T2_SP_PACK(0, apx, HW_MMU7_T2_SP_RGT_L1L2C_WB_WA, global, true, xn);
        }
        return entryTemplate;
    }

    Result  MapSmallPagesWithTemplate(KProcessAddress vAddr, KPhysicalAddress pAddr, size_t numPages, Bit32 entryTemplate);
    Result  MapWithTemplate(
            KProcessAddress vAddr, const KPageGroup& pg,
            size_t numPages, Bit32 entryTemplate, void** ppData);
    Result  MapContinuousWithTemplate(
            KProcessAddress vAddr, KPhysicalAddress pAddr,
            size_t numPages, Bit32 entryTemplate, void** ppData);


    Result  Unmap(KProcessAddress vAddr, size_t numPages, KPageGroup* pPgToBeClose, void** ppData);

    Result  UpdateProperties(KProcessAddress vAddr, size_t numPages, Bit32 entryTemplate, bool breakBeforeMake);
    Result  FreePages(KProcessAddress vAddr, size_t numPages, void** ppData);
    void    MarkUpdated();
    void    MarkUpdated(KProcessAddress va);
    void    MarkUpdatedForSvStack(KProcessAddress va);

    void    OnTableUpdated()
    {
        KCPU::InvalidateTlbByAsidAllCore(m_Asid);
    }
    void    OnTableUpdated(KProcessAddress va)
    {
        KCPU::InvalidateTlbByVaAsidAllCore(m_Asid, va);
    }
    void    OnKernelTableUpdated()
    {
        KCPU::InvalidateEntireTlbNoCodeModificationAllCore();
    }
    void    OnKernelTableUpdatedForSvStack(KProcessAddress va)
    {
        KCPU::InvalidateTlbByMvaNoCodeModificationAllCore(va);
    }

    KVirtualAddress AllocatePageTable(void** ppData);
    void FreePageTable(void** ppData, KVirtualAddress tableAddress);
};

}

}}

