﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../../../kern_Platform.h"
#include "crt0.h"
#include "kern_MemoryMap.h"
#include "../../ARM/kern_RegisterAccess.h"
#include "../kern_KPageTableDefinition.h"

#pragma GCC optimize("no-tree-loop-distribute-patterns")

namespace nn { namespace kern { namespace DEV { namespace ARMv7A { namespace crt0 {
void Step0()            __attribute__((section(".start")));
void Step1(int coreNo)  __attribute__((section(".start")));

namespace {
void InitMMUTable    (int coreNo)
    __attribute__((section(".start")));
void SetT1SuSecEntry (uintptr_t tableBase, uintptr_t vaddrBegin, uintptr_t vaddrEnd, uintptr_t paddrBegin,   Bit32 attr)
    __attribute__((section(".start")));
void SetT1SecEntry   (uintptr_t tableBase, uintptr_t vaddrBegin, uintptr_t vaddrEnd, uintptr_t paddrBegin,   Bit32 attr)
    __attribute__((section(".start")));
void SetT1CoarseEntry(uintptr_t tableBase, uintptr_t vaddrBegin, uintptr_t vaddrEnd, uintptr_t pCoarseBegin, int domain)
    __attribute__((section(".start")));
void SetT2SpEntry    (uintptr_t tableBase, uintptr_t vaddrBegin, uintptr_t vaddrEnd, uintptr_t paddrBegin,   Bit32 attr)
    __attribute__((section(".start")));
void ClearMemory(Bit32* p, size_t size)
    __attribute__((section(".start")));
void FlushCacheLevel(int level)__attribute__((section(".start")));
void FlushCache()__attribute__((section(".start")));
inline uintptr_t GetPrivateRegionPhysicalAddress()__attribute__((section(".start")));
inline void DataSynchronizationBarrier()__attribute__((section(".start")));
inline void DataMemoryBarrier()__attribute__((section(".start")));
void EnableCP15(int coreNo)__attribute__((section(".start")));
void InitMemoryManagement(int coreNo)__attribute__((section(".start")));
extern "C" void ExceptionVector();
}
#define __breakpoint(x) do { asm volatile("bkpt %0"::"I"((x)):); } while (false)
#define __isb(x)        do { asm volatile("isb":::"memory"); } while (false)
#define __dsb(x)        do { asm volatile("dsb":::"memory"); } while (false)
#define __dmb(x)        do { asm volatile("dmb":::"memory"); } while (false)
#undef NN_KERN_ALIGN_ASSERT
#define NN_KERN_ALIGN_ASSERT(a,b)
#if defined NN_BUILD_CONFIG_COMPILER_GCC
#define LDR(reg, value)     "ldr    " #reg ",= %a["#value"]\n"
#define B(dst)              "b      %a["#dst"]\n"
#elif defined NN_BUILD_CONFIG_COMPILER_CLANG
#define LDR(reg, value)     "ldr    " #reg ",= %["#value"]\n"
#define B(dst)              "b      %["#dst"]\n"
#endif


#define BKPT_UNLESS(exp)    if( ! (exp) ) { __breakpoint((__LINE__%100)); }
#define BKPT_UNLESS_ALIGN4K(exp)    BKPT_UNLESS( ((exp) & 0xFFF) == 0 )

namespace
{
    using namespace nn::kern::DEV::ARMv7A::crt0;

    uintptr_t GetPrivateRegionPhysicalAddress()
    {
        uintptr_t a;
        HW_GET_CP15_CONFIGURATION_BASE_ADDRESS(a);
        return a;
    }
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A9
    struct SnoopControlUnit
    {
        volatile Bit32 control;
        volatile Bit32 configuration;
        volatile Bit32 powerStatus;
        volatile Bit32 invalidateAll;
        volatile Bit32 filteringStart;
        volatile Bit32 filteringEnd;
        volatile Bit32 accessControl;
        volatile Bit32 nonSecureAccessControl;
    };

#if defined NN_BUILD_CONFIG_COMPILER_GCC || defined NN_BUILD_CONFIG_COMPILER_CLANG
    __attribute__((section(".start")))
#endif
    SnoopControlUnit* GetSnoopControlUnit()
    {
        return reinterpret_cast<SnoopControlUnit*>(GetPrivateRegionPhysicalAddress());
    }
#endif

    void SetT1SuSecEntry(uintptr_t tableBase, uintptr_t vaddrBegin, uintptr_t vaddrEnd, uintptr_t paddrBegin, Bit32 attr)
    {
        uintptr_t vaddr = (vaddrBegin);
        uintptr_t paddr = (paddrBegin);
        Bit32* p = reinterpret_cast<Bit32*>(tableBase) + (vaddr - 0) / HW_MMU7_T1_SEC_SIZE;
        while (vaddr < (vaddrEnd - 1))
        {
            *p++ = HW_MMU7_T1_SUSEC_PACK2(paddr, (attr));
            vaddr += HW_MMU7_T1_SEC_SIZE;
            paddr += HW_MMU7_T1_SEC_SIZE;
            if (vaddr == 0)
            {
                break;
            }
        }
    }

    void SetT1SecEntry(uintptr_t tableBase, uintptr_t vaddrBegin, uintptr_t vaddrEnd, uintptr_t paddrBegin, Bit32 attr)
    {
        uintptr_t vaddr = (vaddrBegin);
        uintptr_t paddr = (paddrBegin);
        Bit32* p = reinterpret_cast<Bit32*>(tableBase) + (vaddr - 0) / HW_MMU7_T1_SEC_SIZE;
        while (vaddr < (vaddrEnd - 1))
        {
            *p++ = HW_MMU7_T1_SEC_PACK2(paddr, (attr));
            vaddr += HW_MMU7_T1_SEC_SIZE;
            paddr += HW_MMU7_T1_SEC_SIZE;
            if (vaddr == 0)
            {
                break;
            }
        }
    }

    void SetT1CoarseEntry(uintptr_t tableBase, uintptr_t vaddrBegin, uintptr_t vaddrEnd, uintptr_t pCoarseBegin, int domain)
    {
        uintptr_t vaddr = (vaddrBegin);
        uintptr_t paddr = (pCoarseBegin);
        Bit32* p = reinterpret_cast<Bit32*>(tableBase) + (vaddr - 0) / HW_MMU7_T1_SEC_SIZE;
        while (vaddr < (vaddrEnd - 1))
        {
            *p++ = HW_MMU7_T1_COURSE_PACK(paddr, domain);
            vaddr += HW_MMU7_T1_SEC_SIZE;
            paddr += HW_MMU7_T1_CORS_SIZE;
            if (vaddr == 0)
            {
                break;
            }
        }
    }

    void SetT2SpEntry(uintptr_t tableBase, uintptr_t vaddrBegin, uintptr_t vaddrEnd, uintptr_t paddrBegin, Bit32 attr)
    {
        uintptr_t vaddr = (vaddrBegin);
        uintptr_t paddr = (paddrBegin);
        Bit32* p = reinterpret_cast<Bit32*>(tableBase) + (vaddr - 0) % HW_MMU7_T1_SEC_SIZE / HW_MMU7_T2_SP_SIZE;
        while (vaddr < (vaddrEnd - 1))
        {
            *p++ = HW_MMU7_T2_SP_PACK2(paddr, (attr));
            vaddr += HW_MMU7_T2_SP_SIZE;
            paddr += HW_MMU7_T2_SP_SIZE;
            if (vaddr == 0)
            {
                break;
            }
        }
    }
    void ClearMemory(Bit32* p, size_t size)
    {
        NN_KERN_ALIGN_ASSERT(size, 4);

        while(size > 0)
        {
            *p++ = 0;
            size -= 4;
        }
    }

    void FlushCacheLevel(int level)
    {
        Bit32 sizeInfo;

        HW_SET_CP15_CSSELR(level << 1);
        __isb(0xf);
        HW_GET_CP15_CCSIDR(sizeInfo);

        const int32_t numSet   = ((sizeInfo >> 13) & 0x7FFF);
        const int32_t numWay   = ((sizeInfo >>  3) & 0x3FF);
        const int32_t lineSize = ((sizeInfo >>  0) & 0x7);

        const int setShift = lineSize + 4;
        const int wayShift = __builtin_clz(numWay);

        for( int w = 0; w <= numWay; ++w )
        {
            for( int s = 0; s <= numSet; ++s )
            {
                const Bit32 ws = (w << wayShift) | (s << setShift) | (level << 1);
                HW_SET_CP15_CLEAN_INVALIDATE_DATA_CACHE_INDEX(ws);
            }
        }
    }

    void FlushCache()
    {
        Bit32 levelInfo;

        HW_GET_CP15_CLIDR(levelInfo);
        for(uint32_t level = 0; level < ((levelInfo >> 24) & 0x7); level++ )
        {
            FlushCacheLevel(level);
        }
    }

    inline void DataSynchronizationBarrier()
    {
        __dsb(0xf);
    }

    inline void DataMemoryBarrier()
    {
        __dmb(0xf);
    }

#if defined NN_BUILD_CONFIG_CPU_CORTEX_A9
#if defined NN_BUILD_CONFIG_COMPILER_GCC || defined NN_BUILD_CONFIG_COMPILER_CLANG
    __attribute__((section(".start")))
#endif
    void InitPl310( void )
    {
#if defined NN_BUILD_CONFIG_HARDWARE_KZMA9
        // MMU有効前に呼び出し
        // SDCGCLKCTRLを有効化
        *reinterpret_cast<volatile Bit32 *>(0xe0050000) = 1 << 24;

        // SMUの設定
        *reinterpret_cast<volatile Bit32 *>(0xe0110000 + 0xa4c) = 0; // 8-way
        *reinterpret_cast<volatile Bit32 *>(0xe0110000 + 0xa50) = 2; // way size = 32kb

        *reinterpret_cast<volatile Bit32 *>(0xe0110000 + 0x4b8) = 0; // USIBU1GCLKCTRL
        *reinterpret_cast<volatile Bit32 *>(0xe0110000 + 0x65c) = 3; // USIB2SCLKDIV
        *reinterpret_cast<volatile Bit32 *>(0xe0110000 + 0x4b8) = 3; // USIBU1GCLKCTRL

        *reinterpret_cast<volatile Bit32 *>(0xe0110000 + 0x648) = 4; // SDCGCLKCTRL
        *reinterpret_cast<volatile Bit32 *>(0xe0110000 + 0x4c8) = 7; // SDCGCLKCTRL
        *reinterpret_cast<volatile Bit32 *>(0xe0110000 + 0x4d4) = 5; // SDCGCLKCTRL

        // PL310の設定
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x100) = 0;
        // Auxiliary Control Register
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x104) = 0
            | (1 << 30) // BRESP enable
            | (1 << 29) // instruction prefetch enable
            | (1 << 28) // data prefetch enable
            | (1 << 25) // round-robin replacement
            | (1 << 22) // shared attribute internally ignored
            | (2 << 17) // way size = 32kb
            | (0 << 16) // 8-way
            ;
        // Tag and Data RAM Latency Control Registers
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x108) = 0x111;
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x10c) = 0x111;

        // Invalidate all
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x77c) = 0x00ff; // 8-way
        while (*reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x77c) & 0x00ff)
        {
            asm volatile("yield");
        }
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x730) = 0;
#elif defined NN_BUILD_CONFIG_HARDWARE_BDSLIMX6
        // PL310の設定
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x100) = 0;

        // Auxiliary Control Register
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x104) = 0
            | (1 << 29) // instruction prefetch enable
            | (1 << 28) // data prefetch enable
            | (1 << 25) // round-robin replacement
            | (3 << 17) // way size = 64kb
            | (1 << 16) // 16-way
            ;

        // Prefetch Control Register errata: 752271
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0xF60) = 0x30000000;

        // Power Control Register
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0xF80) |= 0x3;

        // Tag and Data RAM Latency Control Registers
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x108) = 0x132;
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x10c) = 0x132;

        // Invalidate all
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x77c) = 0xffff; // 16-way
        while (*reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x77c) & 0xffff)
        {
            asm volatile("yield");
        }
        DataSynchronizationBarrier();
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_P_ADDR_PL310_REGISTER + 0x730) = 0;
#else
#error not defined
#endif

    }

#if defined NN_BUILD_CONFIG_COMPILER_GCC || defined NN_BUILD_CONFIG_COMPILER_CLANG
    __attribute__((section(".start")))
#endif
    void EnablePl310( void )
    {
        // MMU有効後に呼び出し
        *reinterpret_cast<Bit32 *>(NN_KERN_V_ADDR_PL310_REGISTER + 0x100) = 1;
    }

#if defined NN_BUILD_CONFIG_COMPILER_GCC || defined NN_BUILD_CONFIG_COMPILER_CLANG
    __attribute__((section(".start")))
#endif
    void FlushPl310( void )
    {
        // MMU有効後に呼び出し
        // clean Invalidate all
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_V_ADDR_PL310_REGISTER + 0x7fc) = 0xffff; // 16-way
        while (*reinterpret_cast<volatile Bit32 *>(NN_KERN_V_ADDR_PL310_REGISTER + 0x7fc) != 0)
        {
        }
        *reinterpret_cast<volatile Bit32 *>(NN_KERN_V_ADDR_PL310_REGISTER + 0x730) = 0;
    }
#endif

    /*---------------------------------------------------------------------------*
      Name:         EnableCP15

      Description:  CP15 関連の有効化処理
                    - SCU制御レジスタの設定 (core0 のときのみ)
                    - 制御レジスタの設定
                    - 補助制御レジスタの設定

      Arguments:    None

      Returns:      None

      Note:         主に以下の処理が行われる
                    - MMU が有効になる
                    - CPU が SMP モードに設定される
                    - 各種キャッシュが有効(Enable)になる
     *---------------------------------------------------------------------------*/
    void EnableCP15(int coreNo)
    {
        const Bit32 c1_0_c0_0_SetBit    = 0
                                        //| HW_C1_THUMB_EXCEPT          // Thumb例外有効           → 無効
                                        //| HW_C1_FORCE_AP_BIT          // アクセスビット有効      → 無効
                                        //| HW_C1_TEX_CB_REMAP          // TEX リマップ有効        → 無効
                                        //| HW_C1_NMFI_FIQ              // FIQ を NMFI に          → 無効
                                        //| HW_C1_EXCEPT_BIG_ENDIAN     // 例外時エンディアン      → リトル
                                        //| HW_C1_EXCEPT_VEC_UPPER        // 例外ベクタ位置          → ハイベクタ
                                        //| HW_C1_INTERRUPT_VEC_ENABLE  // FIQ、IRQベクタ          → デフォルト
                                        //| HW_C1_FAST_INTERRUPT_EBABLE // 低遅延割り込み          → 無効
                                        //| HW_C1_HW_ACCESS_ENABLE      // ページテーブルのHWアクセスビット → 無効
                                        //| HW_C1_ROUND_ROBIN           // キャッシュのラウンドロビン → 無効
                                        | HW_C1_IC_ENABLE               // L1 命令キャッシュ       → 有効
                                        | HW_C1_DC_ENABLE               // L1 データキャッシュ     → 有効
                                        | HW_C1_BR_PREDICT_ENABLE       // 分岐予測器              → 有効
                                        | HW_C1_SWP_ENABLE              // SWP/SWPB命令            → 有効
                                        //| HW_C1_ALIGN_FAULT_ENABLE    // データアライメント      → アンアラインサポート
                                        | HW_C1_MMU_ENABLE              // MMU                     → 有効
                                    ;

        const Bit32 c1_0_c0_1_SetBit    = 0
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A7
                                        //| HW_C1_DDI                           //
                                        //| HW_C1_DDVM                          //
                                        | HW_C1_L1PCTL_3                      //
                                        //| HW_C1_L1RADIS                       //
                                        //| HW_C1_L2RADIS                       //
                                        //| HW_C1_DODMBS                        //
                                        | HW_C1_SMP                          //
#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A9
                                        //| HW_C1_PARITY_CHECK          // メモリのパリティチェック → 無効
                                        //| HW_C1_ALLOC_1WAY            // 1 キャッシュウェイ       → 無効
                                        //| HW_C1_CACHE_EXCL            // L1, L2キャッシュの排他   → 無効
                                        | HW_C1_SMP_MODE                // SMP or AMP               → SMP
                                        //| HW_C1_WRITE_FULL_LINE       // 0 のフルライン書き込みモード → 無効
                                        | HW_C1_L1_PREFETCH_ENABLE      // L1プリフェッチ           → 有効
                                        | HW_C1_L2_PREFETCH_ENABLE      // L2プリフェッチ           → 有効
                                        | HW_C1_HW_BROADCAST            // Cache, TLBのHWブロードキャスト → 有効
#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A15
                                        | HW_C1_DELAY_EXCLUSIVE               //
                                        //| HW_C1_FORCE_MAIN_CLOCK              //
                                        //| HW_C1_FORCE_VFP_CLOCK               //
                                        | HW_C1_WRITE_STREAMING_L1L2_12       //
                                        | HW_C1_WRITE_STREAMING_L1_4          //
                                        //| HW_C1_NON_CACHEABLE_ENHANCEMENT     //
                                        //| HW_C1_FORCE_IN_ORDER_TO_LINE        //
                                        //| HW_C1_FORCE_IN_ORDER_TO_LOAD        //
                                        //| HW_C1_DISABLE_L2_TLB_PREFETCH       //
                                        //| HW_C1_DISABLE_STAGE2_TLB            //
                                        //| HW_C1_DISABLE_STAGE1_TLB            //
                                        //| HW_C1_DISABLE_STAGE1_TLB_PA         //
                                        //| HW_C1_DISABLE_TLB_OPT               //
                                        //| HW_C1_ENABLE_DEVICE_REPLAY          //
                                        //| HW_C1_FORCE_IN_ORDER_ISSUE          //
                                        //| HW_C1_FORCE_LIMIT_ONE_INST          //
                                        | HW_C1_FLUSH_CP14_CP15               //
                                        //| HW_C1_FORCE_CP14_15_SHADOW          //
                                        //| HW_C1_LIMIT_ONE_INST                //
                                        //| HW_C1_FORCE_SERIALIZE               //
                                        //| HW_C1_DISABLE_FLAG_OPT              //
                                        //| HW_C1_WFI_AS_NOP                    //
                                        //| HW_C1_WFE_AS_NOP                    //
                                        | HW_C1_SMP                           //
                                        //| HW_C1_PLD_AS_NOP                    //
                                        //| HW_C1_DISABLE_INDIRECT_PREDICTOR    //
                                        //| HW_C1_DISABLE_MICRO_BTB             //
                                        //| HW_C1_LIMIT_ONE_LOOP                //
                                        //| HW_C1_DISABLE_LOOP_BUFFER           //
                                        | HW_C1_ENABLE_BTB_INVALIDATE         //
#endif
                                        ;
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A9
        if (coreNo == 0)
        {
            InitPl310();
            GetSnoopControlUnit()->control = 1;
        }
#endif
        const Bit32 vbar = reinterpret_cast<Bit32>(ExceptionVector);
        HW_SET_CP15_VBAR(vbar);

        // CP15 制御レジスタ
        HW_SET_CP15_CONTROL(c1_0_c0_0_SetBit);
        DataSynchronizationBarrier();
        HW_CP15_FLUSH_PREFETCH_BUFFER();

#if defined NN_BUILD_CONFIG_CPU_CORTEX_A9
        if (coreNo == 0)
        {
            EnablePl310();
            DataSynchronizationBarrier();
            HW_CP15_FLUSH_PREFETCH_BUFFER();
        }
#endif

        // CP15 補助制御レジスタ
        HW_SET_CP15_AUX_CONTROL(c1_0_c0_1_SetBit);
        DataSynchronizationBarrier();
        HW_CP15_FLUSH_PREFETCH_BUFFER();
    }


    /*---------------------------------------------------------------------------*
      Name:         InitMemoryManagement

      Description:  MMU の初期化
                    - 変換テーブルの設定

      Arguments:    coreNo:     現在のコア番号

      Returns:      None
     *---------------------------------------------------------------------------*/
    void InitMemoryManagement(int coreNo)
    {
        const uintptr_t tableBaseL1 = ( (coreNo == 0) ? NN_KERN_P_ADDR_L1_PAGE_TABLE_0:
                                        (coreNo == 1) ? NN_KERN_P_ADDR_L1_PAGE_TABLE_1:
                                        (coreNo == 2) ? NN_KERN_P_ADDR_L1_PAGE_TABLE_2:
                                                        NN_KERN_P_ADDR_L1_PAGE_TABLE_3 );
        const Bit32 ttb0mask = (HW_C2_0_T1_BASE_MASK_MAX << NN_KERN_MMU_TTBCR);
        const Bit32 ttb0     = (tableBaseL1 & (ttb0mask));
        const Bit32 ttb1     = (tableBaseL1 & HW_C2_1_T1_BASE_MASK);
        const Bit32 ttbr0    = (ttb0 |
                HW_C2_WALK_OUTER_RGN_PACK(HW_C2_WALK_RGN_WB_WA) |
                HW_C2_WALK_INNER_RGN_PACK(HW_C2_WALK_RGN_WB_WA) |
                HW_C2_WALK_ON_SHARED_MEM);
        const Bit32 ttbr1    = (ttb1 |
                HW_C2_WALK_OUTER_RGN_PACK(HW_C2_WALK_RGN_WB_WA) |
                HW_C2_WALK_INNER_RGN_PACK(HW_C2_WALK_RGN_WB_WA) |
                HW_C2_WALK_ON_SHARED_MEM);
        const Bit32 domain   = HW_C3_DOMAIN_PACK( HW_C3_DM_AP_CLIENT, HW_C3_DM_AP_NA, HW_C3_DM_AP_NA, HW_C3_DM_AP_NA,
                                                  HW_C3_DM_AP_NA,     HW_C3_DM_AP_NA, HW_C3_DM_AP_NA, HW_C3_DM_AP_NA,
                                                  HW_C3_DM_AP_NA,     HW_C3_DM_AP_NA, HW_C3_DM_AP_NA, HW_C3_DM_AP_NA,
                                                  HW_C3_DM_AP_NA,     HW_C3_DM_AP_NA, HW_C3_DM_AP_NA, HW_C3_DM_AP_NA );

        // TLB の無効化(Invalidate)
        HW_CP15_INVALIDATE_UNIFIED_TLB();

        // L1 変換テーブルの物理アドレスを設定
        HW_SET_CP15_TTB0(ttbr0, 0);
        HW_SET_CP15_TTB1(ttbr1, 0);

        // 変換テーブルのページテーブル境界のサイズを設定
        HW_SET_CP15_TTB_CONTROL(NN_KERN_MMU_TTBCR);

        // ASID を初期化
        HW_SET_CP15_CONTEXT_ID(0);

        // ドメインアクセスの許可設定
        HW_SET_CP15_DOMAIN_ACCESS_CONTROL(domain);

        // MMU 変換テーブルの初期化
        InitMMUTable(coreNo);
    }



    /*---------------------------------------------------------------------------*
      Name:         InitMMUTable

      Description:  MMU 変換テーブルの初期化

      Arguments:    coreNo:     現在のコア番号

      Returns:      None
     *---------------------------------------------------------------------------*/
    void InitMMUTable(int coreNo)
    {
        while (!(NN_KERN_V_ADDR_HEAP_END <= NN_KERN_V_ADDR_SVC_STACK)) {}

        const uintptr_t tableBaseL2KernelRegion = ( (coreNo == 0) ? NN_KERN_P_ADDR_L2_PAGE_TABLE_0 :
                                                    (coreNo == 1) ? NN_KERN_P_ADDR_L2_PAGE_TABLE_1 :
                                                    (coreNo == 2) ? NN_KERN_P_ADDR_L2_PAGE_TABLE_2 :
                                                                    NN_KERN_P_ADDR_L2_PAGE_TABLE_3 );
        // L1 ページテーブル
        {
            const Bit32 L1_ATTR_EXECUTABLE  = HW_MMU7_T1_SEC_PACK_ATTR(
                                                HW_MMU7_T1_APX_S_RW_U_NA,
                                                HW_MMU7_T1_RGT_L1L2C_WB_WA,
                                                HW_MMU7_T1_GLOBAL,
                                                HW_MMU7_T1_SHARED,
                                                false,
                                                0 );
            const Bit32 L1_ATTR_NORMAL_SS    = HW_MMU7_T1_SUSEC_PACK_ATTR(
                                                HW_MMU7_T1_APX_S_RW_U_NA,
                                                HW_MMU7_T1_RGT_L1L2C_WB_WA,
                                                HW_MMU7_T1_GLOBAL,
                                                HW_MMU7_T1_SHARED,
                                                HW_MMU7_T1_XN );

            const uintptr_t tableBaseL1 = ( (coreNo == 0) ? NN_KERN_P_ADDR_L1_PAGE_TABLE_0:
                                            (coreNo == 1) ? NN_KERN_P_ADDR_L1_PAGE_TABLE_1:
                                            (coreNo == 2) ? NN_KERN_P_ADDR_L1_PAGE_TABLE_2:
                                                            NN_KERN_P_ADDR_L1_PAGE_TABLE_3 );

            // ページテーブルを初期化

            // L1
            ClearMemory(reinterpret_cast<Bit32*>(tableBaseL1), NN_KERN_L1_PAGE_TABLE_SIZE);

            if( coreNo == 0 )
            {
                // L2
                ClearMemory(reinterpret_cast<Bit32*>(NN_KERN_P_ADDR_L2_PAGE_TABLE), NN_KERN_P_ADDR_L2_PAGE_TABLE_SIZE);
            }

            // crt0 用にカーネル領域を物理アドレスでマッピングする
            SetT1SecEntry(
                tableBaseL1,
                NN_KERN_P_ADDR_KERNEL_REGION,
                NN_KERN_P_ADDR_KERNEL_REGION_END,
                NN_KERN_P_ADDR_KERNEL_REGION,
                L1_ATTR_EXECUTABLE
            );

            // メインメモリリニアマッピング領域
            SetT1SuSecEntry(
                tableBaseL1,
                NN_KERN_V_ADDR_MAIN_MEMORY,
                NN_KERN_V_ADDR_MAIN_MEMORY_END,
                NN_KERN_P_ADDR_MAIN_MEMORY,
                L1_ATTR_NORMAL_SS
            );

            // スーパバイザモードスタック領域
            SetT1CoarseEntry(
                tableBaseL1,
                NN_KERN_V_ADDR_SVC_STACK,
                NN_KERN_V_ADDR_SVC_STACK_END,
                NN_KERN_P_ADDR_L2_PAGE_TABLE_FOR_SVC_STACK,
                0
            );

            // カーネル領域
            SetT1CoarseEntry(
                tableBaseL1,
                NN_KERN_V_ADDR_KERNEL_CODE_IO,
                NN_KERN_V_ADDR_KERNEL_CODE_IO_END,
                tableBaseL2KernelRegion,
                0
            );

        }

        // L2 ページテーブル for カーネル領域
        {
            const Bit32 L2_ATTR_EXECUTABLE  = HW_MMU7_T2_SP_PACK_ATTR(
                                                HW_MMU7_T2_APX_S_RO_U_NA,
                                                HW_MMU7_T2_SP_RGT_L1L2C_WB_WA,
                                                HW_MMU7_T2_GLOBAL,
                                                HW_MMU7_T2_SHARED,
                                                false );
            const Bit32 L2_ATTR_REGISTER    = HW_MMU7_T2_SP_PACK_ATTR(
                                                HW_MMU7_T2_APX_S_RW_U_NA,
                                                HW_MMU7_T2_SP_RGT_SHARED_DEV,
                                                HW_MMU7_T2_GLOBAL,
                                                HW_MMU7_T2_SHARED,
                                                HW_MMU7_T2_SP_XN );
            const Bit32 L2_ATTR_NORMAL      = HW_MMU7_T2_SP_PACK_ATTR(
                                                HW_MMU7_T2_APX_S_RW_U_NA,
                                                HW_MMU7_T2_SP_RGT_L1L2C_WB_WA,
                                                HW_MMU7_T2_GLOBAL,
                                                HW_MMU7_T2_SHARED,
                                                HW_MMU7_T2_SP_XN );
            const Bit32 L2_ATTR_READONLY    = HW_MMU7_T2_SP_PACK_ATTR(
                                                HW_MMU7_T2_APX_S_RO_U_NA,
                                                HW_MMU7_T2_SP_RGT_L1L2C_WB_WA,
                                                HW_MMU7_T2_GLOBAL,
                                                HW_MMU7_T2_SHARED,
                                                HW_MMU7_T2_SP_XN );

            // コード領域 - CODE_1 - SmallPage
            SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_CODE_MAIN_CODE - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_CODE_MAIN_CODE,
                    NN_KERN_V_ADDR_CODE_MAIN_CODE_END,
                    NN_KERN_P_ADDR_CODE_MAIN_CODE,
                    L2_ATTR_EXECUTABLE
                    );

            // コード領域 - RO
            SetT2SpEntry(
                tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_CODE_MAIN_RO - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                NN_KERN_V_ADDR_CODE_MAIN_RO,
                NN_KERN_V_ADDR_CODE_MAIN_RO_END,
                NN_KERN_P_ADDR_CODE_MAIN_RO,
                L2_ATTR_READONLY
            );

            // コード領域 - RW_ZI
            SetT2SpEntry(
                tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_CODE_MAIN_RW_ZI - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                NN_KERN_V_ADDR_CODE_MAIN_RW_ZI,
                NN_KERN_V_ADDR_CODE_MAIN_RW_ZI_END,
                NN_KERN_P_ADDR_CODE_MAIN_RW_ZI,
                L2_ATTR_NORMAL
            );

            // コア 0
            {
                // 共通スタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_CMN_0 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_CMN_0,
                    NN_KERN_V_ADDR_STACK_CMN_0_END,
                    NN_KERN_P_ADDR_STACK_CMN_0,
                    L2_ATTR_NORMAL
                );

                // L1ページテーブル
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_L1_PAGE_TABLE_0 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_L1_PAGE_TABLE_0,
                    NN_KERN_V_ADDR_L1_PAGE_TABLE_0_END,
                    NN_KERN_P_ADDR_L1_PAGE_TABLE_0,
                    L2_ATTR_NORMAL
                );

                // メインスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_MAIN_0 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_MAIN_0,
                    NN_KERN_V_ADDR_STACK_MAIN_0_END,
                    NN_KERN_P_ADDR_STACK_MAIN_0,
                    L2_ATTR_NORMAL
                );

                // アイドルスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_IDLE_0 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_IDLE_0,
                    NN_KERN_V_ADDR_STACK_IDLE_0_END,
                    NN_KERN_P_ADDR_STACK_IDLE_0,
                    L2_ATTR_NORMAL
                );

                // 固有領域
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_CORE_LOCAL_REGION_0 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION_0,
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION_0_END,
                    NN_KERN_P_ADDR_CORE_LOCAL_REGION_0,
                    L2_ATTR_NORMAL
                );
            }

            // コア 1
            {
                // 共通スタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_CMN_1 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_CMN_1,
                    NN_KERN_V_ADDR_STACK_CMN_1_END,
                    NN_KERN_P_ADDR_STACK_CMN_1,
                    L2_ATTR_NORMAL
                );

                // L1ページテーブル
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_L1_PAGE_TABLE_1 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_L1_PAGE_TABLE_1,
                    NN_KERN_V_ADDR_L1_PAGE_TABLE_1_END,
                    NN_KERN_P_ADDR_L1_PAGE_TABLE_1,
                    L2_ATTR_NORMAL
                );

                // メインスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_MAIN_1 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_MAIN_1,
                    NN_KERN_V_ADDR_STACK_MAIN_1_END,
                    NN_KERN_P_ADDR_STACK_MAIN_1,
                    L2_ATTR_NORMAL
                );

                // アイドルスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_IDLE_1 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_IDLE_1,
                    NN_KERN_V_ADDR_STACK_IDLE_1_END,
                    NN_KERN_P_ADDR_STACK_IDLE_1,
                    L2_ATTR_NORMAL
                );

                // 固有領域
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_CORE_LOCAL_REGION_1 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION_1,
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION_1_END,
                    NN_KERN_P_ADDR_CORE_LOCAL_REGION_1,
                    L2_ATTR_NORMAL
                );
            }

            // コア 2
            {
                // 共通スタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_CMN_2 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_CMN_2,
                    NN_KERN_V_ADDR_STACK_CMN_2_END,
                    NN_KERN_P_ADDR_STACK_CMN_2,
                    L2_ATTR_NORMAL
                );

                // L1ページテーブル
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_L1_PAGE_TABLE_2 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_L1_PAGE_TABLE_2,
                    NN_KERN_V_ADDR_L1_PAGE_TABLE_2_END,
                    NN_KERN_P_ADDR_L1_PAGE_TABLE_2,
                    L2_ATTR_NORMAL
                );

                // メインスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_MAIN_2 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_MAIN_2,
                    NN_KERN_V_ADDR_STACK_MAIN_2_END,
                    NN_KERN_P_ADDR_STACK_MAIN_2,
                    L2_ATTR_NORMAL
                );

                // アイドルスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_IDLE_2 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_IDLE_2,
                    NN_KERN_V_ADDR_STACK_IDLE_2_END,
                    NN_KERN_P_ADDR_STACK_IDLE_2,
                    L2_ATTR_NORMAL
                );

                // 固有領域
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_CORE_LOCAL_REGION_2 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION_2,
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION_2_END,
                    NN_KERN_P_ADDR_CORE_LOCAL_REGION_2,
                    L2_ATTR_NORMAL
                );
            }

            // コア 3
            {
                // 共通スタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_CMN_3 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_CMN_3,
                    NN_KERN_V_ADDR_STACK_CMN_3_END,
                    NN_KERN_P_ADDR_STACK_CMN_3,
                    L2_ATTR_NORMAL
                );

                // L1ページテーブル
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_L1_PAGE_TABLE_3 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_L1_PAGE_TABLE_3,
                    NN_KERN_V_ADDR_L1_PAGE_TABLE_3_END,
                    NN_KERN_P_ADDR_L1_PAGE_TABLE_3,
                    L2_ATTR_NORMAL
                );

                // メインスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_MAIN_3 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_MAIN_3,
                    NN_KERN_V_ADDR_STACK_MAIN_3_END,
                    NN_KERN_P_ADDR_STACK_MAIN_3,
                    L2_ATTR_NORMAL
                );

                // アイドルスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_IDLE_3 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_IDLE_3,
                    NN_KERN_V_ADDR_STACK_IDLE_3_END,
                    NN_KERN_P_ADDR_STACK_IDLE_3,
                    L2_ATTR_NORMAL
                );

                // 固有領域
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_CORE_LOCAL_REGION_3 - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION_3,
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION_3_END,
                    NN_KERN_P_ADDR_CORE_LOCAL_REGION_3,
                    L2_ATTR_NORMAL
                );
            }

            // カレントコア
            {
                const uintptr_t curCommonStack =  ( (coreNo == 0) ? NN_KERN_P_ADDR_STACK_CMN_0:
                                                    (coreNo == 1) ? NN_KERN_P_ADDR_STACK_CMN_1:
                                                    (coreNo == 2) ? NN_KERN_P_ADDR_STACK_CMN_2:
                                                                    NN_KERN_P_ADDR_STACK_CMN_3 );
                const uintptr_t curL1PageTable =  ( (coreNo == 0) ? NN_KERN_P_ADDR_L1_PAGE_TABLE_0:
                                                    (coreNo == 1) ? NN_KERN_P_ADDR_L1_PAGE_TABLE_1:
                                                    (coreNo == 2) ? NN_KERN_P_ADDR_L1_PAGE_TABLE_2:
                                                                    NN_KERN_P_ADDR_L1_PAGE_TABLE_3 );
                const uintptr_t curMainStack   =  ( (coreNo == 0) ? NN_KERN_P_ADDR_STACK_MAIN_0:
                                                    (coreNo == 1) ? NN_KERN_P_ADDR_STACK_MAIN_1:
                                                    (coreNo == 2) ? NN_KERN_P_ADDR_STACK_MAIN_2:
                                                                    NN_KERN_P_ADDR_STACK_MAIN_3 );
                const uintptr_t curIdleStack   =  ( (coreNo == 0) ? NN_KERN_P_ADDR_STACK_IDLE_0:
                                                    (coreNo == 1) ? NN_KERN_P_ADDR_STACK_IDLE_1:
                                                    (coreNo == 2) ? NN_KERN_P_ADDR_STACK_IDLE_2:
                                                                    NN_KERN_P_ADDR_STACK_IDLE_3 );
                const uintptr_t curLocalRegion =  ( (coreNo == 0) ? NN_KERN_P_ADDR_CORE_LOCAL_REGION_0:
                                                    (coreNo == 1) ? NN_KERN_P_ADDR_CORE_LOCAL_REGION_1:
                                                    (coreNo == 2) ? NN_KERN_P_ADDR_CORE_LOCAL_REGION_2:
                                                                    NN_KERN_P_ADDR_CORE_LOCAL_REGION_3 );

                // 共通スタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_CMN - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_CMN,
                    NN_KERN_V_ADDR_STACK_CMN_END,
                    curCommonStack,
                    L2_ATTR_NORMAL
                );

                // L1ページテーブル
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_L1_PAGE_TABLE - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_L1_PAGE_TABLE,
                    NN_KERN_V_ADDR_L1_PAGE_TABLE_END,
                    curL1PageTable,
                    L2_ATTR_NORMAL
                );

                // メインスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_MAIN - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_MAIN,
                    NN_KERN_V_ADDR_STACK_MAIN_END,
                    curMainStack,
                    L2_ATTR_NORMAL
                );

                // アイドルスタック
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_STACK_IDLE - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_STACK_IDLE,
                    NN_KERN_V_ADDR_STACK_IDLE_END,
                    curIdleStack,
                    L2_ATTR_NORMAL
                );

                // 固有領域
                // TODO: きちんと属性等を定める必要がある
                SetT2SpEntry(
                    tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_CORE_LOCAL_REGION - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION,
                    NN_KERN_V_ADDR_CORE_LOCAL_REGION_END,
                    curLocalRegion,
                    L2_ATTR_NORMAL
                );
            }

            // UART レジスタ領域
            SetT2SpEntry(
                tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_UART_REGISTER - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                NN_KERN_V_ADDR_UART_REGISTER,
                NN_KERN_V_ADDR_UART_REGISTER_END,
                NN_KERN_P_ADDR_UART_REGISTER,
                L2_ATTR_REGISTER
            );

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
            // UART レジスタ領域
            SetT2SpEntry(
                tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_MC_REGISTER - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                NN_KERN_V_ADDR_MC_REGISTER,
                NN_KERN_V_ADDR_MC_REGISTER_END,
                NN_KERN_P_ADDR_MC_REGISTER,
                L2_ATTR_REGISTER
            );
#endif

            // Core プライベート領域
            SetT2SpEntry(
                tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_CORE_PRIVATE_REGION - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                NN_KERN_V_ADDR_CORE_PRIVATE_REGION,
                NN_KERN_V_ADDR_CORE_PRIVATE_REGION_END,
                GetPrivateRegionPhysicalAddress(),
                L2_ATTR_REGISTER
            );

#if defined NN_BUILD_CONFIG_CPU_CORTEX_A9
            // PL310 レジスタ領域
            SetT2SpEntry(
                tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_PL310_REGISTER - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                NN_KERN_V_ADDR_PL310_REGISTER,
                NN_KERN_V_ADDR_PL310_REGISTER_END,
                NN_KERN_P_ADDR_PL310_REGISTER,
                L2_ATTR_REGISTER
            );
#endif

            // ページテーブル(L2テーブル)
            SetT2SpEntry(
                tableBaseL2KernelRegion + 1024 * ((NN_KERN_V_ADDR_L2_PAGE_TABLE - NN_KERN_V_ADDR_KERNEL_CODE_IO) / 0x00100000),
                NN_KERN_V_ADDR_L2_PAGE_TABLE,
                NN_KERN_V_ADDR_L2_PAGE_TABLE_END,
                NN_KERN_P_ADDR_L2_PAGE_TABLE,
                L2_ATTR_NORMAL
            );
        }
    }   // NOLINT(readability/fn_size)
}

/*---------------------------------------------------------------------------*
  Name:         Step1

  Description:  スタックが用意されてからメモリ管理を有効にするまでの処理

  Arguments:    coreNo: 実行中のコア番号

  Returns:      None.
 *---------------------------------------------------------------------------*/
__attribute__((optimize("-O0")))
void Step1(int coreNo)
{
    if( coreNo == 0 )
    {
        WakeUpOtherCore();
    }
    else
    {
        WaitForCore0(coreNo);
    }

    // メモリ管理の初期化
    InitMemoryManagement(coreNo);

    // CP15 関連の有効化処理 (MMU が有効になる)
    EnableCP15(coreNo);

    if( coreNo == 0 )
    {
        FlushCache();
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A9
        FlushPl310();
#endif
    }
    else
    {
        FlushCache();
    }

    HW_CP15_INVALIDATE_ENTIRE_INSTRUCTION_CACHE();
    HW_CP15_FLUSH_ENTIRE_BRANCH_TARGET_CACHE();
    DataSynchronizationBarrier();
    HW_CP15_FLUSH_PREFETCH_BUFFER();
}
}}}}}

