﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Fpu.h>
#include "../../kern_Platform.h"
#include "kern_KContext.h"
#include "../../kern_Kernel.h"
#include "../../kern_KProcess.h"
#include "../../kern_KThread.h"
#include "../../kern_KScheduler.h"
#include "../../kern_InterruptNameSelect.h"
#include "../../kern_CPUSelect.h"
#include "../../kern_DebugSelect.h"
#include "../../kern_DpcManager.h"
#include "../../kern_Utility.h"
#include "kern_ExceptionContext.h"
#include <nn/nn_BitTypes.h>

#define MAKE_PSR_VALUE(cpuMode, cpuState) (cpuMode | cpuState | HW_PSR_DATA_LITTLE_ENDIAN)

#if defined NN_BUILD_CONFIG_COMPILER_GCC || defined NN_BUILD_CONFIG_COMPILER_CLANG
#define __yield()   do { asm volatile("yield":::"memory"); } while (false)
#endif
namespace nn { namespace kern { namespace ARM {

namespace
{
#if     defined(NN_BUILD_CONFIG_FPU_VFP)
    void EnableFpu()
    {
        Bit32 fpexc;
        asm volatile ("fmrx %0, fpexc":"=r"(fpexc)::"memory");
        fpexc |= HW_FPEXC_VFP_ENABLE;
        asm volatile ("fmxr fpexc, %0"::"r"(fpexc):"memory");
        KCPU::InstructionMemoryBarrier();
    }
    __attribute__((unused))
    bool IsFpuEnabled()
    {
        Bit32 fpexc;
        asm volatile ("fmrx %0, fpexc":"=r"(fpexc)::"memory");
        return !!(fpexc & HW_FPEXC_VFP_ENABLE);
    }
#endif

    KVirtualAddress
    InitializeKernelStackForUserModeThreadStarter(
        KVirtualAddress stack,
        KVirtualAddress userModeEntryPoint,
        KVirtualAddress userModeStackBottom,
        uintptr_t            param )
    {
        const Bit32 cpuState = ((userModeEntryPoint & 0x1) == 0) ? HW_PSR_ARM_STATE: HW_PSR_THUMB_STATE;
        ExceptionContext* pContext = GetTypedPointer<ExceptionContext>(stack) - 1;

        std::memset(pContext, 0, sizeof(ExceptionContext));
        pContext->sp_usr = GetAsInteger(userModeStackBottom);
        pContext->pc = GetAsInteger(userModeEntryPoint);
        pContext->cpsr = MAKE_PSR_VALUE(HW_PSR_USR_MODE, cpuState);
        pContext->r[0] = param;
        return reinterpret_cast<uintptr_t>(pContext);
    }
    KVirtualAddress
    InitializeKernelStackForSupervisorModeThreadStarter(
        KVirtualAddress stack,
        KVirtualAddress entryPoint,
        uintptr_t            param )
    {
        // カーネルスタックに
        //   stack - 4      r0      スレッド引数
        //   stack - 0      PC      スレッドのメイン関数へのポインタ
        // を積む
        Bit32* pStack = GetTypedPointer<Bit32>(stack);
        *--pStack = GetAsInteger(entryPoint);
        *--pStack = param;
        return reinterpret_cast<uintptr_t>(pStack);
    }
}

void onThreadStart()
{
    NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());
    {
        KEnableInterrupt interruptEnabled;
        KDebug::OnDebugEvent(nn::svc::DebugEvent_CreateThread, GetCurrentThread().GetId(), GetAsInteger(GetCurrentThread().GetThreadLocalRegionAddr()), GetCurrentThread().GetEntryAddress());
    }

    while (GetCurrentThread().IsDpcRegistered())
    {
        DpcManager::DpcHandler();
    }
    GetCurrentThread().ClearInExceptionHandler();
}


Result KContext::Initialize(KVirtualAddress initialPc, KVirtualAddress kernelSp, KVirtualAddress userSp, uintptr_t firstArg, bool isUserMode, bool is64Bit, bool isMain)
{
    KVirtualAddress pc;
    KVirtualAddress sp;

    NN_KERN_ASSERT(kernelSp != Null<KVirtualAddress>());

    if( isUserMode )
    {
        pc = reinterpret_cast<uintptr_t>(&UserModeThreadStarter);
        sp = InitializeKernelStackForUserModeThreadStarter(
                RoundDown(kernelSp,8), initialPc, RoundDown(userSp,8), firstArg );
    }
    else
    {
        if (isMain)
        {
            pc = initialPc;
            sp = RoundDown(kernelSp, 8);
        }
        else
        {
            pc = reinterpret_cast<uintptr_t>(&SupervisorModeThreadStarter);
            sp = InitializeKernelStackForSupervisorModeThreadStarter(
                    RoundDown(kernelSp,8), initialPc, firstArg );
        }
    }

    Bit32*  pStack = GetTypedPointer<Bit32>(sp);

    // LR (PC)
    m_Lr = GetAsInteger(pc);

    // SP
    m_Sp    = reinterpret_cast<Bit32>(pStack);

    // R4-R11
    for( size_t i = 0; i < sizeof(m_CpuRegisters) / sizeof(*m_CpuRegisters); ++i )
    {
        m_CpuRegisters[i] = 0;
    }

#if     defined(NN_BUILD_CONFIG_FPU_VFP)
    // FPEXC
    // EN = 0
    m_Fpexc = 0;

    // FPSCR
    // DN = 0, FZ = 0, Rmode = 0
    m_Fpscr = (HW_FPSCR_ROUND_NEAR_MODE << HW_FPSCR_ROUND_MODE_SFT);

    // D0-D15
    for( size_t i = 0; i < sizeof(m_FpuRegisters) / sizeof(*m_FpuRegisters); ++i )
    {
        m_FpuRegisters[i]  = 0;
    }
#endif

    m_Lock = isMain;

    return ResultSuccess();
}

void KContext::SetFirstArgument(uintptr_t firstArg, uintptr_t secondArg)
{
    Bit32* pStack = reinterpret_cast<Bit32*>(m_Sp);
    pStack[0] = firstArg;
    pStack[1] = secondArg;
}

Result KContext::Finalize()
{
    return ResultSuccess();
}


#if     defined(NN_BUILD_CONFIG_FPU_VFP)
const Bit64*  KContext::GetVfpRegisters() const
{
    return m_FpuRegisters;
}

Bit32   KContext::GetFpexc() const
{
    return m_Fpexc;
}

Bit32   KContext::GetFpscr() const
{
    return m_Fpscr;
}

void KContext::SetVfpRegisters(const Bit64* x)
{
    for( size_t i = 0; i < GetNumOfVfpRegisters(); ++i )
    {
        m_FpuRegisters[i] = x[i];
    }
}

void KContext::SetFpexc(Bit32 fpexc)
{
    m_Fpexc = fpexc;
}

void KContext::SetFpscr(Bit32 fpscr)
{
    m_Fpscr = fpscr;
}

void KContext::VfpSwitchHandler(KThread* pNext)
{
    NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());
    NN_KERN_ASSERT(!IsFpuEnabled());

    // VFP を有効に
    EnableFpu();

    // コンテキストを復帰
    RestoreVfp(*pNext->GetContext());

#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
    pNext->IncrementNumFpuSwitch();
#endif
}

void KContext::OnThreadCoreChanged(KThread* pThread)
{
}

void KContext::OnThreadTerminating(const KThread* pThread)
{
}

void KContext::CloneFpuStatus()
{
    Bit32 fpscr;
    KCPU::InstructionMemoryBarrier();
    if (IsFpuEnabled())
    {
        asm volatile ("fmrx %0, fpscr":"=r"(fpscr)::"memory");
    }
    else
    {
        fpscr = GetCurrentThread().GetContext()->GetFpscr();
    }
    SetFpscr(fpscr);
}


#endif

const Bit32*  KContext::GetCpuRegisters() const
{
    return m_CpuRegisters;
}

void KContext::SetCpuRegister(uint32_t index, uint32_t val)
{
    m_CpuRegisters[index] = val;
}
}}}

