﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnc/socket/socket_Types.h>

#include <stdint.h>

#if ! defined(__cplusplus)
#include <stdbool.h> // NOLINT(others/c_header)
#endif


/**
 * @brief This type defines the ResolverOptionKey type and
 * constants that are used to delienate one option from another.
 *
 * @details
 * The range of ResolverOptionKeys are divided into three sub-groups.
 * The naming scheme of each option describes which context it can
 * be used in. There are three contexts: Get, Set, and Request.  Any
 * attempt to use an option outside of its allowed context results
 * in an error being returned to the caller and
 * @ref nn::socket::GetLastError() set
 *
 * - Get keys: Key names that start with "ResolverOptionKey_Get"
 * (range 10000-19999) are used with options that retrieve information
 * from the resolver. The result is a @ref ResolverOption that
 * corresponds to the data described. For more information see
 * @ref nn::socket::ResolverGetOption() and
 * @ref nn::socket::ResolverSetOption().
 *
 * - Set keys: Keys constants that start with "ResolverOptionKey_Set"
 * (range 20000-29999) are used with options that perform an action
 * or set a configuration value in the resolver. For more information
 * see @ref nn::socket::ResolverGetOption() and @ref
 * nn::socket::ResolverSetOption().
 *
 * - Request keys: Key that begin with "ResolverOptionKey_Request"
 * (range 30000-39999) provide options that are used only with a
 * resolver request lookup function such as
 * @ref nn::socket::GetHostEntByName GetHostEntByName(),
 * @ref nn::socket::GetHostEntByAddr GetHostEntByAddr(),
 * @ref nn::socket::GetAddrInfo GetAddrInfo(),
 * and @ref nn::socket::GetNameInfo GetNameInfo().
 */
typedef enum nnsocketResolverOptionKey
{
    // ---------------------------------------------------------------
    // Get Option Block
    // ---------------------------------------------------------------

    /**
     * @brief
     * This constant defines the minimum boundary for Get key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionKey_GetMinimumValue = 10000,

    /**
     * @brief
     * Get a cancel handle from the resolver for use by the client.
     *
     * @details
     * You can use the returned handle to immediately cancel a long-term
     * resolver activity such as gethostbyname().  The API is provided
     * for situations where one no longer wishes to wait for a
     * response from the resolver.
     *
     * @see
     * nn::socket::Finalize()
     *
     * @code
     * // Get the handle
     * ResolverOption option, optionSave;
     * nn::socket::ResolverGetOption(option, ResolverOptionKey::GetCancelHandleInteger);
     * optionSave = option;
     *
     * // Use the cancel handle
     * option.key = ResolverOptionKey::RequestCancelHandleInteger;
     * nn::socket::GetHostEntByName(host, &option, 1);
     *
     * // In another thread you can cancel the request in progress by calling
     * // @ref ResolverSetOption. Note that you must not use the request option
     * // because its memory is not available for read.
     * optionSave.key = SetCancelHandleInteger;
     * nn::socket::ResolverSetOption(optionSave);
     * @endcode
     */
    nnsocketResolverOptionKey_GetCancelHandleInteger = 10001,

    /**
     * @brief
     * This constant defines a maximum boundary for Get key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionKey_GetMaximumValue = 10002,


    // ---------------------------------------------------------------
    // Set Option Block
    // ---------------------------------------------------------------

    /**
     * @brief
     * This constant defines the minimum boundary for Get key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionKey_SetMinimumValue = 20000,

    /**
     * @brief
     * This constant cancels an active DNS request.
     *
     * @code
     * // Get the cancel handle to be used with request
     * ResolverOption option = {
     *     .key = SetCancelHandleInteger,
     *     .type = ResolverOptionType_Integer,
     *     .data.integerValue = cancelHandle
     * };
     *
     * @endcode
     */
    nnsocketResolverOptionKey_SetCancelHandleInteger = 20001,

    /**
     * @brief
     * This constant removes a hostname from the DNS cache, which
     * effectively forces the resolver client to hit the DNS server
     * on subsequent lookups.
     *
     * @code
     * // Get the cancel handle
     * ResolverOption option = {
     *     .key = SetRemoveHostnameFromCacheString,
     *     .type = ResolverOptionType_String,
     * };
     *
     * if (strlen(host) < sizeof(option.data.stringValue))
     * {
     *     option.size = snprintf(option.data.stringValue,
     *                            sizeof(option.data.stringValue),
     *                            "%s", host);
     *
     *     rc = nn::socket::SetResolverOption(option);
     *     //...
     * @endcode
     */
    nnsocketResolverOptionKey_SetRemoveHostnameFromCacheString = 20002,

    /**
     * @brief
     * This constant removes an ip address in network-byte order from
     * the DNS cache, which effectively forces the resolver client to
     * hit the DNS server on subsequent lookups.
     *
     * @code
     * // Get the cancel handle
     * ResolverOption option = {
     *     .key = SetRemoveHostnameFromCacheString,
     *     .type = ResolverOptionType_Unsigned32,
     *     .size = sizeof(uint32_t)
     * };
     *
     * inet_ntop(AF_INET, "1.2.3.4", &option.data.unsigned32Value,
     *           sizeof(uint32_t));
     *
     * rc = nn::socket::SetResolverOption(option);
     * @endcode
     */
    nnsocketResolverOptionKey_SetRemoveIpAddressFromCacheUnsigned32 = 20003,

    /**
     * @brief
     * This constant defines the maximum boundary for Get key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionKey_SetMaximumValue = 20004,

    // ---------------------------------------------------------------
    // Request Option Block
    // ---------------------------------------------------------------

    /**
     * @brief
     * This constant defines the minimum boundary for Request key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionKey_RequestMinimumValue = 30000,

    /**
     * @brief This option enables or disables Network Service
     * Discovery for a single resolver request.
     *
     * @details by default NSD is enabled for the following calls
     * - @ref nn::socket::GetHostEntByName
     * - @ref nn::socket::GetAddrInfo
     *
     * @code
     * ResolverOption option = {
     *     .key = RequestEnableServiceDiscoveryBoolean,
     *     .type = ResolverOptionType_Boolean,
     *     .size = sizeof(bool),
     *     .data.boolValue = true
     * };
     *
     * // then make a resolver call such as:
     * nn::socket::GetHostEntByName(name, &option, 1);
     * @endcode
     */
    nnsocketResolverOptionKey_RequestEnableServiceDiscoveryBoolean = 30001,

    /**
     * @brief This option contains a unique resolver cancellation
     * handle that can be used to cancel a request. It is made
     * available only on the Horizon platform and is ignored on other
     * platforms.
     *
     * @details The total time spent in a DNS call depend on multiple
     * conditions such as DNS delegation and recursion, network
     * availability, server resources, and even security options.
     * Therefore time spent is a resolver call is not deterministic
     * and it might be advisable to abort a DNS. For this reason
     * Nintendo provides a DNS request cancellation API via
     * the ResolverOption interface. The value for this key is a
     * unique DNS handle that is passed to the resolver that allows
     * a developer to stop the request from another thread. It is
     * still necessary to check the result from the resolver API to
     * determine whether the cancellation was successful.
     *
     * @code
     *
     * // Example use case of GetHostEntByName with cancellation on the
     * // DNS request thread
     *
     * ResolverOption option = {
     *     .key = RequestCancelHandleInteger,
     *     .type = ResolverOptionType_Integer,
     *     .size = sizeof(int),
     *     .data.integerValue = nn::socket::RequestCancelHandle()
     * };
     *
     * struct hostent* entry =
     *     nn::socket::GetHostEntByName(name, &option, 1);
     * if (NULL == entry && nn::socket::Errno::ECanceled == nn::socket::GetLastError())
     * {
     *      // ... request was cancelled
     * }
     *
     * @endcode
     */
    nnsocketResolverOptionKey_RequestCancelHandleInteger = 30002,

    /**
     * @brief This option contains a flag that enables or disables the
     * DNS cache for the given resolver request.
     *
     * @detail
     * By default this flag is true.
     *
     * @code
     * ResolverOption option = {
     *     .key = RequestEnableDnsCacheBoolean,
     *     .type = ResolverOptionType_Boolean,
     *     .size = sizeof(bool),
     *     .data.boolValue = false
     * };
     * @endcode
     */
    nnsocketResolverOptionKey_RequestEnableDnsCacheBoolean = 30003,

    /**
     * @brief
     * This constant defines the maximum boundary for Request key values.
     *
     * @details
     * Boundaries are not valid key values and produce an error if used.
     */
    nnsocketResolverOptionKey_RequestMaximumValue = 30004,
}  nnsocketResolverOptionKey;


/**
 * @brief this enum lists all the ResolverOption value types
 *
 * @details the ResolverOptions Container union is used to contain values of
 * multiple types.
 *
 * To reduce error:
 * - With the exception of strings the ResolverOption size
 *   is compared against the size of the type.
 *
 * - strings are compared to the string length
 *
 * - The @ref ResolverOptionKey key also contains the type name
 *
 * For example:
 *
 * -- ResolverOptionKey_RequestEnableServiceDiscoveryBoolean --> ResolverOptionType_Boolean
 * -- ResolverOptionKey_GetCancelHandleInteger --> ResolverOptionType_Integer
 *
 * More generally:
 * -- ResolverOptionKey_OptionNameAndTypeName --> ResolverOptionType_TypeName
 */
typedef enum  nnsocketResolverOptionType
{
    /**
     * @brief the data value is a boolean
     *
     * @details valid sizes are sizeof(bool)
     */
    nnsocketResolverOptionType_Boolean = 100,

    /**
     * @brief the data value is an integer
     *
     * @details valid size: sizeof(int)
     */
    nnsocketResolverOptionType_Integer = 101,

    /**
     * @brief the data value is an unsigned 32 bit integer
     *
     * @details valid size: sizeof(uint32_t)
     */
    nnsocketResolverOptionType_Unsigned32 = 102,

    /**
     * @brief the data value is an unsigned 64 bit integer
     *
     * @details valid size: sizeof(uint64_t)
     */
    nnsocketResolverOptionType_Unsigned64 = 103,

    /**
     * @brief the data value is a double
     *
     * @details valid size: sizeof(double)
     */
    nnsocketResolverOptionType_Double = 104,

    /**
     * @brief the data value is a string or pointer value
     *
     * @details The value is a pointer such as a string. The size
     * parameter reflects the size of the pointer.
     */
    nnsocketResolverOptionType_Pointer = 105,
}  nnsocketResolverOptionType;

/**
 * @brief structure for resolver options
 *
 * @details provided in in @ref ResolverOptionKey and @ref ResolverOptionValue
 */
typedef struct nnsocketResolverOption
{
    /**
     * @brief the resolver option key corresponding to this option
     *
     * @details see @ref ResolverOptionKey
     */
    nnsocketResolverOptionKey key;

    /**
     * @brief the resolver option type corresponding to this option
     *
     * @detail see @ref ResolverOptionType
     */
    nnsocketResolverOptionType type;

    /**
     * @brief the resolver option size
     *
     * @detail proper sizing for each @ref ResolverOptionType are listed below
     *
     * - ResolverOptionType_Boolean    = sizeof(bool)
     * - ResolverOptionType_Integer    = sizeof(int)
     * - ResolverOptionType_Unsigned32 = sizeof(uint32_t)
     * - ResolverOptionType_Unsigned64 = sizeof(uint64_t)
     * - ResolverOptionType_Double     = sizeof(double)
     * - ResolverOptionType_Pointer    = data length or string length
     *                                   from @ref nn::util::Strnlen
     */
    size_t size;

    /**
     * @brief the resolver option container
     * @note all Container data values have corresponding type information in
     * the key & type and the size argument also implicitly corresponds to the
     * size of the value or length string (minus the NULL-terminator)
     */
    union Container
    {
        bool        booleanValue;
        int         integerValue;
        uint32_t    unsigned32Value;
        uint64_t    unsigned64Value;
        double      doubleValue;
        const char* pointerValue;
    }
    data;
} nnsocketResolverOption;

