﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   メッセージキューに関する C 言語用 API の宣言
 */

#pragma once

#include <nnc/nn_Macro.h>
#include <nnc/os/os_MessageQueueTypes.h>
#include <nnc/os/os_MultipleWaitTypes.h>
#include <nnc/os/os_MultipleWaitCommon.h>

//--------------------------------------------------------------------------
//  C 向けの宣言
//--------------------------------------------------------------------------

//! @name メッセージキュー関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューオブジェクトを初期化します。
 *
 * @details
 *  詳細は nn::os::InitializeMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosInitializeMessageQueue(nnosMessageQueueType* messageQueue, uintptr_t* buffer, size_t count);


//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューオブジェクトをファイナライズします。
 *
 * @details
 *  詳細は nn::os::FinalizeMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosFinalizeMessageQueue(nnosMessageQueueType* messageQueue);


//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの最後尾にデータを１つ送信します。
 *
 * @details
 *  詳細は nn::os::SendMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosSendMessageQueue(nnosMessageQueueType* messageQueue, uintptr_t data);

//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの最後尾にデータを１つ送信します。
 *
 * @details
 *  詳細は nn::os::TrySendMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C bool    nnosTrySendMessageQueue(nnosMessageQueueType* messageQueue, uintptr_t data);

//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの最後尾にデータを１つ送信します。
 *
 * @details
 *  詳細は nn::os::TimedSendMessageQueue() を参照して下さい。@n
 *  本 API では、タイムアウト時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C bool    nnosTimedSendMessageQueue(nnosMessageQueueType* messageQueue, uintptr_t data, int64_t timeout);


//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの先頭にデータを１つ送信します。
 *
 * @details
 *  詳細は nn::os::JamMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosJamMessageQueue(nnosMessageQueueType* messageQueue, uintptr_t data);

//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの先頭にデータを１つ送信します。
 *
 * @details
 *  詳細は nn::os::TryJamMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C bool    nnosTryJamMessageQueue(nnosMessageQueueType* messageQueue, uintptr_t data);

//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの先頭にデータを１つ送信します。
 *
 * @details
 *  詳細は nn::os::TimedJamMessageQueue() を参照して下さい。@n
 *  本 API では、タイムアウト時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C bool    nnosTimedJamMessageQueue(nnosMessageQueueType* messageQueue, uintptr_t data, int64_t timeout);


//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの先頭からデータを１つ受信します。
 *
 * @details
 *  詳細は nn::os::ReceiveMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosReceiveMessageQueue(uintptr_t* outData, nnosMessageQueueType* messageQueue);

//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの先頭からデータを１つ受信します。
 *
 * @details
 *  詳細は nn::os::TryReceiveMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C bool    nnosTryReceiveMessageQueue(uintptr_t* outData, nnosMessageQueueType* messageQueue);

//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの先頭からデータを１つ受信します。
 *
 * @details
 *  詳細は nn::os::TimedReceiveMessageQueue() を参照して下さい。@n
 *  本 API では、タイムアウト時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C bool    nnosTimedReceiveMessageQueue(uintptr_t* outData, nnosMessageQueueType* messageQueue, int64_t timeout);


//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの先頭データを参照します。
 *
 * @details
 *  詳細は nn::os::PeekMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosPeekMessageQueue(uintptr_t* outData, const nnosMessageQueueType* messageQueue);

//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの先頭データを参照します。
 *
 * @details
 *  詳細は nn::os::TryPeekMessageQueue() を参照して下さい。
 *
 */
NN_OS_EXTERN_C bool    nnosTryPeekMessageQueue(uintptr_t* outData, const nnosMessageQueueType* messageQueue);

//--------------------------------------------------------------------------
/**
 * @brief   メッセージキューの先頭データを参照します。
 *
 * @details
 *  詳細は nn::os::TimedPeekMessageQueue() を参照して下さい。@n
 *  本 API では、タイムアウト時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C bool    nnosTimedPeekMessageQueue(uintptr_t* outData, const nnosMessageQueueType* messageQueue, int64_t timeout);

//! @}

//! @name 多重待ち関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder を初期化し、messageQueue と関連付けます。
 *
 * @details
 *  詳細は nn::os::InitializeMultiWaitHolder(MultiWaitHolderType* multiWaitHolder, MessageQueueType* messageQueue, MessageQueueWaitType waitType) を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosInitializeMultiWaitHolderForMessageQueue(nnosMultiWaitHolderType* multiWaitHolder, nnosMessageQueueType* messageQueue, enum nnosMessageQueueWaitType waitType);

//! @}
