﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief デバイスコード関連の C リンケージ向け API 宣言
 */

#include <nnc/nn_Common.h>
#include <nnc/nn_Macro.h>
#include <nn/devicecode/detail/devicecode_InnerFormat.h>

#ifdef __cplusplus
extern "C"
{
#endif

typedef uint32_t nndetailDeviceCodeType;

/**
 * @brief デバイスコードのカテゴリを表す型です。
 */
typedef uint32_t nnDeviceCodeCategory;

/**
 * @brief デバイスコードの InnerID を表す型です。
 */
typedef uint32_t nnDeviceCodeInnerId;

/**
 * @brief デバイスコードです。 @ref nn::DeviceCode クラスに相当します。 メンバには直接アクセスしないでください。
 */
typedef struct nnDeviceCode
{
    nndetailDeviceCodeType _value;
} nnDeviceCode;

// C 言語向けにのみ有効な関数群
#if !defined(__cplusplus) || defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

/**
 * @brief デバイスコードを生成します。
 */
static NNC_FORCEINLINE nnDeviceCode nnDeviceCodeMakeCode(nndetailDeviceCodeType value)
{
    nnDeviceCode code;
    code._value = value;
    return code;
}

/**
 * @brief デバイスコード同士を比較します。一致する場合 true を返します。
 */
static NNC_FORCEINLINE bool nnDeviceCodeIsEqual(nnDeviceCode lhs, nnDeviceCode rhs)
{
    return lhs._value == rhs._value;
}

/**
 * @brief 無効なデバイスコードを明示的に生成します。
 */
static NNC_FORCEINLINE nnDeviceCode nnDeviceCodeMakeInvalidCode()
{
    nnDeviceCode code;
    code._value = 0ul;
    return code;
}

/**
 * @brief デバイスコードのカテゴリを返します。
 */
static NNC_FORCEINLINE nnDeviceCodeCategory nnDeviceCodeGetCategory(nnDeviceCode code)
{
    return NN_DETAIL_DEVICECODE_GET_CATEGORY(code._value);
}

/**
 * @brief デバイスコードの InnerID を返します。
 */
static NNC_FORCEINLINE nnDeviceCodeInnerId nnDeviceCodeGetInnerId(nnDeviceCode code)
{
    return NN_DETAIL_DEVICECODE_GET_INNERID(code._value);
}

/**
 * @brief デバイスコードが指定したカテゴリに属するかを返します。
 */
static NNC_FORCEINLINE bool nnDeviceCodeIsCategory(nnDeviceCode code, nnDeviceCodeCategory category)
{
    return nnDeviceCodeGetCategory(code) == category;
}

#endif // !defined(__cplusplus) || defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

#ifdef __cplusplus
}
#endif
