﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <utility>
#include <memory>
#include <mutex>

namespace nn { namespace util {

template <typename Mutex>
class UniqueLock
{
private:

    Mutex* m_pMutex;
    bool m_OwnsLock;

public:

    using mutex_type = Mutex;

    UniqueLock() NN_NOEXCEPT
        : m_pMutex(nullptr)
        , m_OwnsLock(false)
    {
    }

    UniqueLock(UniqueLock&& other) NN_NOEXCEPT
        : m_pMutex(std::exchange(other.m_pMutex, nullptr))
        , m_OwnsLock(std::exchange(other.m_OwnsLock, false))
    {
    }

    explicit UniqueLock(mutex_type& mutex)
        : m_pMutex(std::addressof(mutex))
        , m_OwnsLock(true)
    {
        mutex.lock();
    }

    UniqueLock(mutex_type& mutex, std::defer_lock_t) NN_NOEXCEPT
        : m_pMutex(std::addressof(mutex))
        , m_OwnsLock(false)
    {
    }

    UniqueLock(mutex_type& mutex, std::try_to_lock_t) NN_NOEXCEPT
        : m_pMutex(std::addressof(mutex))
        , m_OwnsLock(mutex.try_lock())
    {
    }

    UniqueLock(mutex_type& mutex, std::adopt_lock_t) NN_NOEXCEPT
        : m_pMutex(std::addressof(mutex))
        , m_OwnsLock(true)
    {
    }

    ~UniqueLock() NN_NOEXCEPT
    {
        if (m_OwnsLock)
        {
            NN_SDK_ASSERT_NOT_NULL(m_pMutex);
            m_pMutex->unlock();
        }
    }

    UniqueLock& operator=(UniqueLock&& rhs) NN_NOEXCEPT
    {
        UniqueLock(std::move(rhs)).swap(*this);
        return *this;
    }

    void swap(UniqueLock& other) NN_NOEXCEPT
    {
        std::swap(this->m_pMutex, other.m_pMutex);
        std::swap(this->m_OwnsLock, other.m_OwnsLock);
    }

    mutex_type* release() NN_NOEXCEPT
    {
        this->m_OwnsLock = false;
        return std::exchange(this->m_pMutex, nullptr);
    }

    mutex_type* mutex() const NN_NOEXCEPT
    {
        return m_pMutex;
    }

    bool owns_lock() const NN_NOEXCEPT
    {
        return m_OwnsLock;
    }

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return m_OwnsLock;
    }

    void lock()
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pMutex);
        NN_SDK_REQUIRES(!m_OwnsLock);
        m_pMutex->lock();
        this->m_OwnsLock = true;
    }

    bool try_lock()
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pMutex);
        NN_SDK_REQUIRES(!m_OwnsLock);
        this->m_OwnsLock = m_pMutex->try_lock();
        return m_OwnsLock;
    }

    void unlock() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pMutex);
        NN_SDK_REQUIRES(m_OwnsLock);
        m_pMutex->unlock();
        this->m_OwnsLock = false;
    }

};

template <typename Mutex>
inline void swap(UniqueLock<Mutex>& x, UniqueLock<Mutex>& y) NN_NOEXCEPT
{
    x.swap(y);
}

template <typename Mutex, typename... Args>
inline UniqueLock<Mutex> MakeUniqueLock(Mutex& mutex, Args&&... args)
{
    return UniqueLock<Mutex>(mutex, std::forward<Args>(args)...);
}

}}
