﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_Result.h>

#include <nn/tsc.h>
#include <nn/tsc/tsc_Types.h>
#include <nn/tsc/tsc_ActiveConfigContext.h>

namespace nn { namespace tsc {

//--------------------------------------------------------------------------
/**
*   @brief    The class to obtain IP information in use inside underlying TCP/IP stack
*
*   @details
*       To obtain IP information currently in use by underlying TCP/IP stack, user is required
*       to issue a request by the mothod provided by this class.
*       After initializing TSC library by @ref Initialize, user will first configure IP
*       information through the object of @ref nn::tsc::Ipv4ConfigContext.
*       After configuration is done, user will create a object of this class and set network
*       interface name by @ref SetInterfaceName. Note that appropriate network interface name
*       needs to be obtained by the interface provided by network interface driver.
*       Each IP information will be obtained by methods provided in this class afterward.
*/
class ActiveConfigContext
{
private:
    char m_InterfaceName[g_MaxInterfaceNameLength];

public:

    /**
    * @brief    The statistics of the interface used by GetInterfaceStatistics().
    */
    struct InterfaceStatistics
    {
        nn::Result interfaceStatus;       //!< The status of the network interface
        struct DhcpStatistics
        {
            uint32_t   renewedLeaseCount; //!< The number of DHCP renew count
            uint32_t   reboundLeaseCount; //!< The number of DHCP rebind count
        } dhcp;
    };

    explicit ActiveConfigContext(const char* pInInterfaceName);
    ~ActiveConfigContext();

    //--------------------------------------------------------------------------
    /**
    *   @brief      Set network interface name.
    *   @details
    *               This API specifies the network interface where IP information is obtained from.
    *               The interface set with this API needs to be obtained by the API of network
    *               interface driver. This API needs to be called before calling other API
    *               in this class.
    *   @pre        User has already called tsc::Initialize.
    *   @post
    *
    *   @param[in] pInInterfaceName    Pointer to the memory where interface name is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Passed interface name is too long
    */
    Result SetInterfaceName(const char* pInInterfaceName) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get interface name stored locally inside TSC.
    *   @details
    *               This API obtains network interface name which is stored inside this class.
    *               The name which is set by @ref SetInterfaceName will be obtaitned.
    *   @pre        User has already called tsc::Initialize.
    *   @post
    *
    *   @param[out] pOutInterfaceName    Pointer to the memory where obtained interface name is stored
    *   @param[in]  nameBufferLen        Length of pInterfaceName
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultBufferTooShort           Passed buffer length (@ref nameBufferLen) is too short
    */
    Result GetInterfaceName(char* pOutInterfaceName, int nameBufferLen) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get active interface address of the network interface.
    *   @details
    *               This API obtains IP address of network interface in use from underlying
    *               TCP/IP stack.
    *   @pre        User has already called tsc::Initialize and set network interface name
    *               by @ref SetInterfaceName.
    *   @post
    *
    *   @param[out] pOutAddress    Pointer to the memory where obtained interface address is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Interface name is not specified yet or invalid
    */
    Result GetInterfaceAddress(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get active subnet mask of the network interface.
    *   @details
    *               This API obtains subnet mask of network interface in use from underlying
    *               TCP/IP stack.
    *   @pre        User has already called tsc::Initialize and set network interface name
    *               by @ref SetInterfaceName.
    *   @post
    *
    *   @param[out] pOutAddress    Pointer to the memory where obtained subnet mask is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Interface name is not specified yet or invalid
    */
    Result GetSubnetMask(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get active default gateway address of the network interface.
    *   @details
    *               This API obtains default gateway address of network interface in use from underlying
    *               TCP/IP stack.
    *   @pre        User has already called tsc::Initialize and set network interface name
    *               by @ref SetInterfaceName.
    *   @post
    *
    *   @param[out] pOutAddress    Pointer to the memory where obtained default gateway address is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Interface name is not specified yet or invalid
    */
    Result GetDefaultGateway(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get active preferred DNS address of the network interface.
    *   @details
    *               This API obtains preferred DNS address of network interface in use from
    *               underlying TCP/IP stack.
    *   @pre        User has already called tsc::Initialize and set network interface name
    *               by @ref SetInterfaceName.
    *   @post
    *
    *   @param[out] pOutAddress    Pointer to the memory where obtained preferred DNS address is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Interface name is not specified yet or invalid
    */
    Result GetPreferredDns(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get active alternative DNS address of the network interface.
    *   @details
    *               This API obtains alternative DNS address of network interface in use from
    *               underlying TCP/IP stack.
    *   @pre        User has already called tsc::Initialize and set network interface name
    *               by @ref SetInterfaceName.
    *   @post
    *
    *   @param[out] pOutAddress    Pointer to the memory where obtained alternative DNS address is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Interface name is not specified yet or invalid
    */
    Result GetAlternativeDns(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get active MTU size of the network interface.
    *               This API obtains MTU value of network interface in use from underlying
    *               TCP/IP stack.
    *   @pre        User has already called tsc::Initialize and set network interface name
    *               by @ref SetInterfaceName.
    *   @post
    *
    *   @param[out] pOutMtuValue    Pointer to the memory where obtained MTU value is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Interface name is not specified yet or invalid
    */
    Result GetMtu(uint32_t* pOutMtuValue) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get the interface statistics.
    *               This API obtains the statistics of the interface. This API is supposed to be
    *               called when the system event which can be obtained by
    *               nn::tsc::ConfigContext::GetSystemEventPointer() is signaled.
    *
    *   @pre        User has already called tsc::Initialize and set network interface name
    *               by @ref SetInterfaceName.
    *   @post
    *
    *   @param[out] pOutStatsValue    Pointer to the memory where obtained interface statistics is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Interface name is not specified yet or invalid
    */
    Result GetInterfaceStatistics(InterfaceStatistics* pOutStatsValue) NN_NOEXCEPT;
};

}}
