﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   socket library configuration for use by system
 */

#pragma once

#include <stdint.h>
#include <nn/socket/socket_Types.h>
#include <nn/socket/socket_Config.h>
#include <nn/socket/socket_Constants.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

namespace nn     {
namespace socket {


/**
 * @brief    Class allowing for full manual configuration of the socket library. Memory pool is included.
 * @param[in] tcpSocketCountMax
 *     Maximum number of TCP sockets for which memory is to be pre-allocated.
 * @param[in] udpSocketCountMax
 *     Maximum number of UDP sockets for which memory is to be pre-allocated.
 * @param[in] tcpInitialSendBufferSize
 *     Base size of socket TCP send buffers.
 * @param[in] tcpInitialReceiveBufferSize
 *     Base size of TCP receive buffers.
 * @param[in] tcpAutoSendBufferSizeMax
 *     This parameter controls the upper bound to which the TCP send buffer would be allowed to grow.
 *     Set to '0' to disable TCP send buffer auto tuning.
 * @param[in] tcpAutoReceiveBufferSizeMax
 *     This parameter controls the upper bound to which the TCP receive buffer would be allowed to grow.
 *     Set to '0' to disable TCP receive buffer auto tuning.
 * @param[in] udpSendBufferSize
 *     Base size of socket UDP send buffers.
 * @param[in] udpReceiveBufferSize
 *     Base size of socket UDP receive buffers.
 * @param[in] socketBufferEfficiency
 *     Buffer multiplying factor by which socket buffer sizes are scaled up.
 *     The valid range for this parameter is from 1 to 8, inclusive.
 *     A value of 1 may sacrifice some aspects of performance for reduced memory use.
 *     A value of 2 will provide a reasonable tradeoff between performance and memory use.
 *     Applications with demanding performance requirements should consider using a value of at least 4.
 *     This parameter is used to provision FreeBSD's internal "sb_efficiency" setting.
 * @param[in] allocatorPoolSize
 *     Amount of memory which is to be reserved for internal use by the socket library allocator.
 *     The following API make use of this memory reservation: @ref nn::socket::GetAddrInfo,
 *     @ref nn::socket::GetHostByAddr, @ref nn::socket::GetHostByName, @ref nn::socket::GetNameInfo.
 * @details
 *     It must be instantiated with global or static scope. As this class contains a memory pool,
 *     when instantiated the resulting object may consume several megabytes of memory.
 */
template<int    tcpSocketCountMax,        int    udpSocketCountMax,
         size_t tcpInitialSendBufferSize, size_t tcpInitialReceiveBufferSize,
         size_t tcpAutoSendBufferSizeMax, size_t tcpAutoReceiveBufferSizeMax,
         size_t udpSendBufferSize,        size_t udpReceiveBufferSize,
         int    socketBufferEfficiency,   size_t allocatorPoolSize>
class SystemConfigWithMemory : public Config
{
public:
    static const size_t PerTcpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
                NN_DETAIL_SOCKET_MAX(tcpInitialSendBufferSize, tcpAutoSendBufferSizeMax)
            ) * socketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
               NN_DETAIL_SOCKET_MAX(tcpInitialReceiveBufferSize, tcpAutoReceiveBufferSizeMax)
            ) * socketBufferEfficiency
        );
    static const size_t PerUdpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(udpSendBufferSize) * socketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(udpReceiveBufferSize) * socketBufferEfficiency
        );
    static const size_t AllocatorPoolSize = NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE(allocatorPoolSize);

    NN_IMPLICIT SystemConfigWithMemory(int concurrencyCountMax) NN_NOEXCEPT
        : Config(m_MemoryPool, sizeof(m_MemoryPool), AllocatorPoolSize,
                 tcpInitialSendBufferSize,
                 tcpInitialReceiveBufferSize,
                 tcpAutoSendBufferSizeMax,
                 tcpAutoReceiveBufferSizeMax,
                 udpSendBufferSize,
                 udpReceiveBufferSize,
                 socketBufferEfficiency,
                 concurrencyCountMax)
    {
        NN_STATIC_ASSERT(tcpSocketCountMax >= 0);
        NN_STATIC_ASSERT(udpSocketCountMax >= 0);
        NN_STATIC_ASSERT((udpSocketCountMax != 0) || (tcpSocketCountMax != 0));
        NN_STATIC_ASSERT((socketBufferEfficiency >= 1) && (socketBufferEfficiency <= 8));
        m_IsSystemClient = true;
    }
private:
#if defined NN_BUILD_CONFIG_OS_WIN32
    uint8_t m_MemoryPool[1];
#else
    uint8_t m_MemoryPool[(PerTcpSocketWorstCaseMemoryPoolSize * tcpSocketCountMax) +
                         (PerUdpSocketWorstCaseMemoryPoolSize * udpSocketCountMax) +
                         AllocatorPoolSize] NN_ALIGNAS(nn::socket::MemoryPoolAlignment);
#endif
};


/**
 * @brief    Class specifying a default configuration. No memory pool is included.
 *
 * @details
 *           This class should be used directly only if the application has the need
 *           to customize memory parameters. Consider using @ref SystemConfigDefaultWithMemory
 *           instead.
 */
class SystemConfigDefault : public Config
{
public:
    static const size_t DefaultTcpInitialSendBufferSize    = 1024 * 32;
    static const size_t DefaultTcpInitialReceiveBufferSize = 1024 * 64;
    static const size_t DefaultTcpAutoSendBufferSizeMax    = 1024 * 256;
    static const size_t DefaultTcpAutoReceiveBufferSizeMax = 1024 * 256;
    static const size_t DefaultUdpSendBufferSize           = 1024 * 9;
    static const size_t DefaultUdpReceiveBufferSize        = 42240;
    static const int    DefaultSocketBufferEfficiency      = 2;
    static const int    DefaultConcurrency                 = 8;
    static const size_t DefaultAllocatorPoolSize           = 1024 * 128;
    static const size_t PerTcpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
                NN_DETAIL_SOCKET_MAX(DefaultTcpInitialSendBufferSize, DefaultTcpAutoSendBufferSizeMax)
            ) * DefaultSocketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
               NN_DETAIL_SOCKET_MAX(DefaultTcpInitialReceiveBufferSize, DefaultTcpAutoReceiveBufferSizeMax)
            ) * DefaultSocketBufferEfficiency
        );
    static const size_t PerUdpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(DefaultUdpSendBufferSize) * DefaultSocketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(DefaultUdpReceiveBufferSize) * DefaultSocketBufferEfficiency
        );

    SystemConfigDefault(void* pMemoryPool, size_t memoryPoolSize, size_t allocatorPoolSize,
                        int concurrencyCountMax=DefaultConcurrency) NN_NOEXCEPT
        : Config(pMemoryPool, memoryPoolSize, allocatorPoolSize,
                 DefaultTcpInitialSendBufferSize,
                 DefaultTcpInitialReceiveBufferSize,
                 DefaultTcpAutoSendBufferSizeMax,
                 DefaultTcpAutoReceiveBufferSizeMax,
                 DefaultUdpSendBufferSize,
                 DefaultUdpReceiveBufferSize,
                 DefaultSocketBufferEfficiency,
                 concurrencyCountMax)
    {
        m_IsSystemClient = true;
    }
};

/**
 * @brief    Class specifying a performance oriented configuration. Memory pool is included.
 * @param[in] tcpSocketCountMax
 *     Maximum number of TCP sockets for which memory is to be pre-allocated.
 * @param[in] udpSocketCountMax
 *     Maximum number of UDP sockets for which memory is to be pre-allocated.
 * @param[in] socketBufferEfficiency
 *     Buffer multiplying factor by which socket buffer sizes are scaled up.
 *     The valid range for this parameter is from 1 to 8, inclusive.
 *     A value of 1 may sacrifice some aspects of performance for reduced memory use.
 *     A value of 2 will provide a reasonable tradeoff between performance and memory use.
 *     Applications with demanding performance requirements should consider using a value of at least 4.
 *     This parameter is used to provision FreeBSD's internal "sb_efficiency" setting.
 * @details
 *     It must be instantiated with global or static scope. As this class contains a memory pool,
 *     when instantiated the resulting object may consume several megabytes of memory.
 */
template<int tcpSocketCountMax, int udpSocketCountMax,
         int socketBufferEfficiency=SystemConfigDefault::DefaultSocketBufferEfficiency>
class SystemConfigDefaultWithMemory : public SystemConfigDefault
{
public:
    static const size_t PerTcpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
                NN_DETAIL_SOCKET_MAX(DefaultTcpInitialSendBufferSize, DefaultTcpAutoSendBufferSizeMax)
            ) * socketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
               NN_DETAIL_SOCKET_MAX(DefaultTcpInitialReceiveBufferSize, DefaultTcpAutoReceiveBufferSizeMax)
            ) * socketBufferEfficiency
        );
    static const size_t PerUdpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(DefaultUdpSendBufferSize) * socketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(DefaultUdpReceiveBufferSize) * socketBufferEfficiency
        );

    SystemConfigDefaultWithMemory() NN_NOEXCEPT
        : SystemConfigDefault(m_MemoryPool, sizeof(m_MemoryPool), DefaultAllocatorPoolSize, DefaultConcurrency)
    {
        CheckParameters();
    }
    NN_IMPLICIT SystemConfigDefaultWithMemory(int concurrencyCountMax) NN_NOEXCEPT
        : SystemConfigDefault(m_MemoryPool,
                              sizeof(m_MemoryPool),
                              DefaultAllocatorPoolSize,
                              concurrencyCountMax)
    {
        CheckParameters();
    }

private:
    void CheckParameters()
    {
        NN_STATIC_ASSERT(tcpSocketCountMax >= 0);
        NN_STATIC_ASSERT(udpSocketCountMax >= 0);
        NN_STATIC_ASSERT((udpSocketCountMax != 0) || (tcpSocketCountMax != 0));
        NN_STATIC_ASSERT((socketBufferEfficiency >= 1) && (socketBufferEfficiency <= 8));
    }

#if defined NN_BUILD_CONFIG_OS_WIN32
    uint8_t m_MemoryPool[1];
#else
    uint8_t m_MemoryPool[(PerTcpSocketWorstCaseMemoryPoolSize * tcpSocketCountMax) +
                         (PerUdpSocketWorstCaseMemoryPoolSize * udpSocketCountMax) +
                         DefaultAllocatorPoolSize] NN_ALIGNAS(nn::socket::MemoryPoolAlignment);
#endif
};

/**
 * @brief    Class specifying a memory saving but less performant configuration. No memory pool is included.
 * @details
 *           This class should be used directly only if the application has the need
 *           to customize memory parameters. Consider using @ref SystemConfigLightDefaultWithMemory
 *           instead.
 */
class SystemConfigLightDefault : public Config
{
public:
    static const size_t DefaultTcpInitialSendBufferSize    = 1024 * 16;
    static const size_t DefaultTcpInitialReceiveBufferSize = 1024 * 32;
    static const size_t DefaultTcpAutoSendBufferSizeMax    = 0;
    static const size_t DefaultTcpAutoReceiveBufferSizeMax = 0;
    static const size_t DefaultUdpSendBufferSize           = 1024 * 9;
    static const size_t DefaultUdpReceiveBufferSize        = 42240;
    static const int    DefaultSocketBufferEfficiency      = 2;
    static const int    DefaultConcurrency                 = 2;
    static const size_t DefaultAllocatorPoolSize           = 1024 * 64;
    static const size_t PerTcpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
                NN_DETAIL_SOCKET_MAX(DefaultTcpInitialSendBufferSize, DefaultTcpAutoSendBufferSizeMax)
            ) * DefaultSocketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
               NN_DETAIL_SOCKET_MAX(DefaultTcpInitialReceiveBufferSize, DefaultTcpAutoReceiveBufferSizeMax)
            ) * DefaultSocketBufferEfficiency
        );
    static const size_t PerUdpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(DefaultUdpSendBufferSize) * DefaultSocketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(DefaultUdpReceiveBufferSize) * DefaultSocketBufferEfficiency
        );

    SystemConfigLightDefault(void* pMemoryPool, size_t memoryPoolSize, size_t allocatorPoolSize,
                             int concurrencyCountMax = DefaultConcurrency) NN_NOEXCEPT
        : Config(pMemoryPool, memoryPoolSize, allocatorPoolSize,
                 DefaultTcpInitialSendBufferSize,
                 DefaultTcpInitialReceiveBufferSize,
                 DefaultTcpAutoSendBufferSizeMax,
                 DefaultTcpAutoReceiveBufferSizeMax,
                 DefaultUdpSendBufferSize,
                 DefaultUdpReceiveBufferSize,
                 DefaultSocketBufferEfficiency,
                 concurrencyCountMax)
    {
        m_IsSystemClient = true;
    }
};

/**
 * @brief    Class specifying a memory saving but less performant configuration. Memory pool is included.
 * @param[in] tcpSocketCountMax
 *     Maximum number of TCP sockets for which memory is to be pre-allocated.
 * @param[in] udpSocketCountMax
 *     Maximum number of UDP sockets for which memory is to be pre-allocated.
 * @param[in] socketBufferEfficiency
 *     Buffer multiplying factor by which socket buffer sizes are scaled up.
 *     The valid range for this parameter is from 1 to 8, inclusive.
 *     A value of 1 may sacrifice some aspects of performance for reduced memory use.
 *     A value of 2 will provide a reasonable tradeoff between performance and memory use.
 *     Applications with demanding performance requirements should consider using a value of at least 4.
 *     This parameter is used to provision FreeBSD's internal "sb_efficiency" setting.
 * @details
 *     It must be instantiated with global or static scope as this class contains a memory pool.
 */
template<int tcpSocketCountMax, int udpSocketCountMax,
         int socketBufferEfficiency=SystemConfigLightDefault::DefaultSocketBufferEfficiency>
class SystemConfigLightDefaultWithMemory : public SystemConfigLightDefault
{
public:
    static const size_t PerTcpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
                NN_DETAIL_SOCKET_MAX(DefaultTcpInitialSendBufferSize, DefaultTcpAutoSendBufferSizeMax)
            ) * socketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS
            (
               NN_DETAIL_SOCKET_MAX(DefaultTcpInitialReceiveBufferSize, DefaultTcpAutoReceiveBufferSizeMax)
            ) * socketBufferEfficiency
        );
    static const size_t PerUdpSocketWorstCaseMemoryPoolSize =
        NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_PAGE
        (
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(DefaultUdpSendBufferSize) * socketBufferEfficiency
            +
            NN_DETAIL_SOCKET_BUFFER_ROUNDUP_TO_MSS(DefaultUdpReceiveBufferSize) * socketBufferEfficiency
        );

    SystemConfigLightDefaultWithMemory() NN_NOEXCEPT
        : SystemConfigLightDefault(m_MemoryPool, sizeof(m_MemoryPool), DefaultAllocatorPoolSize)
    {
        CheckParameters();
    }
    NN_IMPLICIT SystemConfigLightDefaultWithMemory(int concurrencyCountMax) NN_NOEXCEPT
        : SystemConfigLightDefault(m_MemoryPool, sizeof(m_MemoryPool), DefaultAllocatorPoolSize,
                                   concurrencyCountMax)
    {
        CheckParameters();
    }
private:
    void CheckParameters()
    {
        NN_STATIC_ASSERT(tcpSocketCountMax >= 0);
        NN_STATIC_ASSERT(udpSocketCountMax >= 0);
        NN_STATIC_ASSERT((udpSocketCountMax != 0) || (tcpSocketCountMax != 0));
        NN_STATIC_ASSERT((socketBufferEfficiency >= 1) && (socketBufferEfficiency <= 8));
    }

    #if defined NN_BUILD_CONFIG_OS_WIN32
    uint8_t m_MemoryPool[1];
    #else
    uint8_t m_MemoryPool[(PerTcpSocketWorstCaseMemoryPoolSize * tcpSocketCountMax) +
                         (PerUdpSocketWorstCaseMemoryPoolSize * udpSocketCountMax) +
                         DefaultAllocatorPoolSize] NN_ALIGNAS(nn::socket::MemoryPoolAlignment);
    #endif
};

/**
 * @brief This class is for clients that wish to use the network stack for BPF only purposes.
 * @details It allocates the absolute minimum amount of memory to TCP and UDP.
 */
class BpfOnlyConfig :
public Config
{
public:
    BpfOnlyConfig()  NN_NOEXCEPT
        : Config(m_MemoryPool, sizeof(m_MemoryPool), nn::os::MemoryPageSize, // pMemoryPool, memoryPoolSize, allocatorPoolSize,
                 nn::os::MemoryPageSize, nn::os::MemoryPageSize,             // tcpInitialSendBufferSize, tcpInitialReceiveBufferSize,
                 nn::os::MemoryPageSize, nn::os::MemoryPageSize,             // tcpAutoSendBufferSizeMax, tcpAutoReceiveBufferSizeMax,
                 nn::os::MemoryPageSize, nn::os::MemoryPageSize,             // udpSendBufferSize,        udpReceiveBufferSize,
                                      1,                     1)              // socketBufferEfficiency,   concurrencyCountMax)
    {
        m_IsSystemClient = true;
    };

private:
#if defined NN_BUILD_CONFIG_OS_WIN32
    uint8_t m_MemoryPool[1];
#else
    uint8_t m_MemoryPool[6 * nn::os::MemoryPageSize] NN_ALIGNAS(nn::socket::MemoryPoolAlignment);
#endif
};

}} /* nn::socket */
