﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <stdint.h>

/**
 * This file contains private and local resolver options
 */

namespace nn { namespace socket {

/**
 * @brief This enum class defines a set of private ResolverOptionKey
 * values for Get, Set, and Request functions but are not part of
 * the public SDK.
 */
enum class ResolverOptionPrivateKey : uint32_t
{
    /**
     * @brief
     * Defines the minimum boundary for Private key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    GetMinimumValue = 40000,

    /**
     * @brief
     * Option passed to @ref nn::socket::ResolverGetOption(). On
     * success returns the current DNS server addresses.
     *
     *@details
     * As You must set the pointer value to a structure of
     * @ref nn::socket::PrivateDnsAddressArrayArgument.
     *
     * @code
     * nn::socket::PrivateDnsAddressArrayArgument argument = { 0 };
     * nn::socket::ResolverOption option = {
     *     .key = (nn::socket::ResolverOptionKey) nn::socket::ResolverOptionPrivateKey::GetDnsServerAddressesPointer,
     *     .type = nn::socket::ResolverOptionType::Pointer,
     *     .size = sizeof(nn::socket::PrivateDnsAddressArrayArgument),
     *     .data.pointerValue = &argument
     * };
     *
     * result = nn::socket::ResolverGetOption(&option, option.key);
     * @endcode
     */
    GetDnsServerAddressesPointer = 40001,

    /**
     * @brief
     * Option passed to @ref nn::socket::ResolverGetOption().
     * As a pointer argument, you pass a domain name in and on success
     * it returns a count of matching entries in the cache.
     *
     * @code
     * const char* domainname = "www.nintendo.com";
     * nn::socket::ResolverOption option = {
     *    .key = (nn::socket:ResolverOptionKey) nn::socket::ResolverOptionPrivateKey::GetCacheEntryCountForDomainnamePointer;
     *    .type = nn::socket::ResolverOptionType::Pointer;
     *    .size = strlen(domainname) + 1;
     *    .data.pointerValue = domainname;
     * }
     * result = nn::socket::ResolverGetOption(&option, option.key);
     * @endcode
     */
    GetCacheEntryCountForDomainnamePointer = 40002,

    /**
     * @brief
     * Passed to @ref nn::socket::ResolverGetOption(). You pass an
     * ip address as a uint32_t and, on success, it returns an count
     * of matching entries the given address in the cache.
     *
     * @code
     * uint32_t ip;
     * nn::socket::InetPton(nn::socket::Family::Af_Inet, "1.2.3.4", &ip);
     * nn::socket::ResolverOption option = {
     *    .key = (nn::socket:ResolverOptionKey) nn::socket::ResolverOptionPrivateKey::GetCacheEntryCountForIpUnsigned32;
     *    .type = nn::socket::ResolverOptionType::Unsigned32;
     *    .size = sizeof(uint32_t),
     *    .data. = ip;
     * }
     * result = nn::socket::ResolverGetOption(&option, option.key);
     * @endcode
     */
    GetCacheEntryCountForIpUnsigned32 = 40003,

    /**
     * @brief
     * This constant defines the maximum boundary for Private key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    GetMaximumValue = 40004,


    /**
     * @brief
     * This constant defines the minimum boundary for Private key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    SetMinimumValue = 50000,

    /**
     * @brief
     * Passed to @ref nn::socket::ResolverSetOption() and, on success,
     * it sets the system DNS server addresses to those specified in
     * the provided argument.
     *
     *@details
     * To use this functionality you must set the pointer value to
     * a structure of @ref PrivateDnsAddressArrayArgument.
     *
     *@code
     * nn::socket::PrivateDnsAddressArrayArgument argument;
     * nn::socket::InetPton(AF_INET, "1.2.3.4", &argument.addresses.s_addr[0]);
     * argument.addresses[0].sin_port = nn::socket::InetHtons(53);
     * argument.count = 1;
     *
     * nn::socket::ResolverOption option = {
     *     .key = (nn::socket::ResolverOptionKey) nn::socket::ResolverOptionPrivateKey::SetDnsServerAddressesPointer,
     *     .type = nn::socket::ResolverOptionType::Pointer,
     *     .size = sizeof(nn::socket::PrivateDnsAddressArrayArgument),
     *     .data.pointerValue = &argument
     * };
     *
     * result = nn::socket::ResolverSetOption(&option);
     * @endcode
     */
    SetDnsServerAddressesPointer = 50001,

    /**
     * @brief
     * This constant option is passed to @ref
     * nn::socket::ResolverSetOption() and, if enabled
     * for the build, this function sets the current set of DNS
     * addresses.
     *
     *@details
     * To use this functionality you must set the pointer value to
     * a structure of @ref PrivateSetTimeToLiveForDomainnameArgument.
     *
     * @code
     * const char* domain = "www.nintendo.com";
     * nn::socket::PrivateSetTimeToLiveForDomainnameArgument argument;
     * argument.ttl = 60;
     * memset(argument.domainname, domain, strlen(domain)+1);
     * nn::socket::ResolverOption option = {
     *     .key = (nn::socket::ResolverOptionKey) nn::socket::ResolverOptionPrivateKey::SetTimeToLiveForDomainnamePointer,
     *     .type = nn::socket::ResolverOptionType::Pointer,
     *     .size = sizeof(nn::socket::PrivateSetTimeToLiveForDomainnameArgument),
     *     .data.pointerValue = &argument
     * };
     *
     * result = nn::socket::ResolverSetOption(option);
     * @endcode
     */
    SetTimeToLiveForDomainnamePointer = 50002,

    /**
     * @brief
     * This constant option is passed to @ref
     * nn::socket::ResolverSetOption()  and, if enabled
     * for the build, it will flush the entire DNS cache.
     *
     * @code
     * nn::socket::ResolverOption option = {
     *     .key = (nn::socket::ResolverOptionKey) nn::socket::ResolverOptionPrivateKey::SetFlushCacheBoolean,
     *     .type = nn::socket::ResolverOptionType::Boolean,
     *     .size = sizeof(bool),
     *     .data.boolValue = true
     * };
     *
     * nn::socket::ResolverSetOption(option);
     * @endcode
     */
    SetFlushCacheBoolean = 50003,

    /**
     * @brief
     * This constant defines the maximum boundary for Private key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    SetMaximumValue = 50004,

    /**
     * @brief
     * This constant defines the minimum boundary for Local key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    RequestMinimumValue = 60000,

    /**
     * @brief
     * This constant defines the maximum boundary for Local key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    RequestMaximumValue = 60001,
};

/**
 * @brief This set of values are set the private ResolverOptionKeys used
 * locally, i.e. within the context of a resolver client.
 */
enum class ResolverOptionLocalKey : uint32_t
{
    /**
     * @brief
     * This constant defines the minimum boundary for Local key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    MinimumValue = 70000,

    /**
     * @brief This constant is used with @ref nn::socket::ResolverGetOption()
     * and @ref nn::socket::ResolverSetOption().
     * The data value parameter is an unsigned 64-bit value (uint64_t)
     * and that that value is used to determine the size in bytes of the
     * service framework buffer used for @ref GetHostByName GetHostByName.
     */
    GetHostByNameBufferSizeUnsigned64  = 70001,

    /**
     * @brief This constant is used with @ref nn::socket::ResolverGetOption()
     * and @ref nn::socket::ResolverSetOption().
     * The data value parameter is an unsigned 64-bit value (uint64_t)
     * and that that value is used to determine the size in bytes of the
     * service framework buffer used for @ref GetHostByAddr.
     */
    GetHostByAddrBufferSizeUnsigned64  = 70002,

    /**
     * @brief This constant is used with @ref nn::socket::ResolverGetOption()
     * and @ref nn::socket::ResolverSetOption(). The data value
     * parameter is an unsigned 64-bit value (uint64_t) and that that
     * value is used to determine the size in bytes of the service
     * framework buffer used with @ref HStrError
     */
    GetHostErrorStringBufferSizeUnsigned64 = 70003,

    /**
     * @brief This constant is used with @ref nn::socket::ResolverGetOption()
     * and @ref nn::socket::ResolverSetOption(). The data value parameter is an
     * unsigned 64-bit value (uint64_t) and that that value is used to
     * determine the size in bytes of the service framework buffer used
     * for @ref GAIStrError
     */
    GaiErrorStringBufferSizeUnsigned64  = 70004,

    /**
     * @brief This constant is used with @ref nn::socket::ResolverGetOption()
     * and @ref nn::socket::ResolverSetOption().
     * The data value parameter is an unsigned 64-bit value (uint64_t)
     * and that that value is used to determine the size in bytes of the
     * service framework buffer used for @ref GetAddrInfo GetAddrInfo.
     */
    GetAddrInfoBufferSizeUnsigned64 = 70005,

    /**
     * @brief This constant is used with @ref nn::socket::ResolverGetOption(()
     * and @ref nn::socket::ResolverSetOption().
     * The data value parameter is an unsigned 64-bit value (uint64_t)
     * and that that value is used to determine the size in bytes of the
     * service framework buffer used for @ref GetAddrInfo GetAddrInfo
     * 'hints' parameter buffer size.
     */
    GetAddrInfoHintsBufferSizeUnsigned64 = 70006,

    /**
     * @brief
     * This constant defines the maximum boundary for Local key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    MaximumValue = 70007,
};

/**
 * @brief this global constant sets the maximum length of a domain name
 * string used with @ref PrivateSetTimeToLiveForDomainnameArgument.
 */
const size_t MaximumDomainnameStringBufferSize = 255;

/**
 * @brief This structure is the private parameter used with @ref
 * nn::socket::ResolverSetOption() to update the TTL for all
 * answer records matching the domain name provided.
 */
struct PrivateSetTimeToLiveForDomainnameArgument
{
    /**
     * @brief The time to live value.
     */
    uint32_t ttl;

    /**
     * @brief The null-terminated domain name string.
     */
    char domainname[MaximumDomainnameStringBufferSize];
};

/**
 * @brief This global constant sets the maximum number of addresses
 * that might be in the resolver DNS address array.
 */
const unsigned MaximumDnsServerAddresses = 2;

/**
 * @brief This structure is the private parameter used with @ref
 * nn::socket::ResolverGetOption() and @ref nn::socket::ResolverSetOption()
 * to set or get a copy of the resolver DNS address
 * array.
 */
struct PrivateDnsAddressArrayArgument
{
    /**
     * @brief The number of elements in the address array field.
     */
    uint32_t count;

    /**
     * @brief The address array field.
     */
    nn::socket::SockAddrIn addresses[MaximumDnsServerAddresses];
};

}}; // nn::socket
