﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/socket/netdb.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include <nn/socket/socket_TypesPrivate.h>

namespace nn { namespace socket { namespace resolver {

/**
 *@brief This function disables all calls to resolver API.
 */
void DisableResolverCalls() NN_NOEXCEPT;

/**
 *@brief This function enables all calls to resolver API.
 */
void EnableResolverCalls() NN_NOEXCEPT;

/**
 *@brief This function checks whether or not Resolver API calls are
 *currently allowed.
 *
 * return This function returns a boolean that indicates whether
 * resolver calls are enabled.
 */
bool AreResolverCallsEnabled() NN_NOEXCEPT;

/**
 *@brief This function waits for all resolver calls to complete.
 */
void WaitForAllResolverCallsToComplete() NN_NOEXCEPT;

/**
 *@brief This function initializes the client side of the resolver
 *library that is contained within the socket library code.
 *
 *@return This function returns a result indicating success or error.
 */
Result InitializeClient() NN_NOEXCEPT;

/**
 *@brief This function finalizes the client resolver library.
 *
 *@return This function returns a result indicating success or error.
 */
Result FinalizeClient() NN_NOEXCEPT;

nn::socket::HostEnt* GetHostEntByName(const char* pHostname) NN_NOEXCEPT;
struct hostent* GetHostByName(const char* pHostName) NN_NOEXCEPT;

nn::socket::HostEnt* GetHostEntByName(const char* pHostname,
                                      const ResolverOption* pOptions,
                                      size_t optionsCount) NN_NOEXCEPT;
struct hostent* GetHostByName(const char* pHostName,
                              const ResolverOption* pOptions,
                              size_t optionsCount) NN_NOEXCEPT;

nn::socket::HostEnt* GetHostEntByAddr(const void* pAddress,
                                      nn::socket::SockLenT length,
                                      nn::socket::Family addressFamily) NN_NOEXCEPT;
struct hostent* GetHostByAddr(const void* pInaddr,
                              uint32_t socketLengthIn,
                              uint32_t addressFamilyIn) NN_NOEXCEPT;

nn::socket::HostEnt* GetHostEntByAddr(const void* pAddress,
                                      nn::socket::SockLenT length,
                                      nn::socket::Family addressFamily,
                                      const ResolverOption* pOptions,
                                      size_t optionsCount) NN_NOEXCEPT;
struct hostent* GetHostByAddr(const void* pInaddr,
                              uint32_t socketLengthIn,
                              uint32_t addressFamilyIn,
                              const ResolverOption* pOptions,
                              size_t optionsCount) NN_NOEXCEPT;

nn::socket::AiErrno GetAddrInfo(const char* pNodeName,
                                const char* pServername,
                                const nn::socket::AddrInfo* pHints,
                                nn::socket::AddrInfo** pResult) NN_NOEXCEPT;
int GetAddrInfo(const char* pNodeName, const char* pServiceName,
                const struct addrinfo* pHints,
                struct addrinfo** ppAddrinfoOut) NN_NOEXCEPT;

nn::socket::AiErrno GetAddrInfo(const char* pNodeName,
                                const char* pServername,
                                const nn::socket::AddrInfo* pHints,
                                nn::socket::AddrInfo** pResult,
                                const ResolverOption* pOptions,
                                size_t optionsCount) NN_NOEXCEPT;
int GetAddrInfo(const char* pNodeName, const char* pServiceName,
                const struct addrinfo* pHints,
                struct addrinfo** ppAddrinfoOut,
                const ResolverOption* pOptions, size_t optionsCount) NN_NOEXCEPT;

nn::socket::AiErrno GetNameInfo(const nn::socket::SockAddr* pSockaddrIn,
                                nn::socket::SockLenT socketAddressLength,
                                char* pHostOut, size_t hostSizeIn,
                                char* pServiceOut, size_t serviceSizeIn,
                                nn::socket::NameInfoFlag flags) NN_NOEXCEPT;
int GetNameInfo(const struct sockaddr* sa, socklen_t salen,
                char* pHostName, size_t hostLength,
                char* pServiceName, size_t serviceLength,
                int flags) NN_NOEXCEPT;

nn::socket::AiErrno GetNameInfo(const nn::socket::SockAddr* pSockaddrIn,
                                nn::socket::SockLenT socketAddressLength,
                                char* pHostOut, size_t hostSizeIn,
                                char* pServiceOut, size_t serviceSizeIn,
                                nn::socket::NameInfoFlag flags,
                                const ResolverOption* pOptions,
                                size_t optionsCount) NN_NOEXCEPT;
int GetNameInfo(const struct sockaddr* sa, socklen_t salen,
                char* pHostName, size_t hostLength,
                char* pServiceName, size_t serviceLength,
                int flags,
                const ResolverOption* pOptions, size_t optionsCount) NN_NOEXCEPT;

int ResolverSetOption(const ResolverOption& option) NN_NOEXCEPT;

int ResolverGetOption(ResolverOption* pOption, ResolverOptionKey key) NN_NOEXCEPT;

const char* GetGaiStringError(nn::socket::AiErrno errorIn) NN_NOEXCEPT;
const char* GetGaiStringError(uint32_t errorIn) NN_NOEXCEPT;

const char* GetHostStringError(nn::socket::HErrno errorIn) NN_NOEXCEPT;
const char* GetHostStringError(uint32_t errorIn) NN_NOEXCEPT;

};};};  // nn::socket::resolver
