﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn {
namespace socket {

/*
 * Address Resolution Protocol.
 *
 * See RFC 826 for protocol description.  ARP packets are variable
 * in size; the arphdr structure defines the fixed-length portion.
 * Protocol type values are the same as those for 10 Mb/s Ethernet.
 * It is followed by the variable-sized fields ar_sha, arp_spa,
 * arp_tha and arp_tpa in that order, according to the lengths
 * specified.  Field names used correspond to RFC 826.
 */
enum class ArpHardware : uint8_t
{
    ArpHrd_Ether        = 1,        /* ethernet hardware format */
    ArpHrd_Ieee802      = 6,        /* token-ring hardware format */
    ArpHrd_Arcnet       = 7,        /* arcnet hardware format */
    ArpHrd_FRelay       = 15,       /* frame relay hardware format */
    ArpHrd_Ieee1394     = 24,       /* firewire hardware format */
    ArpHrd_Infiniband   = 32,       /* infiniband hardware format */
};

enum class ArpOp
{
    ArpOp_Request       = 1,        /* request to resolve address */
    ArpOp_Reply         = 2,        /* response to previous request */
    ArpOp_RevRequest    = 3,        /* request protocol address given hardware */
    ArpOp_RevReply      = 4,        /* response giving protocol address */
    ArpOp_InvRequest    = 8,        /* request to identify peer */
    ArpOp_InvReply      = 9,        /* response identifying peer */
};

struct ArpHdr
{
    uint16_t    ar_hrd;             /* format of hardware address */
    uint16_t    ar_pro;             /* format of protocol address */
    uint8_t     ar_hln;             /* length of hardware address */
    uint8_t     ar_pln;             /* length of protocol address */
    uint16_t    ar_op;              /* one of: */
};

inline uint32_t ArpHdr_Len2(uint8_t ar_hln, uint8_t ar_pln)
{
    return sizeof(ArpHdr) + 2 * (ar_hln) + 2 * (ar_pln);
}
inline uint32_t ArpHdr_Len(ArpHdr *ap)
{
    return ArpHdr_Len2(ap->ar_hln, ap->ar_pln);
}

/*
 * ARP ioctl request
 */
struct ArpReq
{
    nn::socket::SockAddr arp_pa;    /* protocol address */
    nn::socket::SockAddr arp_ha;    /* hardware address */
    int         arp_flags;          /* flags */
};

/*  arp_flags and at_flags field values */
enum ArpFlag
{
    Atf_InUse           = 0x01,     /* entry in use */
    Atf_Com             = 0x02,     /* completed entry (enaddr valid) */
    Atf_Perm            = 0x04,     /* permanent entry */
    Atf_Publ            = 0x08,     /* publish entry (respond for other host) */
    Atf_UseTrailers     = 0x10      /* has requested trailers */
};

struct ArpStat
{
    /* Normal things that happen: */
    uint64_t    txrequests;         /* # of ARP requests sent by this host. */
    uint64_t    txreplies;          /* # of ARP replies sent by this host. */
    uint64_t    rxrequests;         /* # of ARP requests received by this host. */
    uint64_t    rxreplies;          /* # of ARP replies received by this host. */
    uint64_t    received;           /* # of ARP packets received by this host. */

    uint64_t    arp_spares[4];      /* For either the upper or lower half. */
    /* Abnormal event and error  counting: */
    uint64_t    dropped;            /* # of packets dropped waiting for a reply. */
    uint64_t    timeouts;           /* # of times with entries removed */
                                    /* due to timeout. */
    uint64_t    dupips;             /* # of duplicate IPs detected. */
};

}} /* nn::socket */
